#!/usr/local/bin/perl
# 
# $Header: rdbms/admin/catcon.pl /main/10 2011/10/13 22:27:01 akociube Exp $
#
# catcon.pl
# 
# Copyright (c) 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      catcon.pl - CONtainer-aware Perl script for creating/upgrading CATalogs
#
#    DESCRIPTION
#      This script defines subroutines which can be used to execute one or 
#      more SQL statements or a SQL*Plus script in 
#      - a non-Consolidated Database, 
#      - all Containers of a Consolidated Database, or 
#      - a specified Container of a Consolidated Database
#
#    NOTES
#      Subroutines which handle invocation of SQL*Plus scripts may be invoked 
#      from other scripts (e.g. catctl.pl) or by invoking catcon.pl directly.  
#      Subroutines that execute one or more SQL statements may only be 
#      invoked from other Perl scripts.
#
#    MODIFIED   (MM/DD/YY)
#    akruglik    10/11/11 - Allow user to optionally specify internal connect
#                           string
#    akruglik    10/04/11 - add suport for running SQL statements
#    akruglik    10/03/11 - clean up Usage comments
#    akruglik    09/20/11 - Add support for specifying multiple containers in
#                           which to run - or not run - scripts
#    akruglik    09/20/11 - make --p and --P the default tags for regular and
#                           secret arguments
#    akruglik    06/30/11 - undo changes from the previous version
#    akruglik    06/10/11 - rename CatCon.pm to catcon.pm because oratst get
#                           macro is case-insensitive
#    akruglik    06/10/11 - set to 0 all vars populated by getopts before
#                           calling getopts so we don't get uninitialized value
#                           errors
#    akruglik    05/26/11 - Make use of subroutines in catcon.pm
#    akruglik    05/11/11 - change container query to fetch from container$;
#                           open pdb$seed read/write; wait for auxiliary 
#                           process to terminate before resetting CHLD signal 
#                           handler
#    akruglik    04/19/11 - add support for specifying ERRORLOGGING
#    akruglik    04/08/11 - add support for SQL script parameters
#    akruglik    02/18/11 - a script to run a SQL script or one or more
#                           statements in 1 or all Containers of a CDB or in a
#                           non-CDB
#    akruglik    02/18/11 - Creation
# 

#use strict "vars";
use Getopt::Std;      # to parse command line options

use catcon qw( catconInit catconExec catconWrapUp );

# value returned by various subroutines
my $RetCode = 0;

if (@ARGV < 1) {
    print STDERR <<USAGE;

  Usage: catcon  [-u username[/password]] [-U username[/password]] 
                 [-d directory] [-l directory] 
                 [{-c|-C} container] [-p degree-of-parallelism]
                 [-e] [-s]
                 [-E { ON | errorlogging-table-other-than-SPERRORLOG } ]
                 [-g] 
                 -b log-file-name-base 
                 --
                 { sqlplus-script [arguments] | --x<SQL-statement> } ...

   Optional:
     -u username (optional /password; otherwise prompts for password)
        used to connect to the database to run user-supplied scripts or 
        SQL statements
        defaults to "/ as sysdba"
     -U username (optional /password; otherwise prompts for password)
        used to connect to the database to perform internal tasks
        defaults to "/ as sysdba"
     -d directory containing the file to be run 
     -l directory to use for spool log files
     -c container(s) in which to run sqlplus scripts, i.e. skip all 
        Containers not named here; for example, 
          -c 'PDB1 PDB2', 
     -C container(s) in which NOT to run sqlplus scripts, i.e. skip all 
        Containers named here; for example,
          -C 'CDB$ROOT PDB3'

       NOTE: -c and -C are mutually exclusive

     -p expected number of concurrent invocations of this script on a given 
        host

       NOTE: this parameter rarely needs to be specified

     -e sets echo on while running sqlplus scripts
     -s output of running every script will be spooled into a file whose name 
        will be 
          <log-file-name-base>_<script_name_without_extension>_[<container_name_if_any>].<default_extension>
     -E sets errorlogging on; if ON is specified, default error logging table
        will be used, otherwise, specified error logging table (which must 
        have been created in every Container) will be used
     -g turns on production of debugging info while running this script

   Mandatory:
     -b base name (e.g. catcon_test) for log and spool file names
        
     sqlplus-script - sqlplus script to run OR
     SQL-statement  - a statement to execute

   NOTES:
     - if --x<SQL-statement> is the first non-option string, it needs to be 
       preceeded with -- to avoid confusing module parsing options into 
       assuming that '-' is an option which that module is not expecting and 
       about which it will complain
     - command line parameters to SQL scripts can be introduced using --p
       interactive (or secret) parameters to SQL scripts can be introduced 
       using --P

     For example,
       perl catcon.pl ... x.sql '--pJohn' '--PEnter Password for John:' ...

USAGE
    exit;
}

# set all option vars to 0 to avoid "Use of uninitialized value" errors
our ($opt_u, $opt_U, $opt_d, $opt_l, $opt_b, $opt_c, $opt_C, $opt_n, $opt_p, 
     $opt_e, $opt_s, $opt_a, $opt_A, $opt_E, $opt_g) = (0) x 15;

# Parse command line arguments
#   NOTE: we are not advertising that the caller may specify number of 
#   processes, but if the caller specifies it anyway, we will use it
# NOTE: if adding more options, add $opt_* vars to the 
#         our () = (0) x <number>;
#       lin above and remember to increment the <number>
getopts("c:C:u:U:n:d:l:p:b:a:A:E:esg");

# some things must have been specified:
# - base for log file names
# - at least one sqlplus script
die "Base for log file names must be supplied" if !$opt_b;
die "At least one file name must be supplied" if !@ARGV;

my @PerProcInitStmts = ();
my @PerProcEndStmts = ();

$RetCode =
  catconInit($opt_u, $opt_U, $opt_d, $opt_l, $opt_b, $opt_c, $opt_C, $opt_n, 
             $opt_p, $opt_e, $opt_s, $opt_a, $opt_A, $opt_E,
             @PerProcInitStmts,
             @PerProcEndStmts,
             $opt_g);

if ($RetCode) {
  die "Unexpected error encountered in catconInit; exiting\n";
}

if (@ARGV < 1) {
  die "At least one sqlplus script name or SQL statement must be supplied\n";
}

# verify that neither regular nor secret argument delimiter starts with --x 
# and so can cause confusion when we try to determine whether a string 
# starting with --x represents a statement to be run or an argument to a 
# preceding script
if ($opt_a && $opt_a =~ /^--x/) {
  die "regular argument delimiter may not start with --x\n";
}

if ($opt_A && $opt_A =~ /^--x/) {
  die "secret argument delimiter may not start with --x\n";
}

# prepend script names with @; it is important that they have a leading @ 
# since catconExec uses @ to distinguish script names from SQL statements
# It is also important that we do not prepend @ to the script arguments, if any
my @Scripts;
my $NextItem; # element of @ARGV being processes

# if the user supplied regular and/or secret argument delimiters, this 
# variable will be set to true after we encounter a script name to remind 
# us to check for possible arguments
my $LookForArgs = 0;
    
foreach $NextItem (@ARGV) {

  if ($opt_g) {
    print STDERR "going over ARGV: NextItem = $NextItem\n";
  }

  if (   ($opt_a && $NextItem =~ /^$opt_a/)
      || ($opt_A && $NextItem =~ /^$opt_A/)) {
    # looks like an argument to a script; make sure we are actually 
    # looking for script arguments
    if (!$LookForArgs) {
      die "unexpected script argument ($NextItem) encountered\n";
    }

    if ($opt_g) {
      print STDERR "add script argument string ($NextItem) to Scripts\n";
    }

    push @Scripts, $NextItem;
  } elsif ($NextItem =~ /^--x/) {
    if ($opt_g) {
      print STDERR "add statement (".substr($NextItem, 3).") to Scripts\n";
    }

    push @Scripts, substr($NextItem, 3);

    # no arguments can be passed to statements
    $LookForArgs = 0;
  } else {
    
    if ($opt_g) {
      print STDERR "add script name ($NextItem) to Scripts\n";
    }

    push @Scripts, "@".$NextItem;

    # having seen what looks like a script name, allow for arguments, as long 
    # as the user supplied argument delimiter(s)
    $LookForArgs = 1 if ($opt_a || $opt_A);
  }
}

$RetCode = catconExec(@Scripts, 0, 0, 0);

if ($RetCode) {
  die "catcon.pl: Unexpected error encountered in catconExec; exiting\n";
}

catconWrapUp();

exit 0;
