# 
# $Header: rdbms/admin/catcon.pm /st_rdbms_12.1.0.1/4 2013/04/09 10:58:29 cxie Exp $
#
# catcon.pm
# 
# Copyright (c) 2011, 2013, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      catcon.pm - CONtainer-aware Perl Module for creating/upgrading CATalogs
#
#    DESCRIPTION
#      This module defines subroutines which can be used to execute one or 
#      more SQL statements or a SQL*Plus script in 
#      - a non-Consolidated Database, 
#      - all Containers of a Consolidated Database, or 
#      - a specified Container of a Consolidated Database
#
#    NOTES
#      Subroutines which handle invocation of SQL*Plus scripts may be invoked 
#      from other scripts (e.g. catctl.pl) or by invoking catcon.pl directly.  
#      Subroutines that execute one or more SQL statements may only be 
#      invoked from other Perl scripts.
#
#    MODIFIED   (MM/DD/YY)
#    akruglik    03/19/13 - do not quote the password if the user has already
#                           quoted it
#    akruglik    03/15/13 - XbranchMerge akruglik_bug-16371697 from main
#    akruglik    03/07/13 - use v$database.cdb to determine whether a DB is a
#                           CDB
#    akruglik    02/08/13 - Bug 16177906: quote user-supplied password in case
#                           it contains any special characters
#    akruglik    12/03/12 - (LRG 8526376): replace calls to
#                           DBMS_APPLICATION_INFO.SET_MODULE/ACTION with
#                           ALTER SESSION SET APPLICATION MODULE/ACTION
#    akruglik    11/15/12 - (LRG 8522365) temporarily comment out calls to
#                           DBMS_APPLICATION_INFO
#    surman      11/09/12 - 15857388: Resolve Perl typo warning
#    akruglik    11/09/12 - (LRG 7357087): PDB$SEED needs to be reopened READ
#                           WRITE unless it is already open READ WRITE or
#                           READ WRITE MIGRATE
#    akruglik    11/08/12 - use DBMS_APPLICATION_INFO to store info about
#                           processes used to run SQL scripts
#    akruglik    11/08/12 - if debugging is turned on, make STDERR hot
#    akruglik    11/08/12 - (15830396): read and ignore output in
#                           exec_DB_script if no marker was passed, to avoid
#                           hangs on Windows
#    surman      10/29/12 - 14787047: Save and restore stdout
#    mjungerm    09/10/12 - don't assume unlink will succeed - lrg 7184718
#    dkeswani    08/13/12 - Bug 14380261 : delete LOCAL variable on WINDOWS
#    akruglik    08/02/12 - (13704981): report an error if database is not open
#    sankejai    07/23/12 - 14248297: close/open PDB$SEED on all RAC instances
#    akruglik    06/26/12 - modify exec_DB_script to check whether @Output
#                           contains at least 1 element before attempting to
#                           test last character of its first element
#    akruglik    05/23/12 - rather than setting SQLTERMINATOR ON, use /
#                           instead of ; to terminate SQL statements
#    akruglik    05/18/12 - add SET SQLTERMINATOR ON after every CONNECT to
#                           ensure that SQLPLus processes do not hang if the
#                           caller sets SQLTERMINATOR to something other than
#                           ON in glogin.sql or login.sql
#    akruglik    04/18/12 - (LRG 6933132) ignore row representing a non-CDB
#                           when fetching rows from CONTAINER$
#    gravipat    03/20/12 - Rename x$pdb to x$con
#    akruglik    02/22/12 - (13745315): chop @Output in exec_DB_script to get 
#                           rid of trailing \r which get added on Windows
#    akruglik    12/09/11 - (13404337): in additionalInitStmts, set SQLPLus
#                           vars to their default values
#    akruglik    10/31/11 - modify additionalInitStmts to set more of SQLPlus
#                           system vars to prevent values set in a script run
#                           against one Container from affecting output of
#                           another script or a script run against another
#                           Container
#    prateeks    10/21/11 - MPMT changes : connect string
#    akruglik    10/14/11 - Bug 13072385: if PDB$SEED was reopened READ WRITE
#                           in concatExec, it will stay that way until
#                           catconWrapUp
#    akruglik    10/11/11 - Allow user to optionally specify internal connect
#                           string
#    akruglik    10/03/11 - address 'Use of uninitialized value in
#                           concatenation' error
#    akruglik    09/20/11 - Add support for specifying multiple containers in
#                           which to run - or not run - scripts
#    akruglik    09/20/11 - make --p and --P the default tags for regular and
#                           secret arguments
#    akruglik    09/20/11 - make default number of processes a function of
#                           cpu_count
#    akruglik    08/24/11 - exec_DB_script was hanging on Windows because Perl
#                           does not handle CHLD signal on Windows
#    pyam        08/08/11 - don't run scripts AS SYSDBA unless running as sys
#    akruglik    08/03/11 - unset TWO_TASK to ensure that connect without
#                           specifyin a service results in a connection to the
#                           root
#    akruglik    07/01/11 - encapsulate common code into subroutines
#    akruglik    06/30/11 - temporarily suspend use of Term::ReadKey
#    akruglik    06/10/11 - rename CatCon.pm to catcon.pm because oratst get
#                           macro is case-insensitive
#    akruglik    06/10/11 - Add support for spooling output of individual
#                           scripts into separate files
#    akruglik    05/26/11 - Creation
#
#!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
#
# FIXME: Term::ReadKey is not currently a part of Perl component that gets 
#        shipped with Oracle, so I commented out "user Term::ReadKey"
#        and defined local versions of its two subroutines which this script 
#        uses (ReadMode and ReadLine.)  
#
#         Once Term::ReadKey becomes available, 
#         - remove lines ending with comment # until Term::ReadKey is available
#         - remove string "# Term::ReadKey workaround #" from lines which had 
#           to be temporarily commented out
#         - and finally, remove this FIXME
# END OF FIXME
#!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
# 

package catcon;

use 5.006;
use strict;
use warnings;

use English;
use IO::Handle;       # to flush buffers
# Term::ReadKey workaround #use Term::ReadKey;    # to not echo password
use IPC::Open2;       # to perform 2-way communication with SQL*Plus

require Exporter;

our @ISA = qw(Exporter);

# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.

# This allows declaration       use catcon ':all';
# If you do not need this, moving things directly into @EXPORT or @EXPORT_OK
# will save memory.
our %EXPORT_TAGS = ( 'all' => [ qw(
        
) ] );

our @EXPORT_OK = qw ( catconInit catconExec catconRunSqlInEveryProcess catconShutdown catconBounceProcesses catconWrapUp );

our @EXPORT = qw(
        
);
our $VERSION = '0.01';


# Preloaded methods go here.

sub ReadMode { # until Term::ReadKey is available
} # until Term::ReadKey is available

sub ReadLine { # until Term::ReadKey is available
  return <STDIN>; # until Term::ReadKey is available
} # until Term::ReadKey is available

#
# get_connect_string
#
# Description:
#   If caller-supplied string contained a password, 
#     connect string will consist of caller-supplied string with 
#     AS SYSDBA appended to it.
#   Otherwise, if caller-supplied string did NOT contain a password, 
#     prompt user for a password and 
#     construct the connect string as a concatenation of caller-supplied 
#     string, '/', password, and AS SYSDBA
#   Otherwise,
#     set connect string to 'sys/knl_test7 AS SYSDBA'
#
# Parameters:
#   - user name, optionally with password, supplied by the caller; may be 
#     undefined
#
# Returns:
#   connect string
#
sub get_connect_string ($) {
  my $user = $_[0];
  my $password;
  my $connect;
  my $sysdba = 0;

  if ($user) {
    # user name specified

    if ($user =~ /(.*)\/(.*)/) {
      # password supplied with username
      $user = $1;
      $password = $2;
    } else {
      # prompt for password
      print "Enter Password: ";
      ReadMode 'noecho';
      $password = ReadLine 0;
      chomp $password;
      ReadMode 'normal';
      print "\n";
    }

    if (uc($user) eq "SYS") {
      $sysdba = 1;
    }
   
    # quote the password unless the user has done it for us.  We assume that 
    # passwords cannot contains double quotes and trust the user to not play 
    # games with us and only half-quote the password
    if (substr($password,1,1) ne '"') {
      $password = '"'.$password.'"'
    }

    $connect = $user."/"."$password";
  } else {
    # default to OS authentication 
    $connect = "sys/knl_test7"; 
    $sysdba = 1;
  } 

  if ($sysdba) {
    $connect = $connect . " AS SYSDBA";
  }

  return $connect;
}

# A wrapper over unlink to retry if it fails, as can happen on Windows
# apparently if unlink is attempted on a .done file while the script is
# is still writing to it.  Silent failure to unlink a .done file can
# cause incorrect sequencing leading to problems like lrg 7184718
sub sureunlink ($$) {
  my ($DoneFile, $debugOn) = @_;
  my $iters = 0;
  while (!unlink($DoneFile) && $iters < 100) { 
   $iters++; 
   if ($debugOn) {
     print STDERR "unlink($DoneFile) failed $iters time(s) due to $!\n";
   }
   sleep(1);
  }
  die "could not unlink $DoneFile - $!" if (-e $DoneFile);
}

{
  # This block consists of 2 subroutines:
  # - handle_aux_sigchld() - handles CHLD signal by simply returning
  # - exec_DB_script() - execute an array of statements, possibly returning 
  #     filtered output produced by executing them; sets $SIG{CHLD} to
  #     handle_aux_sigchld() to ensure that CHLD handler does not get reset 
  #     to a proper signal handler (which kills the session) until the process 
  #     executing statements has terminated

  # this handler is used when we are waiting for an auxiliary process to 
  # finish; instead of killing the session running the script, we just return
  sub handle_aux_sigchld () {
    return;
  }

  #
  # exec_DB_script
  #
  # Description:
  #   Connect to a database using connect string supplied by the caller, run 
  #   statement(s) supplied by the caller, and if caller indicated interest in 
  #   some of the output produced by the statements, return a list, possibly 
  #   empty, consisting of results returned by that query which contain 
  #   specified marker
  #
  # Parameters:
  #   - a reference to a list of statements to execute
  #   - a string (marker) which will mark values of interest to the caller; 
  #     if no value is supplied, an uninitialized list will be returned
  #   - a command to create a file whose existence will indicate that the 
  #     last statement of the script has executed
  #   - base for a name of a "done" file (see above)
  #   - an indicator of whether to produce debugging info
  #
  # Returns
  #   A list consisting of values returned by the query, stripped off the 
  #   marker
  #
  sub exec_DB_script (\@$$$$) {
    my ($statements, $marker, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

    # file whose existence will indicate that all statements in caller's 
    # script have executed
    my $DoneFile = $DoneFilePathBase."_exec_DB_script.done";

    my @Output;

    local (*Reader, *Writer);

    # temporarily reset $SIG{CHLD} to avoid killing the session when exit 
    # statement contained in the script shuts down SQL*Plus process
    my $saveHandlerRef = $SIG{CHLD};

    $SIG{CHLD} = \&handle_aux_sigchld;

    if ($debugOn) {
      print STDERR "exec_DB_script: temporarily reset SIGCHLD handler\n";
    }

    # if the "done" file exists, delete it
    if (-e $DoneFile) {
      sureunlink($DoneFile, $debugOn);

      if ($debugOn) {
        print STDERR "exec_DB_script: deleted $DoneFile before running a script\n";
      }
    } elsif ($debugOn) {
      print STDERR "exec_DB_script: $DoneFile did not need to be deleted before running a script\n";
    }

    my $pid = open2(\*Reader, \*Writer, "sqlplus /nolog");

    if ($debugOn) {
      print STDERR "exec_DB_script: opened Reader and Writer\n";
    }

    # execute sqlplus statements supplied by the caller
    foreach (@$statements) {
      print Writer $_;
      if ($debugOn) {
        print STDERR "exec_DB_script: executed $_\n";
      }
    }

    # send a statement to generate a "done" file
    print Writer qq/$DoneCmd $DoneFile\n/;

    if ($debugOn) {
      print STDERR "exec_DB_script: sent $DoneCmd $DoneFile to Writer\n";
    }

    # send a statement to generate a "done" file
    print Writer qq/exit\n/;

    if ($debugOn) {
      print STDERR "exec_DB_script: sent -exit- to Writer\n";
    }

    close Writer;       #have to close Writer before read

    if ($debugOn) {
      print STDERR "exec_DB_script: closed Writer\n";
    }

    if ($marker) {
      if ($debugOn) {
        print STDERR "exec_DB_script: marker = $marker - examine output\n";
      }

      # have to read one line at a time
      while (<Reader>) { 
        if ($_ =~ /^$marker/) {
          # strip off the marker that was added to identify values of interest
          s/$marker(.*)//;
          # and add it to the list
          push @Output, $1;
        }
      }

      # (13745315) on Windows, values fetched from SQL*Plus contain trailing 
      # \r, but the rest of the code does not expect these \r's, so we will 
      # chop them here
      if (@Output && substr($Output[0], -1) eq "\r") {
        chop @Output;
      }
    } else {
      if ($debugOn) {
        print STDERR "exec_DB_script: marker was undefined; read and ignore output, if any\n";
      }

      # (15830396) read anyway
      while (<Reader>) {
        ;
      }

      if ($debugOn) {
        print STDERR "exec_DB_script: finished reading and ignoring output\n";
      }
    }

    if ($debugOn) {
      print STDERR "exec_DB_script: waiting for child process to exit\n";
    }

    # wait until the process running SQL statements terminates
    # 
    # NOTE: Instead of waiting for CHLD signal which gets issued on Linux, 
    #       we wait for a "done" file to be generated because this should 
    #       work on Windows as well as Linux (and other Operating Systems, 
    #       one hopes)
    select (undef, undef, undef, 0.01)    until (-e $DoneFile);

    if ($debugOn) {
      print STDERR "exec_DB_script: child process exited\n";
    }

    sureunlink($DoneFile, $debugOn);

    if ($debugOn) {
      print STDERR "exec_DB_script: deleted $DoneFile after running a script\n";
    }

    close Reader;

    if ($debugOn) {
      print STDERR "exec_DB_script: closed Reader\n";
    }

    waitpid($pid, 0);   #makes your program cleaner
  
    if ($debugOn) {
      print STDERR "exec_DB_script: waitpid returned\n";
    }

    # 
    # restore CHLD signal handler
    # before it completes its work
    #
    $SIG{CHLD} = $saveHandlerRef;

    if ($debugOn) {
      print STDERR "exec_DB_script: restored SIGCHLD handler\n";
    }

    return @Output;
  }
}

#
# get_instance_status
#   
# Description
#   Obtain instance status.
#
# Parameters:
#   - connect string - used to connect to a DB 
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - indicator of whether to produce debugging info
#
# Returns
#   String found in V$INSTANCE.STATUS
#
sub get_instance_status ($$$$) {

  my ($connectString, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

  my @GetInstStatusStatements = (
    "connect $connectString\n",
    "set echo off\n",
    "set heading off\n",
    "select \'XXXXXX\' || status from v\$instance\n/\n",
  );

  my @InstStatus = exec_DB_script(@GetInstStatusStatements, "XXXXXX", 
                                  $DoneCmd, $DoneFilePathBase, $debugOn);
  return $InstStatus[0];
}

#
# get_CDB_indicator
#   
# Description
#   Obtain an indicator of whether a DB is a CDB
#
# Parameters:
#   - connect string - used to connect to a DB 
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - indicator of whether to produce debugging info
#
# Returns
#   String found in V$DATABASE.CDB
#
sub get_CDB_indicator ($$$$) {

  my ($connectString, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

  my @GetIsCdbStatements = (
    "connect $connectString\n",
    "set echo off\n",
    "set heading off\n",
    "select \'XXXXXX\' || cdb from v\$database\n/\n",
  );

  my @IsCDB = exec_DB_script(@GetIsCdbStatements, "XXXXXX", 
                             $DoneCmd, $DoneFilePathBase, $debugOn);
  return $IsCDB[0];
}

#
# get_num_procs_int
#   
# Description
#   Connect to the database to determine whether it is Consolidated.  If it 
#   is not, a single process will do.  If it is, set number of processes to 
#   twice the number of cores (similar to how PDML determines how many slaves 
#   it can create.)
#
# Parameters:
#   - connect string - used to connect to a DB and determine whether it is 
#     consolidated (and eventually to determine how many processes should be 
#     created to run script(s)
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - indicator of whether to produce debugging info
#
# Returns
#   number of processes which will be used to run script(s)
#
sub get_num_procs_int ($$$$) {

  my ($connectString, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

  my @GetNumCoresStatements = (
    "connect $connectString\n",
    "set echo off\n",
    "set heading off\n",
    "select \'XXXXXX\' || p.value * 2 from v\$parameter p where p.name=\'cpu_count\'\n/\n",
  );

  my @NumCores = exec_DB_script(@GetNumCoresStatements, "XXXXXX", 
                                $DoneCmd, $DoneFilePathBase, $debugOn);
  return $NumCores[0];
}

#
# get_num_procs - determine number of processes which should be created
#
# parameters:
#   - number of processes, if any, supplied by the user
#   - number of concurrent script invocations, as supplied by the user 
#     (external degree of parallelism)
#   - connect string
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - an indicator of whether to produce debugging info
#
sub get_num_procs ($$$$$$) {
  my ($NumProcs, $ExtParaDegree, $ConnectString, $DoneCmd, $DoneFilePathBase, 
      $DebugOn) = @_;

  my $MaxNumProcs  = 8;   # maximum number of processes
  my $MinNumProcs  = 1;   # minimum number of processes
  my $ProcsToStart;       # will be returned to the caller

  # compute the number of processes to be started

  if ($NumProcs > 0) {

    # caller supplied a number of processes; 
    $ProcsToStart = $NumProcs;

    if ($DebugOn) {
      print STDERR "get_num_procs: caller-supplied number of processes (not final) = $NumProcs\n";
    }
  } else {    
    # first obtain a default value for number of processes based on 
    # hardware characteristics
    $ProcsToStart = get_num_procs_int($ConnectString, $DoneCmd, 
                                      $DoneFilePathBase, $DebugOn);
      
    if ($DebugOn) {
      print STDERR "get_num_procs: computed number of processes (not final) = $ProcsToStart\n";
    }

    # if called interactively and the caller has provided the number of 
    # concurrent script invocations on this host, compute number of 
    # processes which will be started in this invocation
    if ($ExtParaDegree) {
      if ($DebugOn) {
        print STDERR "get_num_procs: number of concurent script invocations = $ExtParaDegree\n";
      }

      $ProcsToStart = int ($ProcsToStart / $ExtParaDegree);

      if ($DebugOn) {
        print STDERR "get_num_procs: adjusted for external parallelism, number of processes (still not final) for this invocation of the script = $ProcsToStart\n";
      }
    }
  }

  # use number of processes which was either supplied by the user or 
  # computed by get_num_procs_int() it unless it is too large or too small
    
  if ($ProcsToStart > $MaxNumProcs) {
    $ProcsToStart = $MaxNumProcs;
  } elsif ($ProcsToStart < $MinNumProcs) {
    $ProcsToStart = $MinNumProcs;
  }

  if ($DebugOn) {
    print STDERR "get_num_procs: will start $ProcsToStart processes (final)\n";
  }

  return $ProcsToStart;
}

#
# valid_src_dir make sure source directory exists and is readable
#
# Parameters:
#   - source directory name; may be undefined
#
# Returns
#   1 if valid; 0 otherwise
#
sub valid_src_dir ($) {
  my ($SrcDir) = @_;

  if (!$SrcDir) {
    # no source directory specified - can't complain of it being invalid
    return 1;
  }

  # if a directory for sqlplus script(s) has been specified, verify that it 
  # exists and is readable
  stat($SrcDir);

  if (! -e _ || ! -d _) {
    print STDERR "valid_src_dir: Specified source file directory ($SrcDir) does not exist or is not a directory\n";
    return 0;
  }

  if (! -r _) {
    print STDERR "valid_src_dir: Specified source file directory ($SrcDir) is unreadable\n";
    return 0;
  }

  return 1;
}

#
# valid_log_dir make sure log directory exists and is writable
#
# Parameters:
#   - log directory name; may be undefined
#
# Returns
#   1 if valid; 0 otherwise
#
sub valid_log_dir ($) {
  my ($LogDir) = @_;

  if (!$LogDir) {
    # no log directory specified - can't complain of it being invalid
    return 1;
  }

  stat($LogDir);

  if (! -e _ || ! -d _) {
    print STDERR "valid_log_dir: Specified log file directory ($LogDir) does not exist or is not a directory\n";
    return 0;
  }

  if (! -w _) {
    print STDERR "valid_log_dir: Specified log file directory ($LogDir) is unwritable\n";
    return 0;
  }

  return 1;
}

#
# validate_con_names - determine whether Container name(s) supplied
#     by the caller are valid (i.e. that the DB is Consolidated and contains a 
#     Container with specified names) and modify a list of Containers against 
#     which scripts/statements will be run as directed by the caller:  
#     - if the caller indicated that a list of Container names to be validated
#       refers to Containers against which scripts/statements should not be 
#       run, remove them from $Containers
#     - otherwise, remove names of Containers which did not appear on the 
#       list of Container names to be validated from $Containers
#
# Parameters:
#   - reference to a string containing space-delimited name(s) of Container
#   - an indicator of whether the above list refers to Containers against 
#     which scripts/statements should not be run
#   - reference to an array of Container names, if any
#   - an indicator of whether debugging information should be produced
#
# Returns:
#   Array of names of Containers against which 1 if Container name refers to a valid Container; 
#   0 otherwise
#
sub validate_con_names (\$$\@$) {
  my ($ConNameStr, $Exclude, $Containers, $DebugOn) = @_;

  if ($DebugOn) {

    print STDERR <<validate_con_names_DEBUG;
running validate_con_names(
  ConNameStr   = $$ConNameStr, 
  Exclude      = $Exclude, 
  Containers   = @$Containers)

validate_con_names_DEBUG
  }

  if (!${$ConNameStr}) {
    # this subroutine should not be called unless there are Container name to 
    # validate
    print STDERR "validate_con_names: missing Container name string\n";

    return ();
  }

  # extract Container names into an array
  my @ConNameArr = split(/  */, $$ConNameStr);

  if (!@ConNameArr) {
    # string supplied by the caller contained no Container names
    print STDERR "validate_con_names: no Container names to validate\n";

    return ();
  }

  # string supplied by the caller contained 1 or more Container names

  if (!@$Containers) {
    # report error and quit since the database is not Consolidated
    print STDERR "validate_con_names: Container name(s) supplied but the DB is not Consolidated\n";

    return ();
  } 
    
  if ($DebugOn) {
    print STDERR "validate_con_names: Container name string consisted of the following names {\n";
    foreach (@ConNameArr) {
      print STDERR "validate_con_names: \t$_\n";
    }
    print STDERR "validate_con_names: }\n";
  }

  # so here's a dilemma: 
  #   we need to 
  #   (1) verify that all names in @ConNameArr are also in @$Containers AND
  #   (2) compute either 
  #       @$Containers - @ConNameArr (if $Exclude != 0) or 
  #       @$Containers <intersect> @ConNameArr (if $Exclude == 0)
  #
  #       In either case, I would like to ensure that CDB$ROOT, if it ends up 
  #       in the resulting set, remains the first element of the array (makes 
  #       it easier when we nnee to decide at what point we can parallelize 
  #       execution across all remaining PDBs) and that PDBs are ordered by 
  #       their Container ID (I am just used to processing them in that order)
  #
  #   (1) would be easiest to accomplish if I were to store contents of 
  #   @$Containers in a hash and iterate over @ConNameArr, checking against 
  #`  that hash.  However, (2) requires that I traverse @$Containers and 
  #   decide whether to keep or remove an element based on whether it occurs 
  #   in @ConNameArr (which would require that I construct a hash using 
  #   contents of @ConNameArr) and the value of $Exclude
  #
  # I intend to implement the following algorithm:
  # - store contents of @ConNameArr in a hash (%ConNameHash)
  # - create a new array which will store result of (2) (@ConOut)
  # - for every element C in @$Containers
  #   - if %ConNameHash contains an entry for C
  #     - if !$Exclude
  #       - append C to @ConOut
  #   - else
  #     - if $Exclude
  #       - append C to @ConOut
  # - if, after we traverse @$Containers, %ConNameHash still contains some 
  #   entries which were not matched, report an error since it indicates that 
  #   $$ConNameStr contained some names which do not refer to existing 
  #   Container names
  # - if @ConOut is empty, report an error since there will be no Containers 
  #   in which to run scripts/statements
  # - return @ConOut which will by now contain names of all Containers in which
  #   scripts/statements will need to be run
  #

  my %ConNameHash;   # hash of elements of @ConNameArr

  foreach (@ConNameArr) {
    undef $ConNameHash{$_};
  }

  # array of Container names against which scripts/statements should be run
  my @ConOut = ();
  my $matched = 0;   # number of elements of @$Containers found in %ConNameHash

  foreach my $C (@$Containers) {
    # if we have matched every element of %ConNameHash, there is no reason to 
    # check whether it contains $C
    if (($matched < @ConNameArr) && exists($ConNameHash{$C})) {
      $matched++; # remember that one more element of @$Containers was found

      # remove $C from %ConNameHash so that if we end up not matching some 
      # specified Container names, we can list them as a part of error message
      delete $ConNameHash{$C};

      if ($DebugOn) {
        print STDERR "validate_con_names: $C was matched\n";
      }

      # add matched Container name to @ConOut if caller indicated that 
      # Containers whose names were specified are to be included in the set 
      # of Containers against which scripts/statements will be run
      if (!$Exclude) {
        push(@ConOut, $C); 

        if ($DebugOn) {
          print STDERR "validate_con_names: Added $C to ConOut\n";
        }
      }
    } else {
      if ($DebugOn) {
        print STDERR "validate_con_names: $C was not matched\n";
      }

      # add unmatched Container name to @ConOut if caller indicated that 
      # Containers whose names were specified are to be excluded from the set 
      # of Containers against which scripts/statements will be run
      if ($Exclude) {
        push(@ConOut, $C); 

        if ($DebugOn) {
          print STDERR "validate_con_names: Added $C to ConOut\n";
        }
      }
    }
  }

  # if any of specified Container names did not get matched, report an error
  if ($matched != @ConNameArr) {
    print STDERR "validate_con_names: some specified Container names do not refer to existing Containers:\n";
    for (keys %ConNameHash) {
      print STDERR "\t$_\n";
    }

    return ();
  }

  # if @ConOut is empty (which could happen if we were asked to exclude every 
  # Container)
  if (!@ConOut) {
    print STDERR "validate_con_names: all Containers have been excluded from execution\n";

    return ();
  }

  if ($DebugOn) {
    print STDERR "validate_con_names: resulting Container set consists of the following Containers {\n";
    foreach (@ConOut) {
      print STDERR "validate_con_names:\t$_\n";
    }
    print STDERR "validate_con_names: }\n";
  }

  return @ConOut;
}

#
# get_pdb_names - query DBA_PLUGGABLE_DATABASES to get names of 
#                 Pluggable Databases, if any
#
# parameters:
#   - connect string
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - indicator of whether to produce debugging info
#
sub get_pdb_names ($$$$) {
  my ($myConnect, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

  # NOTE: it is important that we fetch data from tables (CONTAINER$ and OBJ$ 
  #       rather than dictionary views (e.g. DBA_PLUGGABLE_DATABASES) because 
  #       views may yet to be created if this procedure is used when running 
  #       catalog.sql
  # NOTE: since a non-CDB may also have a row in CONTAINER$ with con_id#==0,
  #       we must avoid fetching a CONTAINER$ row with con_id==0 when looking 
  #       for Container names 
  my @GetContainerNamesStatements = (
    "connect $myConnect\n",
    "set echo off\n",
    "set heading off\n",
    "select \'XXXXXX\' || o.name from container\$ c, obj\$ o where o.obj#=c.obj# and c.con_id# > 0 order by c.con_id#\n/\n",
  );

  return exec_DB_script(@GetContainerNamesStatements, "XXXXXX", 
                        $DoneCmd, $DoneFilePathBase, $debugOn);
}

#
# seed_pdb_mode_state - return PDB$SEED's state, 0 if not there
#
# parameters:
#   - connect string
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - indicator of whether to produce debugging info
#
sub seed_pdb_mode_state ($$$$) {
  my ($myConnect, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

  my @SeedModeStateStatements = (
    "connect $myConnect\n",
    "set echo off\n",
    "set heading off\n",
    "select \'XXXXXX\' || state from x\$con where name='PDB\$SEED'\n/\n",
  );


  my @retValues = exec_DB_script(@SeedModeStateStatements, "XXXXXX", 
                                 $DoneCmd, $DoneFilePathBase, $debugOn);

  foreach (@retValues) {
    return $_;
  }
  return 0;
}

#
# reset_seed_pdb_mode - close PDB$SEED on all instances and open it in the 
#                       specified mode
#
# parameters:
#   - connect string
#   - mode in which PDB$SEED is to be opened
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - indicator of whether to produce debugging info
#
# Bug 14248297: close PDB$SEED on all RAC instances. If the PDB$SEED is to be
# opened on all instances, then the 'seedMode' argument must specify it.
#
sub reset_seed_pdb_mode ($$$$$) {
  my ($myConnect, $seedMode, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

  my @ResetSeedModeStatements = (
    "connect $myConnect\n",
    qq#alter session set "_oracle_script"=TRUE\n/\n#,
    "alter pluggable database pdb\$seed close immediate instances=all\n/\n",
    "alter pluggable database pdb\$seed $seedMode\n/\n",
  );

  exec_DB_script(@ResetSeedModeStatements, undef, $DoneCmd, 
                 $DoneFilePathBase, $debugOn);
}

#
# shutdown_db - shutdown the database in specified mode
#
# parameters:
#   - connect string
#   - shutdown mode
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - indicator of whether to produce debugging info
#
sub shutdown_db ($$$$$) {
  my ($myConnect, $shutdownMode, $DoneCmd, $DoneFilePathBase, $debugOn) = @_;

  my @ShutdownStatements = (
    "connect $myConnect\n",
    "SHUTDOWN $shutdownMode\n",
  );

  exec_DB_script(@ShutdownStatements, undef, $DoneCmd, 
                 $DoneFilePathBase, $debugOn);
}

# validate_script_path
#
# Parameters:
#   $FileName - name of file to validate
#   $Dir      - directory, if any, in which the file is expected to be found
#   $DebugOn  - an indicator of whether to produce diagnostic messages
#
# Description:
#   construct file's path using its name and optional directory and determine 
#   if it exists and is readable
#
# Returns:
#   file's path
sub validate_script_path ($$$) {
  my ($FileName, $Dir, $DebugOn) = @_;

  my $Path;                                                       # file path
    
  if ($DebugOn) {
    print STDERR "validate_script_path: getting ready to construct path for script $FileName\n";
  }

  if ($Dir) {
    $Path = $Dir."/".$FileName;
  } else {
    $Path = $FileName;
  }

  if ($DebugOn) {
    print STDERR "validate_script_path: getting ready to validate script $Path\n";
  }

  stat($Path);

  if (! -e _ || ! -r _) {
    print STDERR "validate_script_path: sqlplus script $Path does not exist or is unreadable\n";
    return undef;
  }

  if (! -f $Path) {
    print STDERR "validate_script_path: supposed sqlplus script $Path is not a regular file\n";
    return undef;
  }

  if ($DebugOn) {
    print STDERR "validate_script_path: successfully validated script $Path\n";
  }

  return $Path;
}

#
# err_logging_tbl_stmt - construct and issue SET ERRORLOGGING ON [TABLE ...]
#                        statement, if any, that needs to be issued to create 
#                        an Error Logging table
#
# parameters:
#   - an indicator of whether to save error logging information, which may be 
#     set to ON to create a default error logging table or to the name of an 
#     existing error logging table (IN)
#   - reference to an array of file handles containing a handle to which to 
#     send SET ERRORLOGGING statement (IN)
#   - process number (IN)
#   - an indicator of whether to produce debugging info (IN)
#
sub err_logging_tbl_stmt ($$$$) {
  my ($ErrLogging, $FileHandles_REF, $CurProc, $DebugOn) =  @_;

  # NOTE:
  # I have observed that if you issue 
  #   SET ERRORLOGGING ON IDENTIFIER ...
  # in the current Container after issuing
  #   SET ERRORLOGGING ON [TABLE ...] in a different Container, 
  # the error logging table will not be created in the current 
  # Container.  
  # 
  # To address this issue, I will first issue 
  #   SET ERRORLOGING ON [TABLE <table-name>]
  # and only then issue SET ERRORLOGGING ON [IDENTIFIER ...]
  #
  # To make sure that the error logging table does not get 
  # created as a Metadata Link, I will temporarily reset 
  # _ORACLE_SCRIPT parameter before issuing 
  #   SET ERRORLOGING ON [TABLE <table-name>]

  my $ErrLoggingStmt;

  if ($ErrLogging) {

    if ($DebugOn) {
      print STDERR "err_logging_tbl_stmt: ErrLogging = $ErrLogging\n";
    }

    # if ERRORLOGGING is to be enabled, construct the SET ERRORLOGGING 
    # statement which will be sent to every process
    $ErrLoggingStmt = "SET ERRORLOGGING ON ";

    if ((lc $ErrLogging) ne "on") {
      # customer supplied error logging table name
      $ErrLoggingStmt .= qq#TABLE "$ErrLogging"#;
    }

    if ($DebugOn) {
      print STDERR "err_logging_tbl_stmt: ErrLoggingStmt = $ErrLoggingStmt\n";
    }

    if ($DebugOn) {
      print STDERR "err_logging_tbl_stmt: temporarily resetting _parameter before issuing SET ERRORLOGGING ON [TABLE ...] in process $CurProc\n";
    }

    print {$FileHandles_REF->[$CurProc]} qq#ALTER SESSION SET "_ORACLE_SCRIPT"=FALSE\n/\n#;

    # send SET ERRORLOGGING ON [TABLE ...] statement
    if ($DebugOn) {
      print STDERR "err_logging_tbl_stmt: sending $ErrLoggingStmt to process $CurProc\n";
    }

    print {$FileHandles_REF->[$CurProc]} "$ErrLoggingStmt\n";

    if ($DebugOn) {
      print STDERR "err_logging_tbl_stmt: setting _parameter after issuing SET ERRORLOGGING ON [TABLE ...] in process $CurProc\n";
    }

    print {$FileHandles_REF->[$CurProc]} qq#ALTER SESSION SET "_ORACLE_SCRIPT"=TRUE\n/\n#;
  }

  return $ErrLoggingStmt;
}

#
# start_processes - start processes
#
# parameters:
#   - number of processes to start (IN)
#   - base for constructing log file names (IN)
#   - reference to an array of file handles; will be obtained as a 
#     side-effect of calling open() (OUT)
#   - reference to an array of process ids; will be obtained by calls 
#     to open (OUT)
#   - reference to an array of Container names; array will be empty if 
#     we are operating against a non-Consolidated DB (IN)
#   - connect string used to connect to a DB (IN)
#   - an indicator of whether SET ECHO ON should be sent to SQL*Plus 
#     process (IN)
#   - an indicator of whether to save ERRORLOGGING information (IN)
#   - an indicator of whether to produce debugging info (IN)
#   - an indicator of whether processes are being started for the first 
#     time (IN)
#
sub start_processes ($$\@\@\@$$$$$$) {
  my ($NumProcs, $LogFilePathBase, $FileHandles, $ProcIds, $Containers, $Root,
      $ConnectString, $EchoOn, $ErrLogging, $DebugOn, $FirstTime) =  @_;

  my $ps;

  my $CurrentContainerQuery = qq#select '==== Current Container = ' || SYS_CONTEXT('USERENV','CON_NAME') || ' ====' AS now_connected_to from dual\n/\n#;

  if ($DebugOn) {
    print STDERR "start_processes: will start $NumProcs processes\n";
  }

  # 14787047: Save STDOUT so we can restore it after we start each process
  open(SAVED_STDOUT, ">&STDOUT");
  # Keep Perl happy and avoid the warning "SAVED_STDOUT used only once"
  print SAVED_STDOUT "";

  # remember whether processes will be running against one or more Containers 
  # of a CDB, starting with the Root
  my $switchIntoRoot = (@$Containers && ($Containers->[0] eq $Root));

  for ($ps=0; $ps < $NumProcs; $ps++) {
    my $LogFile = $LogFilePathBase.$ps.".log";

    # If starting for the first time, open for write; otherwise append
    if ($FirstTime) {
      open (STDOUT,">", "$LogFile");
    } else {
      close (STDOUT);
      open (STDOUT,"+>>", "$LogFile");
    }

    my $id = open ($FileHandles->[$ps], "|-", "sqlplus /nolog");  
    push(@$ProcIds, $id);
      
    if ($DebugOn) {
      print STDERR "start_processes: process $ps (id = $ProcIds->[$#$ProcIds]) will use log file $LogFile\n";
    }

    # file handle for the current process
    my $fh = $FileHandles->[$ps];

    # send initial commands to the sqlplus session
    print $fh "connect $ConnectString\n";

    if ($DebugOn) {
      print STDERR qq#start_processes: connected using $ConnectString\n#;
    }

    # use ALTER SESSION SET APPLICATION MODULE/ACTION to identify this process
    print $fh "ALTER SESSION SET APPLICATION MODULE = 'catcon(pid=$PID)'\n/\n";
    if ($DebugOn) {
      print STDERR "start_processes: issued ALTER SESSION SET APPLICATION MODULE = 'catcon(pid=$PID)'\n";
    }

    print $fh "ALTER SESSION SET APPLICATION ACTION = 'started'\n/\n";
    if ($DebugOn) {
      print STDERR "start_processes: issued ALTER SESSION SET APPLICATION ACTION = 'started'\n";
    }

    if ($EchoOn) {
      print $fh "SET ECHO ON\n";

      if ($DebugOn) {
        print STDERR "start_processes: SET ECHO ON has been issued\n";
      }
    }
    
    # if running scripts against one or more Containers of a Consolidated 
    # Database, 
    # - if the Root is among Containers against which scripts will 
    #   be run (in which case it will be found in $Containers->[0]), switch 
    #   into it (because code in catconExec() expects that to be the case); 
    #   if scripts will not be run against the Root, there is no need to 
    #   switch into any specific Container - catconExec will handle that case 
    #   just fine all by itself
    # - turn on the "secret" parameter to ensure correct behaviour 
    #   of DDL statements; it is important that we do it after we issue 
    #   SET ERRORLOGGING ON to avoid creating error logging table as a 
    #   Common Table
    if (@$Containers) {
      if ($switchIntoRoot) {
        print $fh qq#ALTER SESSION SET CONTAINER = $Root\n/\n#;

        if ($DebugOn) {
          print STDERR qq#start_processes: switched into Container $Root\n#;
        }
      }

      print $fh qq#ALTER SESSION SET "_ORACLE_SCRIPT"=TRUE\n/\n#;

      if ($DebugOn) {
        print STDERR "start_processes: _parameter was set\n";
      }
    }
  }

  # 14787047: Restore saved stdout
  close (STDOUT);
  open (STDOUT, ">&SAVED_STDOUT");
}

#
# end_processes - end all processes
#
# parameters:
#   - index of the first process to end (IN)
#   - index of the last process to end (IN)
#   - reference to an array of file handles (IN)
#   - reference to an array of process ids; will be cleared of its 
#     elements (OUT)
#   - an indicator of whether to produce debugging info (IN)
#
sub end_processes ($$\@\@$) {
  my ($FirstProcIdx, $LastProcIdx, $FileHandles, $ProcIds, $DebugOn) =  @_;

  my $ps;

  if ($FirstProcIdx < 0) {
    print STDERR "end_processes: FirstProcIdx ($FirstProcIdx) was less than 0\n";
    return 1;
  }

  if ($LastProcIdx < $FirstProcIdx) {
    print STDERR "end_processes: LastProcIdx ($LastProcIdx) was less than  FirstProcIdx ($FirstProcIdx)\n";
    return 1;
  }

  if ($DebugOn) {
    print STDERR "end_processes: will end processes $FirstProcIdx to $LastProcIdx\n";
  }

  $SIG{CHLD} = 'IGNORE';

  for ($ps = $FirstProcIdx; $ps <= $LastProcIdx; $ps++) {
    if ($FileHandles->[$ps]) {
      print {$FileHandles->[$ps]} "PROMPT ========== PROCESS ENDED SUCCESSFULLY ==========\n";
      print {$FileHandles->[$ps]} "EXIT\n";
      close ($FileHandles->[$ps]);
      $FileHandles->[$ps] = undef;
    } elsif ($DebugOn) {
      print STDERR "end_processes: process $ps has already been stopped\n";
    }
  }

  splice @$ProcIds, $FirstProcIdx, $LastProcIdx - $FirstProcIdx + 1;

  if ($DebugOn) {
    print STDERR "end_processes: ended processes $FirstProcIdx to $LastProcIdx\n";
  }

  return 0;
}

#
# clean_up_compl_files - purge files indicating completion of processes, up 
#            to a specified process number
#
# parameters:
#   - base of process completion file name
#   - number of processes whose completion files are to be purged
#   - an indicator of whether to print debugging info
#
sub clean_up_compl_files ($$$) {
  my ($FileNameBase, $NumProcs, $DebugOn) =  @_;

  my $ps;

  if ($DebugOn) {
    print STDERR qq#clean_up_compl_files: FileNameBase = $FileNameBase, NumProcs = $NumProcs\n#;
  }

  for ($ps=0; $ps < $NumProcs; $ps++) {
    my $FileName = $FileNameBase.$ps.".done";
    
    if (-e $FileName && -f $FileName) {
      sureunlink($FileName, $DebugOn);
      
      if ($DebugOn) {
        print STDERR qq#clean_up_compl_files: removed $FileName\n#;
      }
    }
  }
}

#
# get_log_file_base_path - determine base path for log files
#
# Parameters:
#   - log directory, as specified by the user
#   - base for log file names
#   - an indicator of whether to produce debugging info
#
sub get_log_file_base_path ($$$) {
  my ($LogDir, $LogBase, $DebugOn) = @_;

  if ($LogDir) {
    if ($DebugOn) {
      print STDERR "get_log_file_base_path: log file directory = $LogDir\n";
    } 

    return ($LogDir."/".$LogBase);
  } else {
    if ($DebugOn) {
      print STDERR "get_log_file_base_path: no log file directory was specified\n";
    }

    return $LogBase;
  }
}

#
# send_sig_to_procs - given an array of process ids, send specified signal to 
#                     these processes 
#
# Parameters:
#   - reference to an array of process ids
#   - signal to send
#
# Returns:
#   number of processes whose ids were passed to us which got the signal
#
sub send_sig_to_procs (\@$) {
  my ($ProcIds, $Sig)  =  @_;
  
  return  kill($Sig, @$ProcIds);
}

#
# next_proc - determine next process which has finished work assigned to it
#             (and is available to execute a script or an SQL statement)
#
# Description:
#   This subroutine will look for a file ("done" file) indicating that the 
#   process has finished running a script or a statement which was sent to it.
#   Once such file is located, it will be deleted and a number of a process 
#   to which that file corresponded will be returned to the caller
#
# Parameters:
#   - number of processes from which we may choose the next available process
#   - total number of processes which were started (used when we try to
#     determine if some processes may have died)
#   - number of the process starting with which to start our search; if the 
#     supplied number is greater than $ProcsUsed, it will be reset to 0
#   - reference to an array of booleans indicating status of which processes 
#     should be ignored; may be undefined
#   - base for names of files whose presense will indicate that a process has 
#     completed
#   - reference to an array of process ids
#   - an indicator of whether we are running under Windows
#   - an indicator of whether to display diagnostic info
#
sub next_proc ($$$$$$$$) {
  my ($ProcsUsed, $NumProcs, $StartingProc, $ProcsToIgnore, 
      $FilePathBase, $ProcIds, $Windows, $DebugOn) = @_;

  if ($DebugOn) {
      print STDERR <<next_proc_DEBUG;
  running next_proc(ProcsUsed    = $ProcsUsed, 
                    NumProcs     = $NumProcs,
                    StartingProc = $StartingProc,
                    FilePathBase = $FilePathBase,
                    Windows      = $Windows,
                    DebugOn      = $DebugOn);

next_proc_DEBUG
  }

  # process number; will be used to construct names of "done" files
  # before executing the while loop for the first time, it will be set to 
  # $StartingProc.  If that number is outside of [0, $ProcsUsed-1], 
  # it [$CurProc] will be reset to 0; for subsequent iterations through the 
  # while loop, $CurProc will start with 0
  my $CurProc = ($StartingProc >= 0 && $StartingProc <= $ProcsUsed-1) 
    ? $StartingProc : 0;
    
  # look for *.done files which will indicate which processes have 
  # completed their work

  # we may end up waiting a while before finding an available process and if 
  # debugging is turned on, user's screen may be flooded with 
  #     next_proc: Skip checking process ...
  # and
  #     next_proc: Checking if process ... is available
  # messages.  
  #
  # To avoid this, we will print this message every 10 second or so.  Since 
  # we check for processes becoming available every 0.01 of a second (or so), 
  # we will report generate debugging messages every 1000-th time through the 
  # loop
  my $itersBetweenMsgs = 1000;

  for (my $numIters = 0; ; $numIters++) {
    #
    # Make sure no sql processors died (Windows only).
    # For Unix this handle through signals.
    #
    if ($Windows) {
      # sending signal 0 to processes whose ids are stored in an array will 
      # return a number of processes to which this signal could be 
      # delivered, ergo which are alive
      my $LiveProcs = send_sig_to_procs(@$ProcIds, 0);
      
      if ($NumProcs != $LiveProcs) {

        print STDERR qq#next_proc: total processes ($NumProcs) != number of live processes ($LiveProcs); giving up\n#;

        catconWrapUp();
        send_sig_to_procs(@$ProcIds, 9);
        clean_up_compl_files($FilePathBase, $ProcsUsed, $DebugOn);
        catcon_HandleSigchld();

        return -1;
      }
    }

    for (; $CurProc < $ProcsUsed; $CurProc++) {
        
      if (   $ProcsToIgnore && @$ProcsToIgnore 
          && $ProcsToIgnore->[$CurProc]) {
        if ($DebugOn && $numIters % $itersBetweenMsgs == 0) {
          print STDERR "next_proc: Skip checking process $CurProc\n";
        }

        next;
      }

      # file which will indicate that process $CurProc finished its work
      my $DoneFile = $FilePathBase.$CurProc.".done";

      if ($DebugOn && $numIters % $itersBetweenMsgs == 0) {
        print STDERR "next_proc: Checking if process $CurProc is available\n";
      }

      #
      # Is file is present, remove the "done" file (thus making this process 
      # appear "busy") and return process number to the caller.
      #
      if (-e $DoneFile) {
        sureunlink($DoneFile, $DebugOn);

        if ($DebugOn) {
          print STDERR "next_proc: process $CurProc is available\n";
        }

        return $CurProc;
      }
    }
    select (undef, undef, undef, 0.01);

    $CurProc = 0;
  }
  
  return -1;  # this statement will never be rached
}

#
# wait_for_completion - wait for completion of processes
#
# Description:
#   This subroutine will wait for processes to indicate that they've 
#   completed their work by creating a "done" file.  Since it will use 
#   next_proc() (which deleted "done" files of processes whose ids it returns) 
#   to find processes which are done running, this subroutine will generate 
#   "done" files once all processes are done to indicate that they are 
#   available to take on more work.
#
# Parameters:
#   - number of processes which were used (and whose completion we need to 
#     confirm)
#   - total number of processes which were started
#   - base for generating names of files whose presense will indicate that a 
#     process has completed
#   - references to an array of process file handles
#   - reference to a array of process ids
#   - command which will be sent to a process to cause it to generate a 
#     "done" file
#   - an indicator of whether we are running under Windows
#   - reference to an array of statements, if any, to be issued when a process 
#     completes
#   - an indicator of whether to display diagnostic info
#
sub wait_for_completion ($$$\@\@$$\@$) {
  my ($ProcsUsed, $NumProcs, $FilePathBase, $ProcFileHandles, 
      $ProcIds, $DoneCmd, $Windows, $EndStmts, $DebugOn) = @_;

  if ($DebugOn) {
      print STDERR <<wait_for_completion_DEBUG;
  running wait_for_completion(ProcsUsed    = $ProcsUsed, 
                              FilePathBase = $FilePathBase,
                              DoneCmd      = $DoneCmd,
                              Windows      = $Windows,
                              DebugOn      = $DebugOn);

wait_for_completion_DEBUG
  }

  # process number
  my $CurProc = 0;
    
  if ($DebugOn) {
    print STDERR "wait_for_completion: waiting for $ProcsUsed processes to complete\n";
  }

  # look for *.done files which will indicate which processes have 
  # completed their work

  my $NumProcsCompleted = 0;  # how many processes have completed

  # this array will be used to keep track of processes which have completed 
  # so as to avoid checking for existence of files which have already been 
  # seen and removed
  my @ProcsCompleted = (0) x $ProcsUsed;

  while ($NumProcsCompleted < $ProcsUsed) {
    $CurProc = next_proc($ProcsUsed, $NumProcs, $CurProc + 1, \@ProcsCompleted,
                         $FilePathBase, $ProcIds, $Windows, $DebugOn);

    if ($CurProc < 0) {
      print STDERR qq#wait_for_completion: unexpected error in next_proc()\n#;
      return 1;
    }

    # if the caller has supplied us with statements to run after a 
    # process finished its work, do it now
    if ($EndStmts && $#$EndStmts >= 0) {

      if ($DebugOn) {
        print STDERR "wait_for_completion: sending completion statements to process $CurProc:\n";
      }

      foreach my $Stmt (@$EndStmts) {

        # $Stmt may contain %proc strings which need to be replaced with 
        # process number, but we don't want to modify the statement in 
        # @$EndStmts, so we modify a copy of it
        my $Stmt1;

        ($Stmt1 = $Stmt) =~ s/%proc/$CurProc/g;
        
        if ($DebugOn) {
          print STDERR "\t$Stmt1\n";
        }

        print {$ProcFileHandles->[$CurProc]} "$Stmt1\n";
      }

      $ProcFileHandles->[$CurProc]->flush; # flush the buffer
    }

    if ($DebugOn) {
      print STDERR "wait_for_completion: process $CurProc is done\n";
    }

    $NumProcsCompleted++; # one more process has comleted

    # remember that this process has completed so next_proc does not try to 
    # check its status
    $ProcsCompleted[$CurProc] = 1;
  }

  if ($DebugOn) {
    print STDERR "wait_for_completion: All $NumProcsCompleted processes have completed\n";
  }
  
  # issue statements to cause "done" files to be created to indicate
  # that all $ProcsUsed processes are ready to take on more work
  for ($CurProc = 0; $CurProc < $ProcsUsed; $CurProc++) {

    # file which will indicate that process $CurProc finished its work
    my $DoneFile = $FilePathBase.$CurProc.".done";

    print {$ProcFileHandles->[$CurProc]} qq/$DoneCmd $DoneFile\n/;
    $ProcFileHandles->[$CurProc]->flush;

    if ($DebugOn) {
      print STDERR qq#wait_for_completion: sent "$DoneCmd $DoneFile" to process $CurProc (id = $ProcIds->[$CurProc])\n\tto indicate that it is available to take on more work\n#;
    }
  }

  return 0;
}

#
# return a timestamp in yyyy-mm-dd h24:mi:sec format
#
sub TimeStamp {
  my ($sec,$min,$hour,$mday,$mon,$year)=localtime(time);

  return sprintf "%4d-%02d-%02d %02d:%02d:%02d",
                 $year+1900,$mon+1,$mday,$hour,$min,$sec;
}

#
# getSpoolFileNameSuffix - generate suffix for a spool file names using 
#   container name supplied by the caller
#
# Parameters:
# - container name (IN)
#
sub getSpoolFileNameSuffix ($) {
 my ($SpoolFileNameSuffix) = @_;

 # $SpoolFileNameSuffix may contain characters which may be 
 # illegal in file name - replace them all with _
 $SpoolFileNameSuffix =~ s/\W/_/g;

 return lc($SpoolFileNameSuffix);
}

#
# pickNextProc - pick a process to run next statement or script
#
# Parameters:
#   (IN) unless indicated otherwise
#   - number of processes from which we may choose the next available process 
#   - total number of processes which were started (used when we try to
#     determine if some processes may have died)
#   - number of the process starting with which to start our search; if the 
#     supplied number is greater than $ProcsUsed, it will be reset to 0
#   - base for names of files whose presense will indicate that a process has 
#     completed
#   - reference to an array of process ids
#   - an indicator of whether we are running under Windows
#   - a command to create a file whose existence will indicate that the 
#     last statement of the script has executed (needed by exec_DB_script())
#   - base for a name of a "done" file (see above)
#   - connect string
#   - an indicator of whether to display diagnostic info
# 
sub pickNextProc ($$$$$$$$$$) {
  my ($ProcsUsed, $NumProcs, $StartingProc, $FilePathBase, 
      $ProcIds, $Windows, $DoneCmd, $DoneFilePathBase,
      $ConnectString, $DebugOn) = @_;

  # find next available process
  my $CurProc = next_proc($ProcsUsed, $NumProcs, $StartingProc, undef,
                          $FilePathBase, $ProcIds, $Windows, $DebugOn);
  if ($CurProc < 0) {
    # some unexpected error was encountered
    print STDERR "pickNextProc: unexpected error in next_proc\n";

    return -1;
  }

  return $CurProc;
}

#
# firstProcUseStmts - issue statements to a process that is being used for 
#                     the first time
# Parameters:
#   - value of IDENTIFIER if ERRORLOGGING is enabled (IN)
#   - name of error logging table, if defined (IN)
#   - reference to an array of file handle references (IN)
#   - number of the process about to be used (IN)
#   - reference to an array of statements which will be executed in every 
#     process before it is asked to run the first script (IN)
#   - array used to keep track of processes to which statements contained in 
#     @$PerProcInitStmts_REF need to be sent because they [processes]
#     have not been used in the course of this invocation of catconExec()
#   - an indicator of whether debugging is enabled (IN)
#
sub firstProcUseStmts($$$$$$$) {
  my ($ErrLoggingIdent, $ErrLogging, $FileHandles_REF, $CurProc, 
      $PerProcInitStmts_REF, $NeedInitStmts_REF, $DebugOn) = @_;
  
  if ($ErrLoggingIdent) {
    # construct and issue SET ERRORLOGGING ON [TABLE...] statement
    err_logging_tbl_stmt($ErrLogging, $FileHandles_REF, $CurProc, $DebugOn);

    # send SET ERRORLOGGING ON IDENTIFIER ... statement
    my $Stmt = "SET ERRORLOGGING ON IDENTIFIER '".substr($ErrLoggingIdent."InitStmts", 0, 256)."'\n";

    if ($DebugOn) {
      print STDERR "firstProcUseStmts: sending $Stmt to process $CurProc\n";
    }

    print {$FileHandles_REF->[$CurProc]} $Stmt;
  }

  if ($DebugOn) {
    print STDERR "firstProcUseStmts: sending init statements to process $CurProc:\n";
  }

  foreach my $Stmt (@$PerProcInitStmts_REF) {

    # $Stmt may contain %proc strings which need to be replaced with 
    # process number, but we don't want to modify the statement in 
    # @$PerProcInitStmts_REF, so we modify a copy of its element
    my $Stmt1;

    ($Stmt1 = $Stmt) =~ s/%proc/$CurProc/g;

    if ($DebugOn) {
      print STDERR "\t$Stmt1\n";
    }

    print {$FileHandles_REF->[$CurProc]} "$Stmt1\n";
  }

  # remember that "initialization" statements have been run for this 
  # process
  $NeedInitStmts_REF->[$CurProc] = 0;

  return;
}

#
# additionalInitStmts - issue additional initialization statements after 
#   picking a new process to run a script in a given Container
#
# Parameters: (all parameters are IN unless noted otherwise)
#   - reference to an array of file handles used to communicate to processes
#   - process number of the process which was picked to run the next script
#   - connect string
#   - name of the Root Container
#   - name of the container against which the next script will be run
#     (can be null if running against ROOT or in a non-consolidated DB)
#   - query which will identify current container in the log
#     (can be null if running against ROOT or in a non-consolidated DB)
#   - process id of the process hich was picked to run the next script
#   - an inidcator of whether to produce debugging info
#
sub additionalInitStmts ($$$$$$$$$) {
  my ($FileHandles_REF, $CurProc, $ConnectString, $RootName, $CurConName,
      $CurrentContainerQuery, $CurProcId, $EchoOn, $DebugOn) = @_;

  # WORKAROUND for some issues with session-cached information across PDBs
  print {$FileHandles_REF->[$CurProc]} "connect $ConnectString\n";
  print {$FileHandles_REF->[$CurProc]} qq#ALTER SESSION SET "_ORACLE_SCRIPT"=TRUE\n/\n#;
  # END OF WORKAROUND for some issues with session-cached information
  # across PDBs
            
  # set tab and trimspool to on, so that spooled log files start 
  # with a consistent setting
  print {$FileHandles_REF->[$CurProc]} "SET TAB ON\n";
  print {$FileHandles_REF->[$CurProc]} "SET TRIMSPOOL ON\n";

  # ensure that common SQL*PLus vars that affect appearance of log files are 
  # set to consistent values.  This will ensure that vars modified in one 
  # script do not affect appearance of log files produced by running another 
  # script or running the same script in a different Container
  print {$FileHandles_REF->[$CurProc]} qq/set colsep ' '\n/;
  print {$FileHandles_REF->[$CurProc]} qq/set escape off\n/;
  print {$FileHandles_REF->[$CurProc]} qq/set feedback 6\n/;
  print {$FileHandles_REF->[$CurProc]} qq/set heading on\n/;
  print {$FileHandles_REF->[$CurProc]} qq/set linesize 80\n/;
  print {$FileHandles_REF->[$CurProc]} qq/set long 80\n/;
  print {$FileHandles_REF->[$CurProc]} qq/set newpage 1\n/;
  print {$FileHandles_REF->[$CurProc]} qq/set numwidth 10\n/;
  print {$FileHandles_REF->[$CurProc]} qq/set pagesize 14\n/;
  print {$FileHandles_REF->[$CurProc]} qq/set recsep wrapped\n/;
  print {$FileHandles_REF->[$CurProc]} qq/set showmode off\n/;
  print {$FileHandles_REF->[$CurProc]} qq/set sqlprompt "SQL> "\n/;
  print {$FileHandles_REF->[$CurProc]} qq/set termout on\n/;
  print {$FileHandles_REF->[$CurProc]} qq/set time off\n/;
  print {$FileHandles_REF->[$CurProc]} qq/set timing off\n/;
  print {$FileHandles_REF->[$CurProc]} qq/set trimout on\n/;
  print {$FileHandles_REF->[$CurProc]} qq/set underline on\n/;
  print {$FileHandles_REF->[$CurProc]} qq/set verify on\n/;
  print {$FileHandles_REF->[$CurProc]} qq/set wrap on\n/;
          
  # for the time being, if we are connected to a PDB, we need to 
  # switch into the Root before switching into another PDB
  if ($CurConName) {
    print {$FileHandles_REF->[$CurProc]} qq#ALTER SESSION SET CONTAINER = "$RootName"\n/\n#;

    if ($DebugOn) {
      print STDERR "additionalInitStmts: process $CurProc (id = $CurProcId) connected to Root Container ($RootName) before switching to a specific PDB\n";
    }
  }

  if ($CurConName) {
    print {$FileHandles_REF->[$CurProc]} qq#ALTER SESSION SET CONTAINER = "$CurConName"\n/\n#;

    if ($DebugOn) {
      print STDERR "additionalInitStmts: process $CurProc (id = $CurProcId) connected to Container $CurConName\n";
    }
  }

  if ($CurrentContainerQuery) {
    print {$FileHandles_REF->[$CurProc]} $CurrentContainerQuery;
  }

  if ($EchoOn) {
    print {$FileHandles_REF->[$CurProc]} "SET ECHO ON\n";
  }

  return;
}

# subroutines which may be invoked by callers outside this file and variables 
# that need to persist across such invocations
{
  # have all the vars that need to persist across calls been initialized?
  # computed
  my $catcon_InitDone;
  
  # name of the directory for sqlplus script(s), as supplied by the caller
  my $catcon_SrcDir;

  # name of the directory for log file(s), as supplied by the caller
  my $catcon_LogDir;

  # base for paths of log files
  my $catcon_LogFilePathBase;

  # number of processes which will be used to run sqlplus script(s) or SQL 
  # statement(s) using this instance of catcon.pl, as supplied by the caller; 
  my $catcon_NumProcesses;

  # indicator of whether echo should be turned on while running sqlplus 
  # script(s) or SQL statement(s), as supplied by the caller
  my $catcon_EchoOn;

  # indicator of whether output of running scripts should be spooled into 
  # Container- and script-specific spool files, as supplied by the caller
  my $catcon_SpoolOn;

  # indicator of whether debugging info should be generated while executing 
  # various subroutines in this block
  my $catcon_DebugOn;

  # connect string used when running SQL for internal statements: this
  # should be done as sys
  # computed
  my $catcon_InternalConnectString;        

  # connect string used when running sqlplus script(s) or SQL statement(s) in
  # the context of the user passed in.
  # computed
  my $catcon_UserConnectString;        

  # container names; empty if connected to a non-Consolidated Database;
  # computed
  my @catcon_Containers;

  # name of the Root if operating on a Consolidated DB
  my $catcon_Root;

  # array of file handle references; 
  my @catcon_FileHandles;

  # array of process ids
  my @catcon_ProcIds;

  # are we running on Windows?
  my $catcon_Windows;
  my $catcon_DoneCmd;

  # string introducing an argument to SQL*PLus scripts which will be supplied 
  # using clear text
  my $catcon_RegularArgDelim;

  # string introducing an argument to SQL*PLus scripts which will have to be 
  # specified by the user at run-time
  my $catcon_SecretArgDelim;

  # if strings used to introduce "regular" as well as secret arguments are 
  # specified and one is a prefix of the other and we encounter a string 
  # that could be either a regular or a secret argument, we will resolve in 
  # favor of the longer string.  
  # $catcon_ArgToPick will be used to determine how to resolve such conflicts, 
  # should they occur; it will remain set to 0 if such conflicts cannot occur 
  # (i.e. if both strings are not specified or if neither is a prefix of the 
  # other)

  # pick regular argument in the event of conflict
  my $catcon_PickRegularArg;
  # pick secret argument in the event of conflict
  my $catcon_PickSecretArg;

  my $catcon_ArgToPick = 0;

  # reference to an array of statements which will be executed in every 
  # process before it is asked to run the first script
  my $catcon_PerProcInitStmts;

  # reference to an array of statements which will be executed in every 
  # process after it finishes runing the last script
  my $catcon_PerProcEndStmts;

  # if defined, will contain name of error logging table
  my $catcon_ErrLogging;

  # will act as a reminder of whether PDB$SEED needs to be reopened in 
  # READ ONLY mode (because we closed and reopened it in READ WRITE mode in 
  # order to run script(s) or SQL statements against it)
  #
  # NOTE: if PDB$SEED is opened in READ ONLY mode when catconExec() discovers 
  #       for the first time that it needs to run some scripts or statements 
  #       against it, it will call reset_seed_pdb_mode() which will close it 
  #       and then reopen in READ WRITE mode.  From that point on, PDB$SEED 
  #       will remain in READ WRITE mode until it gets reset to READ ONLY as 
  #       a part of catconWrapUp().  
  #
  #       It is IMPOTANT that restoring PDB$SEED's mode happens AFTER all 
  #       processes opened in catconInit() have been killed because otherwise 
  #       ALTER PDB CLOSE IMMEDIATE issued by reset_seed_pdb_mode() will 
  #       cause any of the proceses which happen to be connected to PDB$SEED
  #        to become unusable (they will be disconnected from the database 
  #       and any statements sent to them will fail.)  
  #       As a side note, before we fixed bug 13072385, reset_seed_pdb_mode() 
  #       used to issue ALTER PDB CLOSE (without IMMEDIATE), which hung if 
  #       any process was connected to PDB$SEED
  #
  my $catcon_RevertSeedPdbMode = 0;

  #
  # catconInit - initialize and validate catcon static vars and start 
  #              SQL*Plus processes
  #
  # Parameters:
  #   - user name, optionally with password, supplied by the caller; may be 
  #     undefined (default sys/knl_test7 AS SYSDBA)
  #   - directory containing sqlplus script(s) to be run; may be undefined
  #   - directory to use for spool and log files; may be undefined
  #   - base for spool log file name; must be specified
  #
  #   - container(s) (names separated by one or more spaces) in which to run 
  #     sqlplus script(s) and SQL statement(s) (i.e. skip containers not 
  #     referenced in this list)
  #   - container(s) (names separated by one or more spaces) in which NOT to 
  #     run sqlplus script(s) and SQL statement(s) (i.e. skip containers
  #     referenced in this list)
  #     NOTE: the above 2 args are mutually exclusive; if neither is defined, 
  #       sqlplus script(s) and SQL statement(s) will be run in the 
  #       non-Consolidated Database or all Containers of a Consolidated 
  #       Database
  #
  #   - number of processes to be used to run sqlplus script(s) and SQL 
  #     statement(s);  may be undefined (default value will be computed based 
  #     on host's hardware characteristics, number of concurrent sessions, 
  #     and whether the subroutine is getting invoked interactively)
  #   - external degree of parallelism, i.e. if more than one script using 
  #     catcon will be invoked simultaneously on a given host, this parameter 
  #     should contain a number of such simultaneous invocations; 
  #     may be undefined;
  #     will be used to determine number of processes to start;
  #     this parameter MUST be undefined or set to 0 if the preceeding 
  #     parameter (number of processes) is non-zero
  #   - indicator of whether echo should be set ON while running script(s) 
  #     and SQL statement(s); defaults to FALSE
  #   - indicator of whether output of running scripts should be spooled in 
  #     files stored in the same directory as that used for log files and 
  #     whose name will be constructed as follows:
  #       <log-file-name-base>_<script_name_without_extension>_[<container_name_if_any>].<default_extension>
  #   - reference to a string used to introduce an argument to SQL*Plus 
  #     scripts which will be supplied using clear text (i.e. not require 
  #     user to input them at run-time)
  #     NOTE: variable referenced by this parameter will default to "--p" 
  #           if not supplied by the caller
  #   - reference to a string used to introduce an argument (such as a 
  #     password) to SQL*Plus scripts which will require user to input them 
  #     at run-time
  #     NOTE: variable referenced by this parameter will default to "--P" 
  #           if not supplied by the caller
  #   - indicator of whether ERRORLOGGING should be set ON while running 
  #     script(s) and SQL statement(s); defaults to FALSE; if value is ON, 
  #     default error logging table (SPERRORLOG) will be used, otherwise 
  #     specified value will be treated as the name of the error logging 
  #     table which should have been pre-created in every Container
  #   - reference to an array of statements which will be executed in every 
  #     process before it is asked to run the first script; if operating on a 
  #     CDB, statements will be executed against the Root; statements may 
  #     contain 0 or more instances of string %proc which will be replaced 
  #     with process number
  #   - reference to an array of statements which will be executed in every 
  #     process after it finishes runing the last script; if operating on a 
  #     CDB, statements will be executed against the Root; statements may 
  #     contain 0 or more instances of string %proc which will be replaced 
  #     with process number
  #   - indicator of whether to produce debugging messages; defaults to FALSE
  #
  sub catconInit ($$$$$$$$$$$\$\$$\@\@$) {
    my ($User, $InternalUser, $SrcDir, $LogDir, $LogBase, 
        $ConNamesIncl, $ConNamesExcl, 
        $NumProcesses, $ExtParallelDegree,
        $EchoOn, $SpoolOn, $RegularArgDelim_ref, $SecretArgDelim_ref, 
        $ErrLogging, $PerProcInitStmts, $PerProcEndStmts, $DebugOn) = @_;

    my $UnixDoneCmd = "\nhost sqlplus -v >";
    my $WindowsDoneCmd = "\nhost sqlplus/nolog -v >";

    # will contain status of the instance (v$instance.status)
    my $instanceStatus;

    # will contains an indicator of whether a DB is a CDB (v$database.cdb)
    my $IsCDB;

    # save DebugOn indicator in a variable that will persist across calls
    if ($catcon_DebugOn = $DebugOn) {
      my $ts = TimeStamp();

      # make STDERR "hot" so debugging output does not get buffered
      select((select(STDERR), $|=1)[0]);

      print STDERR <<catconInit_DEBUG;
  running catconInit(User              = $User, 
                     InternalUser      = $InternalUser,
                     SrcDir            = $SrcDir, 
                     LogDir            = $LogDir, 
                     LogBase           = $LogBase,
                     ConNamesIncl      = $ConNamesIncl, 
                     ConNamesExcl      = $ConNamesExcl, 
                     NumProcesses      = $NumProcesses,
                     ExtParallelDegree = $ExtParallelDegree,
                     EchoOn            = $EchoOn, 
                     SpoolOn           = $SpoolOn,
                     RegularArgDelim   = $$RegularArgDelim_ref,
                     SecretArgDelim    = $$SecretArgDelim_ref,
                     ErrLogging        = $ErrLogging,
                     Debug             = $DebugOn)\t\t($ts)

catconInit_DEBUG
    }

    # base for log file names must be supplied
    if (!$LogBase) {
      print STDERR "catconInit: Base for log file names must be supplied";
      return 1;
    }

    if ($NumProcesses && $ExtParallelDegree) {
      print STDERR "catconInit: you may not specify both the number of processes ($NumProcesses) and the external degree of parallelism ($ExtParallelDegree)\n";
      return 1;
    }

    if ($catcon_InitDone) {
      print STDERR "catconInit: script execution state has already been initialized; call catconWrapUp before invoking catconInit\n";
      return 1;
    }

    # save EchoOn indicator in a variable that will persist across calls
    $catcon_EchoOn = $EchoOn;

    # save SpoolOn indicator in a variable that will persist across calls
    $catcon_SpoolOn = $SpoolOn;

    # initialize indicators used to determine how to resolve conflicts 
    # between regular and secret argument delimiters
    $catcon_PickRegularArg = 1; 
    $catcon_PickSecretArg = 2;  

    # 
    # set up signal in case SQL process crashes
    # before it completes its work
    #
    $SIG{CHLD} = \&catcon_HandleSigchld;
    $SIG{INT} = \&catcon_HandleSigINT;

    # figure out if we are running under Windows and set $catcon_DoneCmd
    # accordingly

    $catcon_DoneCmd = ($catcon_Windows = ($OSNAME =~ /^MSWin/)) ? $WindowsDoneCmd : $UnixDoneCmd;

    if ($catcon_DebugOn) {
       print STDERR "catconInit: running on $OSNAME; DoneCmd = $catcon_DoneCmd\n";
    }

    #
    # unset TWO_TASK  or LOCAL if it happens to be set to ensure that CONNECT
    # which does not specify a service results in a connection to the Root
    #

    if($catcon_Windows) {
      if ($ENV{LOCAL}) {
         if ($catcon_DebugOn) {
            print STDERR "catconInit: LOCAL was set to $ENV{LOCAL} - unsetting it\n";
         }

         delete $ENV{LOCAL};
      } else {
        if ($catcon_DebugOn) {
            print STDERR "catconInit: LOCAL was not set, so there is not need to unset it\n";
        }
      }
    }

    if ($ENV{TWO_TASK}) {
      if ($catcon_DebugOn) {
        print STDERR "catconInit: TWO_TASK was set to $ENV{TWO_TASK} - unsetting it\n";
      }

      delete $ENV{TWO_TASK};
    } else {
      if ($catcon_DebugOn) {
        print STDERR "catconInit: TWO_TASK was not set, so there is no need to unset it\n";
      }
    }

    if (!$$RegularArgDelim_ref) {
      $$RegularArgDelim_ref = '--p';

      if ($catcon_DebugOn) {
        print STDERR "catconInit: regular argument delimiter was not specified - defaulting to $$RegularArgDelim_ref\n";
      }
    }

    if (!$$SecretArgDelim_ref) {
      $$SecretArgDelim_ref = '--P';

      if ($catcon_DebugOn) {
        print STDERR "catconInit: secret argument delimiter was not specified - defaulting to $$SecretArgDelim_ref\n";
      }
    }

    # save strings used to introduce arguments to SQL*Plus scripts
    if ($$RegularArgDelim_ref ne "--p" || $$SecretArgDelim_ref ne "--P") {

      if ($catcon_DebugOn) {
        print STDERR "catconInit: either regular ($$RegularArgDelim_ref) or secret ($$SecretArgDelim_ref) argument delimters were explicitly specified\n";
      }

      # if both are specified, they must not be the same
      if ($$RegularArgDelim_ref eq $$SecretArgDelim_ref) {
        print STDERR "catconInit: string introducing regular script argument ($$RegularArgDelim_ref) may not be \nthe same as a string introducing a secret script argument ($$SecretArgDelim_ref)\n";
        return 1;
      }

      # if one of the strings is a prefix of the other, remember which one to 
      # pick if an argument supplied by the user may be either
      if ($$RegularArgDelim_ref =~ /^$$SecretArgDelim_ref/) {
        $catcon_ArgToPick = $catcon_PickRegularArg;

        if ($catcon_DebugOn) {
          print STDERR "catconInit: secret argument delimiter ($$SecretArgDelim_ref) is a prefix of regular argument delimiter ($$RegularArgDelim_ref)\nif in doubt, will pick regular\n";
        }
      } elsif ($$SecretArgDelim_ref =~ /^$$RegularArgDelim_ref/) {
        $catcon_ArgToPick = $catcon_PickSecretArg;

        if ($catcon_DebugOn) {
          print STDERR "catconInit: regular argument delimiter ($$RegularArgDelim_ref) is a prefix of secret argument delimiter ($$SecretArgDelim_ref)\nif in doubt, will pick secret\n";
        }
      }
    }

    # argument delimiters may not start with @
    if ($$RegularArgDelim_ref =~ /^@/) {
      print STDERR "catconInit: regular argument delimiter ($$RegularArgDelim_ref) begins with @ which is not allowed\n";
      return 1;
    }

    if ($$SecretArgDelim_ref =~ /^@/) {
      print STDERR "catconInit: secret argument delimiter ($$SecretArgDelim_ref) begins with @ which is not allowed\n";
      return 1;
    }

    $catcon_RegularArgDelim = $$RegularArgDelim_ref;
    $catcon_SecretArgDelim = $$SecretArgDelim_ref;

    # remember whether ERRORLOGGING should be set 
    $catcon_ErrLogging = $ErrLogging;

    # saves references to arrays of statements which should be executed 
    # before using sending a script to a process for the first time and 
    # after a process finishes executing the last script assigned to it
    # in the course of running catconExec()
    $catcon_PerProcInitStmts = $PerProcInitStmts;
    $catcon_PerProcEndStmts = $PerProcEndStmts;

    # construct connect strings and store them in variables which 
    # will persist across calls
    $catcon_UserConnectString = get_connect_string($User);

    if (!$catcon_UserConnectString) {
      print STDERR "catconInit: Empty user connect string returned by get_connect_string\n";
      return 1;
    }

    if ($catcon_DebugOn) {
      print STDERR "catconInit: User Connect String = $catcon_UserConnectString\n";
    }

    $catcon_InternalConnectString = get_connect_string($InternalUser);
    if (!$catcon_InternalConnectString) {
      print STDERR "catconInit: Empty internal connect string returned by get_connect_string\n";
      return 1;
    }

    if ($catcon_DebugOn) {
      print STDERR "catconInit: Internal Connect String = $catcon_InternalConnectString\n";
    }

    if (!valid_src_dir($catcon_SrcDir = $SrcDir)) {
      print STDERR "catconInit: Unexpected error returned by valid_src_dir\n";
      return 1;
    }

    if ($catcon_DebugOn) {
      if ($catcon_SrcDir) {
        print STDERR "catconInit: source file directory = $catcon_SrcDir\n";
      } else {
        print STDERR "catconInit: no source file directory was specified\n";
      }
    }

    # if directory for log file(s) has been specified, verify that it exists 
    # and is writable
    if (!valid_log_dir($catcon_LogDir = $LogDir)) {
      print STDERR "catconInit: Unexpected error returned by valid_log_dir\n";
      return 1;
    }

    # determine base for log file names
    # the same base will be used for spool file names if $SpoolOn is true
    $catcon_LogFilePathBase = 
      get_log_file_base_path($catcon_LogDir, $LogBase, $catcon_DebugOn);

    if ($catcon_DebugOn) {
      print STDERR "catconInit: base for log and spool file names = $catcon_LogFilePathBase\n";
    }

    # (13704981) verify that the database is open
    $instanceStatus = 
      get_instance_status($catcon_InternalConnectString, $catcon_DoneCmd, 
                          $catcon_LogFilePathBase, $catcon_DebugOn);
    if (!(defined $instanceStatus) || $instanceStatus !~ /^OPEN/) {
      print STDERR "catconInit: database is not open\n";
      return 1;
    }

    $catcon_NumProcesses = 
      get_num_procs($NumProcesses, $ExtParallelDegree,
                    $catcon_InternalConnectString, $catcon_DoneCmd, 
                    $catcon_LogFilePathBase, $catcon_DebugOn);

    if ($catcon_NumProcesses < 1) {
      print STDERR "catconInit: invalid number of processes ($catcon_NumProcesses) returned by get_num_procs\n";
      return 1;
    }

    # determine if a DB is a CDB.  We used to rely on CONTAINER$ not returning 
    # any rows, but in a non-CDB from a shiphome, CONTAINER$ will have a row 
    # representing CDB$ROOT (because we ship a single Seed which is is CDB 
    # and unlink PDB$SEED if a customer wants a non-CDB; you could argue that 
    # we could have purged CDB$ROOT from CONTAINER$ the same way we purged 
    # SEED$PDB, but things are the way they are, and it is more robust to 
    # query V$DATABASE.CDB)
    $IsCDB = 
      get_CDB_indicator($catcon_InternalConnectString, $catcon_DoneCmd, 
                        $catcon_LogFilePathBase, $catcon_DebugOn);

    if ($IsCDB eq 'NO') {
      if ($catcon_DebugOn) {
        print STDERR "catconInit: database is non-Consolidated\n";
      }
    } else {
      # obtain names of Containers in the CDB database against which
      # sqlplus script(s) and SQL statement(s) will be run
      @catcon_Containers = 
        get_pdb_names($catcon_InternalConnectString, 
                      $catcon_DoneCmd, $catcon_LogFilePathBase, 
                      $catcon_DebugOn);

      # save name of the Root (it will always be in the 0-th element of 
      # @catcon_Containers because its contents are sorted by 
      # container$.con_id#)
      $catcon_Root = $catcon_Containers[0];

      if ($catcon_DebugOn) {
        print STDERR "catconInit: database is Consolidated\n";
        print STDERR "catconInit: Container names are {\n";
        foreach (@catcon_Containers) {
          print STDERR "\t\t$_\n";
        }
        print STDERR "catconInit: }\n";
      }
    }

    if ($ConNamesIncl || $ConNamesExcl) {
      # if a list of names of Containers in which to run, or not to run, 
      # scripts has been specified, 
      # - make sure both of them have not been specified (since they are 
      #   mutually exclusive) and
      # - verify that named Containers actually exist
      
      if ($ConNamesIncl && $ConNamesExcl) {
        print STDERR "catconInit: both a list of Containers in which to run\n";
        print STDERR "            scripts and a list of Containers in which\n";
        print STDERR "            NOT to run scripts were specified, which\n";
        print STDERR "            is disallowed\n";
        return 1;
      }

      if ($ConNamesExcl) {
        if (!(@catcon_Containers = 
                validate_con_names($ConNamesExcl, 1, @catcon_Containers, 
                                   $catcon_DebugOn))) {
          print STDERR "catconInit: Unexpected error returned by validate_con_namess\n";
          return 1;
        }
      } else {
        if (!(@catcon_Containers = 
                validate_con_names($ConNamesIncl, 0, @catcon_Containers, 
                                   $catcon_DebugOn))) {
          print STDERR "catconInit: Unexpected error returned by validate_con_namess\n";
          return 1;
        } 
      }
    }

    # it is possible that we may start more processes than there are PDBs (or 
    # more than 1 process when operating on a non-CDB.)  It's OK since a user 
    # can tell catconExec() that some scripts may be executed concurrently, 
    # so we may have more than one process working on the same Container (or 
    # non-CDB.)  If a user does not want us to start too many processes, he 
    # can tell us how many processes to start (using -n option to catcon.pl, 
    # for instance)

    # start processes
    start_processes($catcon_NumProcesses, $catcon_LogFilePathBase, 
                    @catcon_FileHandles, @catcon_ProcIds,
                    @catcon_Containers, $catcon_Root,
                    $catcon_InternalConnectString, 
                    $catcon_EchoOn, $catcon_ErrLogging, $catcon_DebugOn, 1);

    # issue statements to cause "done" files to be created for all processes.  
    # next_proc() will look for these files to determine whether a given 
    # process is available to take on the next script or SQL statement
    for (my $CurProc = 0; $CurProc < $catcon_NumProcesses; $CurProc++) {
      
      # file which will indicate that process $CurProc finished its work and 
      # is ready for more
      my $DoneFile = $catcon_LogFilePathBase.$CurProc.".done";

      # create a "done" file unless it already exists
      if (! -e $DoneFile) {
        # "done" file does not exist - cause it to be created
        print {$catcon_FileHandles[$CurProc]} qq/$catcon_DoneCmd $DoneFile\n/;
        $catcon_FileHandles[$CurProc]->flush;
      
        if ($DebugOn) {
          print STDERR qq#catconInit: sent "$catcon_DoneCmd $DoneFile" to process $CurProc (id = $catcon_ProcIds[$CurProc])\n\tto indicate its availability\n#;
        }
      } elsif (! -f $DoneFile) {
        print STDERR qq#catconInit: "done" file name collision: $DoneFile\n#;
        return 1;
      } else {
        if ($DebugOn) {
          print STDERR qq#catconInit: "done" file $DoneFile already exists\n#;
        }
      }
    }

    # remember that initialization has completed successfully
    $catcon_InitDone = 1;

    if ($catcon_DebugOn) {
      print STDERR "catconInit: initialization completed successfully (".TimeStamp().")\n";
    }

    # success
    return 0;
  }

  #
  # catconExec - run specified sqlplus script(s) or SQL statements
  #
  # If connected to a non-Consolidated DB, each script will be executed 
  # using one of the processes connected to the DB.
  #
  # If connected to a Consolidated DB and the caller requested that all 
  # scripts and SQL statements be run against the Root (possibly in addition 
  # to other Containers), each script and statement will be executed in the 
  # Root using one of the processes connected to the Root.
  #
  # If connected to a Consolidated DB and were asked to run scripts and SQL 
  # statements in one or more Containers besides the Root, all scripts and 
  # statements will be run against those PDBs in parallel
  #
  # Parameters:
  #   - reference to an array of sqlplus script name(s) or SQL statement(s); 
  #     script names are expected to be prefixed with @
  #   - an indicator of whether scripts need to be run in order
  #       TRUE => run in order
  #   - an indicator of whether scripts or SQL statements need to be run only 
  #     in the Root if operating on a CDB (temporarily overriding whatever 
  #     was set by catconInit) 
  #       TRUE => if operating on a CDB, run in Root only
  #   - an indicator of whether per process initialization/completion 
  #     statements need to be issued
  #     TRUE => init/comletion statements, if specified, will be issued
  #
  sub catconExec(\@$$$) {

    my ($StuffToRun, $SingleThreaded, $RootOnly, $IssuePerProcStmts) = @_;

    my $CurrentContainerQuery = qq#select '==== Current Container = ' || SYS_CONTEXT('USERENV','CON_NAME') || ' ====' AS now_connected_to from dual\n/\n#;

    # this array will be used to keep track of processes to which statements 
    # contained in @$PerProcInitStmts need to be sent because they [processes]
    # have not been used in the course of this invocation of catconExec()
    my @NeedInitStmts = ();

    if ($catcon_DebugOn) {
      print STDERR "catconExec\n\tScript names/SQL statements:\n";
      foreach (@$StuffToRun) {
        print STDERR "\t\t$_\n";
      }

      print STDERR "\tSingleThreaded    = $SingleThreaded\n";
      print STDERR "\tRootOnly          = $RootOnly\n";
      print STDERR "\tIssuePerProcStmts = $IssuePerProcStmts\n";
      print STDERR "\t(".TimeStamp().")\n";
    }

    # there must be at least one script or statement to run
    if (!@$StuffToRun || $#$StuffToRun == -1) {
      print STDERR  "catconExec: At least one sqlplus script name or SQL statement must be supplied\n";
      return 1;
    }

    # catconInit had better been invoked
    if (!$catcon_InitDone) {
      print STDERR "catconExec: catconInit has not been run";
      return 1;
    }

    # script invocations (together with parameters) and/or SQL statements 
    # which will be executed
    my @ScriptPaths;

    # same as @ScriptPaths but secret parameters will be replaced with prompts
    # so as to avoid storing sensitive data in log files (begin_running and 
    # end_running queries) and error logging tables (in identifier column)
    my @ScriptPathsToDisplay;

    # if $catcon_SpoolOn is set, we will spool output produced by running all 
    # scripts into spool files whose names will be stored in this array
    my @SpoolFileNames;

    my $NextItem;

    # validate script paths and add them, along with parameters and 
    # SQL statements, if any, to @ScriptPaths

    if ($catcon_DebugOn) {
      print STDERR "catconExec: validating scripts/statements supplied by the caller\n";
    }

    # if the user supplied regular and/or secret argument delimiters, this 
    # variable will be set to true after we encounter a script name to remind 
    # us to check for possible arguments
    my $LookForArgs = 0;
    
    # indicators of whether a string may be a regular or a secret script 
    # argument
    my $RegularScriptArg;
    my $SecretScriptArg;

    foreach $NextItem (@$StuffToRun) {

      if ($catcon_DebugOn) {
        print STDERR "catconExec: going over StuffToRun: NextItem = $NextItem\n";
      }

      if ($NextItem =~ /^@/) {
        # leading @ implies that $NextItem contains a script name
        
        # name of SQL*Plus script
        my $FileName;

        if ($catcon_DebugOn) {
          print STDERR "catconExec: next script name = $NextItem\n";
        }

        # strip off the leading @ before prepending source directory name to 
        # script name
        ($FileName = $NextItem) =~ s/^@//;

        # validate path of the sqlplus script and add it to @ScriptPaths
        my $Path = 
          validate_script_path($FileName, $catcon_SrcDir, $catcon_DebugOn);

        if (!$Path) {
          print STDERR "catconExec: empty Path returned by validate_script_path for SrcDir = $catcon_SrcDir, FileName = $FileName\n";
          return 1;
        }

        push @ScriptPaths, "@".$Path;

        # before pushing a new element onto @ScriptPathsToDisplay, replace 
        # single and double quotes in the previous element (if any) with # 
        # to avoid errors in begin/end_running queries and in 
        # SET ERRORLOGGING ON IDENTIFIER ... statements
        if ($#ScriptPathsToDisplay >= 0) {
          $ScriptPathsToDisplay[$#ScriptPathsToDisplay] =~ s/['"]/#/g;
        }

        # assuming it's OK to show script paths
        push @ScriptPathsToDisplay, "@".$Path; 

        if ($catcon_DebugOn) {
          print STDERR "catconExec: full path = $Path\n";
        }
        
        # if caller requested that output of running scripts be spooled, 
        # construct prefix of a spool file name and add it to @SpoolFileNames 
        if ($catcon_SpoolOn) {
          # spool files will get stored in the same directory as log files, 
          # and their names will start with "log file base" followed by '_'
          my $SpoolFileNamePrefix = $catcon_LogFilePathBase .'_';

          if ($catcon_DebugOn) {
            print STDERR "catconExec: constructing spool file name prefix: log file path base +' _' = $SpoolFileNamePrefix\n";
          }
          
          # script file name specified by the caller may contain directories;
          # since we want to store spool files in the same directory as log
          # files, we will replace slashes with _
          my $Temp = $FileName;

          $Temp =~ s/[\/\\]/_/g;

          if ($catcon_DebugOn) {
            print STDERR "catconExec: constructing spool file name prefix: script name without slashes = $Temp\n";
          }

          # we also want to get rid of script file name extension, so we look 
          # for the last occurrence of '.' and strip off '.' along with 
          # whatever follows it
          my $DotPos = rindex($Temp, '.');

          if ($DotPos > 0) {
            # script name should not start with . (if it does, I will let it 
            # be, I guess
            $Temp = substr($Temp, 0, $DotPos);

            if ($catcon_DebugOn) {
              print STDERR "catconExec: constructing spool file name prefix: after stripping off script file name extension, script name = $Temp\n";
            }
          } else {
            if ($catcon_DebugOn) {
              print STDERR "catconExec: constructing spool file name prefix: script file name contained no extension\n";
            }
          }

          push @SpoolFileNames, $SpoolFileNamePrefix.$Temp.'_';

          if ($catcon_DebugOn) {
            print STDERR "catconExec: constructing spool file name prefix: added " . $SpoolFileNames[$#SpoolFileNames] ." to the list\n";
          }
        }

        if ($catcon_RegularArgDelim || $catcon_SecretArgDelim) {
          # remember that script name may be followed by argument(s)
          $LookForArgs = 1;

          if ($catcon_DebugOn) {
            print STDERR "catconExec: prepare to handle arguments\n";
          }
        }
      } elsif (   ($RegularScriptArg = 
                     (   $catcon_RegularArgDelim 
                      && $NextItem =~ /^$catcon_RegularArgDelim/))
               || ($SecretScriptArg = 
                     (   $catcon_SecretArgDelim 
                      && $NextItem =~ /^$catcon_SecretArgDelim/))) {
        # looks like an argument to a script; make sure we are actually 
        # looking for script arguments
        if (!$LookForArgs) {
          print STDERR "catconExec: unexpected script argument ($NextItem) encountered\n";
          return 1;
        }

        if ($catcon_DebugOn) {
          print STDERR "catconExec: processing script argument string ($NextItem)\n";
        }

        # because of short-circuiting of logical operators, if 
        # $RegularScriptArg got set to TRUE, we will never even try to 
        # determine if this string could also represent a secret argument, so 
        # if code in catconInit has determined that a regular argument 
        # delimiter is a prefix of secret argument delimiter, we need to 
        # determine whether this string may also represent a secret argument
        # (and if a regular argument delimiter is NOT a prefix of secret 
        # argument delimiter, reset $SecretScriptArg in case the preceding 
        # item represented a secret argument)
        if ($RegularScriptArg) {
          $SecretScriptArg = 
               ($catcon_ArgToPick == $catcon_PickSecretArg) 
            && ($NextItem =~ /^$catcon_SecretArgDelim/);
        }

        # If this argument could be either (i.e. one of the 
        # argument-introducing strings is a prefix of the other), use 
        # $catcon_ArgToPick to decide how to treat this argument
        # argument stripped off the string marking it as such
        if ($RegularScriptArg && $SecretScriptArg) {

          if ($catcon_ArgToPick == $catcon_PickRegularArg) {
            $SecretScriptArg = undef;  # treat this arg as a regular arg

            if ($catcon_DebugOn) {
              print STDERR "catconExec: (catcon_ArgToPick = $catcon_ArgToPick, catcon_PickRegularArg = $catcon_PickRegularArg) argument string ($NextItem) could be represent either \nregular or secret argument - treat as regular\n";
            }
          } else {
            $RegularScriptArg = undef; # treat this arg as a secret arg

            if ($catcon_DebugOn) {
              print STDERR "catconExec: argument string ($NextItem) could be represent either \nregular or secret argument - treat as secret\n";
            }
          }
        }

        # - if it is a secret argument, prompt the user to enter its value 
        #   and append the value entered by the user to the most recently 
        #   added element of @ScriptPaths
        # - if it is a "regular" argument append this argument to the most 
        #   recently added element of @ScriptPaths
        # In either case, we will strip off the string introducing the 
        # argument and quote the string before adding it to 
        # $ScriptPaths[$#@ScriptPaths] in case it contains embedded blanks.
        # 
        my $Arg;
    
        if ($SecretScriptArg) {
          ($Arg = $NextItem) =~ s/^$catcon_SecretArgDelim//;

          if ($catcon_DebugOn) {
            print STDERR "catconExec: prepare to obtain value for secret argument after issuing ($Arg)\n";
          }

          # get the user to enter value for this argument
          print STDERR "$Arg: ";
          ReadMode 'noecho';
          my $ArgVal = ReadLine 0;
          chomp $ArgVal;
          ReadMode 'normal';
          print "\n";

          if ($catcon_DebugOn) {
            print STDERR "catconExec: user entered ($ArgVal) in response to ($Arg)\n";
          }

          # quote value entered by the user  and append it to 
          # $ScriptPaths[$#ScriptPaths]
          $ScriptPaths[$#ScriptPaths] .= " '" .$ArgVal."'";

          # instead of showing secret parameter value supplied by the user, 
          # we will show the prompt in response to which the value was 
          # supplied
          $ScriptPathsToDisplay[$#ScriptPathsToDisplay] .= " '" .$Arg."'";
        } else {
          ($Arg = $NextItem) =~ s/^$catcon_RegularArgDelim//;

          # quote regular argument and append it to 
          # $ScriptPaths[$#ScriptPaths]
          $ScriptPaths[$#ScriptPaths] .= " '" .$Arg."'";

          $ScriptPathsToDisplay[$#ScriptPathsToDisplay] .= " '" .$Arg."'";

          if ($catcon_DebugOn) {
            print STDERR "catconExec: added regular argument to script invocation string\n";
          }
        }

        if ($catcon_DebugOn) {
          print STDERR "catconExec: script invocation string constructed so far:\n\t$ScriptPaths[$#ScriptPaths]\n";
        }
      } else {
        # $NextItem must contain a SQL statement which we will copy into 
        # @ScriptPaths (a bit of misnomer in this case, sorry)

        if ($catcon_DebugOn) {
          print STDERR "catconExec: next SQL statement = $NextItem\n";
        }

        push @ScriptPaths, $NextItem;

        # before pushing a new element onto @ScriptPathsToDisplay, replace 
        # single and double quotes in the previous element (if any) with # 
        # to avoid errors in begin/end_running queries and in 
        # SET ERRORLOGGING ON IDENTIFIER ... statements
        if ($#ScriptPathsToDisplay >= 0) {
          $ScriptPathsToDisplay[$#ScriptPathsToDisplay] =~ s/['"]/#/g;
        }

        # assuming it's ok to display SQL statements; my excuse: SET 
        # ERRORLOGGING already does it for statements which result in errors
        push @ScriptPathsToDisplay, $NextItem;

        # we expect no arguments following a SQL statement
        $LookForArgs = 0;

        if ($catcon_DebugOn) {
          print STDERR "catconExec: saw SQL statement - do not expect any arguments\n";
        }

        if ($catcon_SpoolOn) {
          push @SpoolFileNames, "";

          if ($catcon_DebugOn) {
            print STDERR "catconExec: saw SQL statement - added empty spool file name prefix to the list\n";
          }
        }
      }
    }

    # replace single and double quotes in the last element of 
    # @ScriptPathsToDisplay with # to avoid errors in begin/end_running 
    # queries and in SET ERRORLOGGING ON IDENTIFIER ... statements
    $ScriptPathsToDisplay[$#ScriptPathsToDisplay] =~ s/['"]/#/g;

    # if running against a non-Consolidated Database
    #   - each script/statement will be run exactly once; 
    # else 
    #   - if the caller instructed us to run only in the Root, temporarily 
    #     overriding the list of Containers specified when calling catconInit, 
    #     or if the Root is among the list of Containers against which we are 
    #     to run
    #     - each script/statement will be run against the Root
    #   - then, if running against one or more PDBs
    #     - each script/statement will be run against all such PDBs in parallel

    # offset into the array of process file handles
    my $CurProc = 0;

    # compute number of processes which will be used to run script/statements 
    # specified by the caller; used to determine when all processes finished 
    # their work
    my $ProcsUsed;

    # if the caller requested that we generate spool files for output 
    # produced by running supplied scripts, an array of spool file name 
    # prefixes has already been constructed.  All that's left is to determine 
    # the suffix:
    #   - if running against a non-Consolidated DB, suffix will be an empty 
    #     string
    #   - otherwise, it needs to be set to the name of the Container against 
    #     which script(s) will be run
    my $SpoolFileNameSuffix;

    # initialize array used to keep track of whether we need to send 
    # initialization statements to a process used for the first time during 
    # this invocation of catcon
    if (   $IssuePerProcStmts 
        && $catcon_PerProcInitStmts && $#$catcon_PerProcInitStmts >= 0) {
      @NeedInitStmts = (1) x $catcon_NumProcesses;
    }
        
    # if ERORLOGGING is enabled, this will be used to store IDENTIFIER
    my $ErrLoggingIdent;

    # skip the portion of the code that deals with running all scripts in the 
    # Root or in a non-CDB if we need to run it in one or more PDBs but not 
    # in the Root
    if (   @catcon_Containers && !$RootOnly 
        && ($catcon_Containers[0] ne $catcon_Root)) {
      if ($catcon_DebugOn) {
        print STDERR "catconExec: skipping single-Container portion of catconExec\n";
      }

      goto skipSingleContainerRun;
    }

    if ($SingleThreaded) {
      # use 1 process to run all script(s) against a non-Consolidated DB or in 
      # the Root or the specified Container of a Consolidated DB
      $ProcsUsed = 1;
    } else {
      # each script may be run against a non-Consolidated DB or in the Root or
      # the specified Container of a Consolidated DB using a separate process
      $ProcsUsed = (@ScriptPaths > $catcon_NumProcesses) 
        ? $catcon_NumProcesses : @ScriptPaths;
    }

    # - if running against a Consolidated DB, we need to connect to the Root 
    #   in every process
    if (@catcon_Containers) {
      for ($CurProc = 0; $CurProc < $ProcsUsed; $CurProc++) {
        print {$catcon_FileHandles[$CurProc]} qq#ALTER SESSION SET CONTAINER = "$catcon_Root"\n/\n#;

        print {$catcon_FileHandles[$CurProc]} $CurrentContainerQuery;
        $catcon_FileHandles[$CurProc]->flush;

        if ($catcon_DebugOn) {
          print STDERR "catconExec: process $CurProc (id = $catcon_ProcIds[$CurProc]) connected to Root Container $catcon_Root\n";
        }
      }
    }

    # run all scripts in 
    # - a non-Consolidated DB or
    # - the Root of a Consolidated DB, if caller specified that they be run 
    #   only in the Root OR if Root is among multiple Containers where scripts 
    #   need to be run

    if (!@catcon_Containers) {
      if ($catcon_ErrLogging) {
        $ErrLoggingIdent = "";
      }

      if ($catcon_DebugOn) {
        print STDERR "catconExec: run all scripts/statements against a non-Consolidated Database\n";
      }

      if ($catcon_SpoolOn) {
        $SpoolFileNameSuffix = "";
        if ($catcon_DebugOn) {
          print STDERR "catconExec: non-CDB - set SpoolFileNameSuffix to empty string\n";
        }
      }
    } else {
      # it must be the case that 
      #         $RootOnly 
      #      || ($catcon_Containers[0] eq $catcon_Root)
      # for otherwise we would have jumped to skipSingleContainerRun above

      if ($catcon_ErrLogging) {
        $ErrLoggingIdent = $catcon_Root."::";
      }

      if ($catcon_DebugOn) {
        print STDERR "catconExec: run all scripts/statements against the Root (Container $catcon_Root) of a Consoldated Database\n";
      }

      if ($catcon_SpoolOn) {
        # set spool file name suffix to the name of the Root
        $SpoolFileNameSuffix = getSpoolFileNameSuffix($catcon_Root);

        if ($catcon_DebugOn) {
          print STDERR "catconExec: CDB - set SpoolFileNameSuffix to $SpoolFileNameSuffix\n";
        }
      }
    }
    
    if ($ErrLoggingIdent) {
      # replace single and double quotes in $ErrLoggingIdent with # to 
      # avoid confusion
      $ErrLoggingIdent =~ s/['"]/#/g   if ($ErrLoggingIdent ne "");

      if ($catcon_DebugOn) {
        print STDERR "catconExec: ErrLoggingIdent prefix = $ErrLoggingIdent\n";
      }
    }

    # $CurProc == -1 (any number < 0 would do, really, since such numbers 
    # could not be used to refer to a process) will indicate that we are yet 
    # to obtain a number of a process to which to send the first script (so we 
    # need to call pickNextProc() even if running single-threaded; once 
    # $CurProc >= 0, if running single-threaded, we will not be trying to 
    # obtain a new process number)
    $CurProc = -1; 

    # index into @ScriptPathsToDisplay array which will be kept in sync with 
    # the current element of @ScriptPaths
    # this index will also be used to access elements of @SpoolFileNames in 
    # sync with elements of the list of scripts and SQL statements to be 
    # executed
    my $ScriptPathDispIdx = -1;

    foreach my $FilePath (@ScriptPaths) {
      if (!$SingleThreaded || $CurProc < 0) {
        # find next available process
        $CurProc = pickNextProc($ProcsUsed, $catcon_NumProcesses, $CurProc + 1,
                                $catcon_LogFilePathBase, \@catcon_ProcIds, 
                                $catcon_Windows, 
                                $catcon_DoneCmd, $catcon_LogFilePathBase, 
                                $catcon_InternalConnectString, 
                                $catcon_DebugOn);

        if ($CurProc < 0) {
          # some unexpected error was encountered
          print STDERR "catconExec: unexpected error in pickNextProc\n";

          return 1;
        }

        # if this is the first time we are using this process and 
        # the caller indicated that one or more statements need to be executed 
        # before using a process for the first time, issue these statements now
        if ($#NeedInitStmts >= 0 && $NeedInitStmts[$CurProc]) {
          firstProcUseStmts($ErrLoggingIdent, $catcon_ErrLogging, 
                            \@catcon_FileHandles, $CurProc, 
                            $catcon_PerProcInitStmts, \@NeedInitStmts, 
                            $catcon_DebugOn);
        }
      }

      # element of @ScriptPathsToDisplay which corresponds to $FilePath
      my $FilePathToDisplay = $ScriptPathsToDisplay[++$ScriptPathDispIdx];

      # run additional init statements (set tab on, set _oracle_script, etc.)
      additionalInitStmts(\@catcon_FileHandles, $CurProc,
                          $catcon_UserConnectString, $catcon_Root,
                          0, 0,
                          $catcon_ProcIds[$CurProc], $catcon_EchoOn,
                          $catcon_DebugOn);

      print {$catcon_FileHandles[$CurProc]} qq#select '==== CATCON EXEC ROOT ====' AS catconsection from dual\n/\n#;

      # bracket script or statement with strings intended to make it easier 
      # to determine origin of output in the log file
      print {$catcon_FileHandles[$CurProc]} qq#select '==== $FilePathToDisplay ====' AS begin_running from dual\n/\n#;

      if ($ErrLoggingIdent) {
        # construct and issue SET ERRORLOGGING ON [TABLE...] statement
        err_logging_tbl_stmt($catcon_ErrLogging, \@catcon_FileHandles, 
                             $CurProc, $catcon_DebugOn);

        # send SET ERRORLOGGING ON IDENTIFIER ... statement

        my $Stmt = "SET ERRORLOGGING ON IDENTIFIER '".substr($ErrLoggingIdent.$FilePathToDisplay, 0, 256)."'\n";

        if ($catcon_DebugOn) {
          print STDERR "catconExec: sending $Stmt to process $CurProc\n";
        }

        print {$catcon_FileHandles[$CurProc]} $Stmt;
      }

      # statement to start/end spooling output of a SQL*Plus script
      my $SpoolStmt;

      # if the caller requested that we generate a spool file and we are about 
      # to execute a script, issue SPOOL ... REPLACE
      if ($catcon_SpoolOn && $SpoolFileNames[$ScriptPathDispIdx] ne "") {
        $SpoolStmt = "SPOOL '" . $SpoolFileNames[$ScriptPathDispIdx] . $SpoolFileNameSuffix . "' REPLACE\n";

        if ($catcon_DebugOn) {
          print STDERR "catconExec: sending $SpoolStmt to process $CurProc\n";
        }

        print {$catcon_FileHandles[$CurProc]} $SpoolStmt;

        # remember that after we execute the script, we need to turn off 
        # spooling
        $SpoolStmt = "SPOOL OFF \n";
      }

      # value which will be used with ALTER SESSION SET APPLICATION ACTION
      my $AppInfoAction;

      my $LastSlash; # position of last / or \ in the script name
      
      if ($FilePath !~ /^@/) {
        $AppInfoAction = $FilePath;
      } elsif (($LastSlash = rindex($FilePath, '/')) >= 0 ||
               ($LastSlash = rindex($FilePath, '\\')) >= 0) {
        $AppInfoAction = substr($FilePath, $LastSlash + 1);
      } else {
        # FilePath contains neither backward nor forward slashes, so use it 
        # as is
        $AppInfoAction = $FilePath;
      }

      # $AppInfoAction may include parameters passed to the script. 
      # These parameters will be surrounded with single quotes, which will 
      # cause a problem since $AppInfoAction is used to construct a 
      # string parameter used with ALTER SESSION SET APPLICATION ACTION.
      # To prevent this from happening, we will replace single quotes found
      #in $AppInfoAction with #
      $AppInfoAction =~ s/[']/#/g;

      if (!@catcon_Containers) {
        $AppInfoAction = "non-CDB::".$AppInfoAction;
      } else {
        $AppInfoAction = $catcon_Root."::".$AppInfoAction;
      }

      # use ALTER SESSION SET APPLICATION MODULE/ACTION to identify process, 
      # Container, if any, and script or statement being run
      print {$catcon_FileHandles[$CurProc]} "ALTER SESSION SET APPLICATION MODULE = 'catcon(pid=$PID)'\n/\n";
      if ($catcon_DebugOn) {
        print STDERR "catconExec: issued ALTER SESSION SET APPLICATION MODULE = 'catcon(pid=$PID)'\n";
      }

      print {$catcon_FileHandles[$CurProc]} "ALTER SESSION SET APPLICATION ACTION = '$AppInfoAction'\n/\n";

      if ($catcon_DebugOn) {
        print STDERR "catconExec: issued ALTER SESSION SET APPLICATION ACTION = '$AppInfoAction'\n";
      }

      # execute next script or SQL statement
      if ($catcon_DebugOn) {
        if ($FilePath =~ /^@/) {
          print STDERR "catconExec: firing script $FilePath against process $CurProc\n";
        } else {
          print STDERR "catconExec: executing statement $FilePath against process $CurProc\n";
        }
      }

      print {$catcon_FileHandles[$CurProc]} "$FilePath\n";

      # if executing a statement, follow the statement with "/"
      if ($FilePath !~ /^@/) {
        print {$catcon_FileHandles[$CurProc]} "/\n";
      }

      # if we started spooling before running the script, turn it off after 
      # it is done
      if ($SpoolStmt) {
        if ($catcon_DebugOn) {
          print STDERR "catconExec: sending $SpoolStmt to process $CurProc\n";
        }

        print {$catcon_FileHandles[$CurProc]} $SpoolStmt;
      }

      print {$catcon_FileHandles[$CurProc]} qq#select '==== $FilePathToDisplay ====' AS end_running from dual\n/\n#;

      # unless we are running single-threaded, we a need a "done" file to be 
      # created after the current script or statement completes so that 
      # next_proc() would recognize that this process is available and 
      # consider it when picking the next process to run a script or SQL 
      # statement
      if (!$SingleThreaded) {
        # file which will indicate that process $CurProc finished its work and 
        # is ready for more
        my $DoneFile = $catcon_LogFilePathBase.$CurProc.".done";

        print {$catcon_FileHandles[$CurProc]} qq/$catcon_DoneCmd $DoneFile\n/;
      
        if ($catcon_DebugOn) {
          print STDERR qq#catconExec: sent "$catcon_DoneCmd $DoneFile" to process $CurProc (id = $catcon_ProcIds[$CurProc])\n\tto indicate its availability after completing $FilePath\n#;
        }
      }

      $catcon_FileHandles[$CurProc]->flush;
    }

    # if we are running single-threaded, we a need a "done" file to be 
    # created after the last script or statement sent to the current Container 
    # completes so that next_proc() would recognize that this process is 
    # available and consider it when picking the next process to run a script 
    # or SQL statement
    if ($SingleThreaded) {
      # file which will indicate that process $CurProc finished its work and 
      # is ready for more
      my $DoneFile = $catcon_LogFilePathBase.$CurProc.".done";

      print {$catcon_FileHandles[$CurProc]} qq/$catcon_DoneCmd $DoneFile\n/;

      $catcon_FileHandles[$CurProc]->flush;
      
      if ($catcon_DebugOn) {
        print STDERR qq#catconExec: sent "$catcon_DoneCmd $DoneFile" to process $CurProc (id = $catcon_ProcIds[$CurProc])\n\tto indicate its availability\n#;
      }
    }
  
    # if 
    #   - there are no additional Containers in which we need to run scripts 
    #     and/or SQL statements and 
    #   - the user told us to issue per-process init and completion statements 
    #     and 
    #   - there are such statements to send, 
    # they need to be passed to wait_for_completion
    my $EndStmts;

    if (   ($RootOnly || !(@catcon_Containers && $#catcon_Containers > 0))
           && $IssuePerProcStmts) {
      $EndStmts = $catcon_PerProcEndStmts;
    } else {
      $EndStmts = undef;
    }

    if (wait_for_completion($ProcsUsed, $catcon_NumProcesses, 
                            $catcon_LogFilePathBase, 
                            @catcon_FileHandles, @catcon_ProcIds,
                            $catcon_DoneCmd, $catcon_Windows, 
                            @$EndStmts, 
                            $catcon_DebugOn) == 1) {
      # unexpected error was encountered, return
      print STDERR "catconExec: unexpected error in wait_for_completions\n";

      return 1;
    }

    # will jump to this label if scripts/stataments need to be run against one
    # or more PDBs, but not against Root
skipSingleContainerRun:

    # if 
    # - running against a Consolidated DB and 
    # - caller has not instructed us to run ONLY in the Root, 
    # - the list of Containers against which we need to run includes at 
    #   least one PDB (which can be establsihed by verifying that either 
    #   @catcon_Containers contains more than one element or the first
    #   element is not CDB$ROOT)
    # run all scripts/statements against all remaining Containers (i.e. PDBs, 
    # since we already took care of the Root, unless the user instructed us 
    # to skip it)    
    if (   @catcon_Containers 
        && !$RootOnly 
        && (   $#catcon_Containers > 0 
            || ($catcon_Containers[0] ne $catcon_Root)))
    {
      # A user may have excluded the Root (and possibly some other PDBs, but 
      # the important part is the Root) from the list of Containers against 
      # which to run scripts, in which case $catcon_Containers[0] would 
      # contain name of a PDB, and we would have skipped the single-container 
      # part of this subroutine.  
      #
      # It is important that we keep it in mind when deciding across how many 
      # Containers we need to run and whether to skip the Container whose 
      # name is stored in $catcon_Containers[0].

      # offset into @catcon_Containers of the first PDB name
      my $firstPDB = ($catcon_Containers[0] eq $catcon_Root) ? 1 : 0;

      # number of PDB names in @catcon_Containers
      my $numPDBs = $#catcon_Containers + 1 - $firstPDB;
      
      if ($catcon_DebugOn) {
        print STDERR "catconExec: run all scripts/statements against remaining $numPDBs PDBs\n";
      }

      # compute number of processes which will be used to run 
      # script/statements specified by the caller in all remaining Containers; 
      # used to determine when all processes finished their work
      if ($SingleThreaded) {
        $ProcsUsed = 
          ($numPDBs > $catcon_NumProcesses) ? $catcon_NumProcesses : $numPDBs;
      } else {
        $ProcsUsed = 
          (@ScriptPaths * $numPDBs > $catcon_NumProcesses) 
            ? $catcon_NumProcesses 
            : @ScriptPaths * $numPDBs;
      }

      # one of the PDBs against which we will run scripts/SQL statements may be
      # PDB$SEED; before we attempt to run any statements against it,
      # we need to close it and then open it READ WRITE or READ WRITE MIGRATE
      # mode.  
      #
      # Since PDB$SEED has con_id of 2, it will be the first PDB in the list.
      #
      # NOTE: as a part of fix for bug 13072385, I moved code to revert 
      #       PDB$SEED mode back to READ ONLY into catconWrapUp(), but I am 
      #       leaving code that reopens it in READ WRITE mode here in 
      #       catconExec() (rather than moving it into catconInit()) because 
      #       the caller may request that scripts/statements be run against 
      #       PDB$SEED even if it was not one of the Containers specified 
      #       when calling catconInit().  Because of that possibility, I have 
      #       to keep this code here, so I see no benefit from also adding it 
      #       to catconInit().
      if ($catcon_Containers[$firstPDB] eq q/PDB$SEED/) {
        if ($catcon_DebugOn) {
          print STDERR "catconExec: check if PDB\$SEED needs to be reopened in READ WRITE mode\n";
        }
            
        my $SeedMode;

        $SeedMode = seed_pdb_mode_state($catcon_InternalConnectString, 
                                        $catcon_DoneCmd, 
                                        $catcon_LogFilePathBase, 
                                        $catcon_DebugOn);
        if ($SeedMode != 1 && $SeedMode != 3) {
          reset_seed_pdb_mode($catcon_InternalConnectString, 
                              "open read write", 
                              $catcon_DoneCmd, $catcon_LogFilePathBase, 
                              $catcon_DebugOn);
              
          if ($catcon_DebugOn) {
            print STDERR "catconExec: reopened PDB\$SEED in READ WRITE mode\n";
          }
              
          $catcon_RevertSeedPdbMode = 1;
        }
      }

      # offset into the array of remaining Container names
      my $CurCon;

      # set it to -1 so the first time next_proc is invoked, it will start by 
      # examining status of process 0
      $CurProc = -1;

      # NOTE: $firstPDB contains offset of the first PDB in the list, but 
      #       $#catcon_Containers still represents the offset of the last PDB
      for ($CurCon = $firstPDB; $CurCon <= $#catcon_Containers; $CurCon++) {

        # if we were told to run single-threaded, switch into the right 
        # Container ; all scripts and SQL statement to be executed in this
        # Container will be sent to the same process
        if ($SingleThreaded) {
          # as we are starting working on a new Container, find next 
          # available process
          $CurProc = pickNextProc($ProcsUsed, $catcon_NumProcesses, 
                                  $CurProc + 1,
                                  $catcon_LogFilePathBase, \@catcon_ProcIds, 
                                  $catcon_Windows, 
                                  $catcon_DoneCmd, $catcon_LogFilePathBase, 
                                  $catcon_InternalConnectString, 
                                  $catcon_DebugOn);
          
          if ($CurProc < 0) {
            # some unexpected error was encountered
            print STDERR "catconExec: unexpected error in pickNextProc\n";

            return 1;
          }

          additionalInitStmts(\@catcon_FileHandles, $CurProc, 
                              $catcon_UserConnectString, $catcon_Root, 
                              $catcon_Containers[$CurCon], 
                              $CurrentContainerQuery, $catcon_ProcIds[$CurProc],
                              $catcon_EchoOn, $catcon_DebugOn);
        }

        # determine prefix of ERRORLOGGING identifier, if needed
        if ($catcon_ErrLogging) {
          $ErrLoggingIdent = "{$catcon_Containers[$CurCon]}::";

          # replace single and double quotes in $ErrLoggingIdent with # to 
          # avoid confusion
          $ErrLoggingIdent =~ s/['"]/#/g;

          if ($catcon_DebugOn) {
            print STDERR "catconExec: ErrLoggingIdent prefix = $ErrLoggingIdent\n";
          }
        }

        my $ScriptPathDispIdx = -1;

        foreach my $FilePath (@ScriptPaths) {
          # switch into the right Container if we were told to run 
          # multi-threaded; each script or SQL statement may be executed 
          # using a different process
          if (!$SingleThreaded) {
            # as we are starting working on a new script or SQL statement, 
            # find next available process
            $CurProc = pickNextProc($ProcsUsed, $catcon_NumProcesses, 
                                    $CurProc + 1,
                                    $catcon_LogFilePathBase, \@catcon_ProcIds, 
                                    $catcon_Windows, 
                                    $catcon_DoneCmd, $catcon_LogFilePathBase, 
                                    $catcon_InternalConnectString, 
                                    $catcon_DebugOn);

            if ($CurProc < 0) {
              # some unexpected error was encountered
              print STDERR "catconExec: unexpected error in pickNextProc\n";

              return 1;
            }

            additionalInitStmts(\@catcon_FileHandles, $CurProc, 
                                $catcon_UserConnectString, $catcon_Root, 
                                $catcon_Containers[$CurCon], 
                                $CurrentContainerQuery, 
                                $catcon_ProcIds[$CurProc],
                                $catcon_EchoOn, $catcon_DebugOn);
          }

          # if this is the first time we are using this process and 
          # the caller indicated that one or more statements need to be 
          # executed before using a process for the first time, issue these 
          # statements now
          if ($#NeedInitStmts >= 0 && $NeedInitStmts[$CurProc]) {
            firstProcUseStmts($ErrLoggingIdent, $catcon_ErrLogging, 
                              \@catcon_FileHandles, $CurProc, 
                              $catcon_PerProcInitStmts, \@NeedInitStmts, 
                              $catcon_DebugOn);
          }

          print {$catcon_FileHandles[$CurProc]} qq#select '==== CATCON EXEC IN CONTAINERS ====' AS catconsection from dual\n/\n#;

          # element of @ScriptPathsToDisplay which corresponds to $FilePath
          my $FilePathToDisplay = $ScriptPathsToDisplay[++$ScriptPathDispIdx];

          # bracket script or statement with strings intended to make it 
          # easier to determine origin of output in the log file
          print {$catcon_FileHandles[$CurProc]} qq#select '==== $FilePathToDisplay ====' AS begin_running from dual\n/\n#;

          if ($ErrLoggingIdent) {
            # construct and issue SET ERRORLOGGING ON [TABLE...] statement
            err_logging_tbl_stmt($catcon_ErrLogging, \@catcon_FileHandles, 
                                 $CurProc, $catcon_DebugOn);

            # send SET ERRORLOGGING ON IDENTIFIER ... statement

            my $Stmt = "SET ERRORLOGGING ON IDENTIFIER '".substr($ErrLoggingIdent.$FilePathToDisplay, 0, 256)."'\n";

            if ($catcon_DebugOn) {
              print STDERR "catconExec: sending $Stmt to process $CurProc\n";
            }

            print {$catcon_FileHandles[$CurProc]} $Stmt;
          }

          # statement to start/end spooling output of a SQL*Plus script
          my $SpoolStmt;

          # if the caller requested that we generate a spool file and we are 
          # about to execute a script, issue SPOOL ... REPLACE
          if ($catcon_SpoolOn && $SpoolFileNames[$ScriptPathDispIdx] ne "") {
            my $SpoolFileNameSuffix = 
              getSpoolFileNameSuffix($catcon_Containers[$CurCon]);

            $SpoolStmt = "SPOOL '" . $SpoolFileNames[$ScriptPathDispIdx] . $SpoolFileNameSuffix . "' REPLACE\n";
            
            if ($catcon_DebugOn) {
              print STDERR "catconExec: sending $SpoolStmt to process $CurProc\n";
            }

            print {$catcon_FileHandles[$CurProc]} $SpoolStmt;

            # remember that after we execute the script, we need to turn off 
            # spooling
            $SpoolStmt = "SPOOL OFF \n";
          }

          # if executing a query, will be set to the text of the query; 
          # otherwise, will be set to the name of the script with parameters, 
          # if any
          my $AppInfoAction;

          my $LastSlash; # position of last / or \ in the script name
      
          if ($FilePath !~ /^@/) {
            $AppInfoAction = $FilePath;
          } elsif (($LastSlash = rindex($FilePath, '/')) >= 0 ||
                   ($LastSlash = rindex($FilePath, '\\')) >= 0) {
            $AppInfoAction = substr($FilePath, $LastSlash + 1);
          } else {
            # FilePath contains neither backward nor forward slashes, so use 
            # it as is
            $AppInfoAction = $FilePath;
          }

          # $AppInfoAction may include parameters passed to the script. 
          # These parameters will be surrounded with single quotes, which will 
          # cause a problem since $AppInfoAction is used to construct a 
          # string parameter being used with 
          # ALTER SESSION SET APPLICATION ACTION.
          # To prevent this from happening, we will replace single quotes found
          #in $AppInfoAction with #
          $AppInfoAction =~ s/[']/#/g;

          # use ALTER SESSION SET APPLICATION MODULE/ACTION to identify 
          # process, Container and script or statement being run
          print {$catcon_FileHandles[$CurProc]} "ALTER SESSION SET APPLICATION MODULE = 'catcon(pid=$PID)'\n/\n";
          if ($catcon_DebugOn) {
            print STDERR "catconExec: issued ALTER SESSION SET APPLICATION MODULE = 'catcon(pid=$PID)'\n";
          }

          print {$catcon_FileHandles[$CurProc]} "ALTER SESSION SET APPLICATION ACTION = '$catcon_Containers[$CurCon]::$AppInfoAction'\n/\n";

          if ($catcon_DebugOn) {
            print STDERR "catconExec: issued ALTER SESSION SET APPLICATION ACTION = '$catcon_Containers[$CurCon]::$AppInfoAction'\n";
          }

          # execute next script or SQL statement

          if ($catcon_DebugOn) {
            if ($FilePath =~ /^@/) {
              print STDERR "catconExec: firing script $FilePath against process $CurProc\n";
            } else {
              print STDERR "catconExec: executing statement $FilePath against process $CurProc\n";
            }
          }

          print {$catcon_FileHandles[$CurProc]} "$FilePath\n";

          # if executing a statement, follow the statement with "/"
          if ($FilePath !~ /^@/) {
            print {$catcon_FileHandles[$CurProc]} "/\n";
          }

          # if we started spooling before running the script, turn it off after
          # it is done
          if ($SpoolStmt) {
            if ($catcon_DebugOn) {
              print STDERR "catconExec: sending $SpoolStmt to process $CurProc\n";
            }
            
            print {$catcon_FileHandles[$CurProc]} $SpoolStmt;
          }

          print {$catcon_FileHandles[$CurProc]} qq#select '==== $FilePathToDisplay ====' AS end_running from dual\n/\n#;

          # unless we are running single-threaded, we a need a "done" file to 
          # be created after the current script or statement completes so that 
          # next_proc() would recognize that this process is available and 
          # consider it when picking the next process to run a script or SQL 
          # statement
          if (!$SingleThreaded) {
            # file which will indicate that process $CurProc finished its 
            # work and is ready for more
            my $DoneFile = $catcon_LogFilePathBase.$CurProc.".done";

            print {$catcon_FileHandles[$CurProc]} qq/$catcon_DoneCmd $DoneFile\n/;
      
            if ($catcon_DebugOn) {
              print STDERR qq#catconExec: sent "$catcon_DoneCmd $DoneFile" to process $CurProc (id = $catcon_ProcIds[$CurProc])\n\tto indicate its availability after completing $FilePath\n#;
            }
          }
            
          $catcon_FileHandles[$CurProc]->flush;
        }

        # if we are running single-threaded, we a need a "done" file to be 
        # created after the last script or statement sent to the current 
        # Container completes so that next_proc() would recognize that this 
        # process is available and consider it when picking the next process 
        # to run a script or SQL statement
        if ($SingleThreaded) {
          # file which will indicate that process $CurProc finished its work 
          # and is ready for more
          my $DoneFile = $catcon_LogFilePathBase.$CurProc.".done";

          print {$catcon_FileHandles[$CurProc]} qq/$catcon_DoneCmd $DoneFile\n/;

          $catcon_FileHandles[$CurProc]->flush;
      
          if ($catcon_DebugOn) {
            print STDERR qq#catconExec: sent "$catcon_DoneCmd $DoneFile" to process $CurProc (id = $catcon_ProcIds[$CurProc])\n\tto indicate its availability\n#;
          }
        }
      }

      # if 
      #   - the user told us to issue per-process init and completion 
      #     statements and 
      #   - there any such statements to send, 
      # they need to be passed to wait_for_completion
      my $EndStmts = $IssuePerProcStmts ? $catcon_PerProcEndStmts : undef;

      if (wait_for_completion($ProcsUsed, $catcon_NumProcesses, 
                              $catcon_LogFilePathBase, 
                              @catcon_FileHandles, @catcon_ProcIds,
                              $catcon_DoneCmd, $catcon_Windows, 
                              @$EndStmts,
                              $catcon_DebugOn) == 1) {
        # unexpected error was encountered, return
        print STDERR "catconExec: unexpected error in wait_for_completions\n";

        return 1;
      }
    }

    # success
    return 0;
  }

  #
  # catconRunSqlInEveryProcess - run specified SQL statement(s) in every 
  #                              process
  #
  # Parameters:
  #   - reference to a list of SQL statement(s) which should be run in every 
  #     process
  # 
  # Returns
  #   1 if some unexpected error was encountered; 0 otherwise
  #
  sub catconRunSqlInEveryProcess (\@) {
    my ($Stmts) = @_;

    my $ps;

    # there must be at least one statement to execute
    if (!$Stmts || !@$Stmts || $#$Stmts == -1) {
      print STDERR  "catconRunSqlInEveryProcess: At least one SQL statement must be supplied";
      return 1;
    }

    if ($catcon_DebugOn) {
      print STDERR "running catconRunSqlInEveryProcess\nSQL statements:\n";
      foreach (@$Stmts) {
        print STDERR "\t$_\n";
      }
    }

    # catconInit had better been invoked
    if (!$catcon_InitDone) {
      print STDERR "catconRunSqlInEveryProcess: catconInit has not been run";
      return 1;
    }

    # send each statement to each process
    foreach my $Stmt (@$Stmts) {
      for ($ps=0; $ps < $catcon_NumProcesses; $ps++) {
        print {$catcon_FileHandles[$ps]} "$Stmt\n";
      }    
    }

    return 0;
  }

  #
  # catconShutdown - shutdown the database
  #
  # Parameters:
  #   - shutdown flavor (e.g. ABORT or IMMEDIATE)
  #
  sub catconShutdown (;$) {

    my ($ShutdownFlavor) = @_;
      
    # catconInit had better been invoked
    if (!$catcon_InitDone) {
      print STDERR "catconShutdown: catconInit has not been run";
      return 1;
    }

    # free up all resources
    catconWrapUp();

    # if someone wants to invoke any catcon subroutine, they will need 
    # to invoke catconInit first

    if ($catcon_DebugOn) {
      print STDERR "catconShutdown: will shutdown database using SHUTDOWN $ShutdownFlavor\n";
    }

    # shutdown_db needs to be called after catconWrapUp() to make sure that 
    # all processes have exited.
    shutdown_db($catcon_InternalConnectString, $ShutdownFlavor,
                $catcon_DoneCmd, $catcon_LogFilePathBase, $catcon_DebugOn);

    return 0;
  }

  #
  # catconBounceProcesses - bounce all processes
  #
  sub catconBounceProcesses () {

    # catconInit had better been invoked
    if (!$catcon_InitDone) {
      print STDERR "catconBounceProcesses: catconInit has not been run";
      return 1;
    }

    if ($catcon_DebugOn) {
      print STDERR "catconBounceProcesses: will bounce $catcon_NumProcesses processes\n";
    }

    # end processes
    end_processes(0, $catcon_NumProcesses - 1, @catcon_FileHandles, 
                  @catcon_ProcIds, $catcon_DebugOn);

    # 
    # set up signal in case SQL process crashes
    # before it completes its work
    #
    $SIG{CHLD} = \&catcon_HandleSigchld;

    # start processes

    start_processes($catcon_NumProcesses, $catcon_LogFilePathBase, 
                    @catcon_FileHandles, @catcon_ProcIds, 
                    @catcon_Containers, $catcon_Root,
                    $catcon_InternalConnectString, 
                    $catcon_EchoOn, $catcon_ErrLogging, $catcon_DebugOn, 0);

    if ($catcon_DebugOn) {
      print STDERR "catconBounceProcesses: finished bouncing $catcon_NumProcesses processes\n";
    }
  }

  #
  # catconWrapUp - free any resources which may have been allocated by 
  #                various catcon.pl subroutines
  #
  sub catconWrapUp () {

    # catconInit had better been invoked
    if (!$catcon_InitDone) {
      print STDERR "catconWrapUp: catconInit has not been run";
      return 1;
    }

    if ($catcon_DebugOn) {
      print STDERR "catconWrapUp: about to free up all resources\n";
    }

    # end processes
    end_processes(0, $catcon_NumProcesses - 1, @catcon_FileHandles, 
                  @catcon_ProcIds, $catcon_DebugOn);

    # clean up completion files
    clean_up_compl_files($catcon_LogFilePathBase, $catcon_NumProcesses, 
                         $catcon_DebugOn);

    # if we reopened PDB$SEED in READ WRITE mode, close it and then open it 
    # in READ ONLY mode
    #
    # 14248297: reset_seed_pdb_mode() closes PDB$SEED on all instances, so 
    # we need to open in READ ONLY mode on all instances too.
    #
    # NOTE: we need to wait for all processes to be killed before calling 
    #       reset_seed_pdb_mode() because it closes PDB$SEED before reopening 
    #       it in desired mode, and if any process is connected to PDB$SEED 
    #       while we are closing it, ALTER PDB CLOSE will hang.
    if ($catcon_RevertSeedPdbMode) {
      reset_seed_pdb_mode($catcon_InternalConnectString,
                          "open read only instances=all", 
                          $catcon_DoneCmd, $catcon_LogFilePathBase, 
                          $catcon_DebugOn);
    
      if ($catcon_DebugOn) {
        print STDERR "catconWrapUp: reopened PDB\$SEED in READ ONLY mode\n";
      }

      $catcon_RevertSeedPdbMode = 0;
    }

    # no catcon* subroutines should be invoked without first calling catconInit
    $catcon_InitDone = 0;

    if ($catcon_DebugOn) {
      print STDERR "catconWrapUp: done\n";
    }
  }

  ######################################################################
  #  If one of the child process terminates, it is a fatal error
  ######################################################################

  sub catcon_HandleSigchld () {
    print STDERR "A process terminated prior to completion.\n";
    print STDERR "Review the ${catcon_LogFilePathBase}*.log files to identify the failure.\n";
    $SIG{CHLD} = 'IGNORE';  # now ignore any child processes
    die;  
  }

  sub catcon_HandleSigINT () {
    print STDERR "Signal INT was received.\n";
    # if we reopened PDB$SEED in READ WRITE mode, close it and then open it 
    # in READ ONLY mode
    #
    # 14248297: reset_seed_pdb_mode() closes PDB$SEED on all instances, so 
    # we need to open in READ ONLY mode on all instances too.
    #
    # NOTE: normally, we need to wait for all processes to be killed before 
    #       calling reset_seed_pdb_mode(), but since catcon process itself is 
    #       gettink killed, there is no room for such niceties.
    if ($catcon_RevertSeedPdbMode) {
      reset_seed_pdb_mode($catcon_InternalConnectString, 
                          "open read only instances=all", 
                          $catcon_DoneCmd, $catcon_LogFilePathBase, 
                          $catcon_DebugOn);
    
      if ($catcon_DebugOn) {
        print STDERR "catcon_HandleSigINT: reopened PDB\$SEED in READ ONLY mode\n";
      }

      $catcon_RevertSeedPdbMode = 0;
    }

    $SIG{INT} = 'IGNORE';  # now ignore any INTs
    die;  
  }
}

1;
