#
# $Header: rdbms/admin/catctl.pl /st_rdbms_12.1.0.1/3 2013/03/05 10:22:25 krajaman Exp $
#
# catctl.pl
# 
# Copyright (c) 2005, 2013, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      catctl.pl - CATalog ConTroL PerL program
#
#    DESCRIPTION
#      This perl program processes sqlplus files and organizes
#      them for parallel processing based on annotations within
#      the files.
#
#    NOTES
#      Used by catupgrd shell/bat scripts to run parallel upgrades
#      Connects to database specified by ORACLE_SID environment variable
#
#    MODIFIED   (MM/DD/YY)
#    jerrede     02/11/13 - Fix Password containing special characters bug
#                           16177906
#    jerrede     01/10/13 - Ignore sqlsessstart and sqlsessend in driver files
#    jerrede     12/11/12 - xbranchmerge of jerrede_lrg-7343558
#    jerrede     11/06/12 - Add Display option for patch group
#    bmccarth    10/30/12 - call utlucdir
#    jerrede     10/16/12 - Fix lrg 7284666
#    jerrede     10/11/12 - Fix Security bug 14750812
#    jerrede     10/03/12 - Fix lrg 7291461
#    jerrede     08/28/12 - Mandatory Post upgrade.
#    jerrede     07/19/12 - Remove Passing Password at Command Line
#                           Use /n\/n as the SQL Terminator for all
#                           Sql Statements
#    jerrede     05/24/12 - Add Display of SQL File Executing
#    jerrede     10/18/11 - Parallel Upgrade ntt Changes
#    jerrede     09/12/11 - Fix Bug 12959399
#    jerrede     09/01/11 - Parallel Upgrade Project no 23496
#    rburns      10/23/07 - remove multiple processes; fix password mgmt
#    rburns      10/20/06 - add session script
#    rburns      12/16/05 - perl script for parallel sqlplus processing 
#    rburns      12/16/05 - Creation
# 

# use strict;
use English;
use Getopt::Std;       # to parse command line options
use IO::Handle;        # to flush buffers
use Term::ReadKey;     # to not echo password

if (@ARGV < 1) {
    print STDERR <<USAGE;

  Usage: catctl [-u username] [-n processes] [-d directory] 
                [-t table] [-l directory] [-s script] 
                [-e] [-i] [-c] filename

  Supported Options:

     -u username (prompts for password)
     -n the number of processes to use for parallel operations (default 4)
     -d directory containing the files to be run 
     -t table name for phase logging
     -l directory to use for spool log files
     -s SQL script to initialize sessions
     -e sets echo off while running the scripts
     -p restart phase (skip successful phases on a rerun)
     -i identifier to use when creating spool log files 
     -y display phases only
        
     filename top-level sqlplus script to run

USAGE
    exit 1; # ERROR
}

# Global variables

# Constants
my $single = 1;
my $multi  = 2;
my $true   = 1;
my $false  = 0;
my $Windows = $false;
my $numofphasefiles = 0; # Number of files in phase 
my $shutdownphase = 0;   # Shutdown phase 
my $noprocess = 0;
my $startprocess = 1;
my $endprocess = 2;
my $SqlTerm = "\n/\n";
my $orawait_sql_command = "ora_wait_for_thread.sql";
my $LoadWithoutComp_sql_command = "ora_load_without_comp.sql";
my $LoadWithComp_sql_command = "ora_load_with_comp.sql";
my $TWOSPACE  = "  ";
my $SPACE     = " ";
my $NOSPACE   = "";
my $longsqljobs    = "catnodpt.sql catmetviews.sql catmetgrant1.sql catmetgrant2.sql";
my $SHUTDOWNJOB    = "catshutdown.sql";
my $NOTHINGJOB     = "nothing.sql";
my $POSTUPGRADEJOB = "catuppst.sql";
my $UnixDoneCmd = "\nhost sqlplus -v >";
my $WindowsDoneCmd = "\nhost sqlplus/nolog -v >";
my $HostCmd = $UnixDoneCmd;
my $LoadWithoutComp = "ALTER SESSION SET \"_LOAD_WITHOUT_COMPILE\" = plsql";
my $LoadWithComp = "ALTER SESSION SET \"_LOAD_WITHOUT_COMPILE\" = none";
my $bLoadWithoutComp = $false;
my $bLoadWithComp = $true;
my $bFoundPfile = $false;   # Assume Not Found
my $bCreatePfile = $false;  # Not Created
my $LoadComp = $LoadWithComp;
my $restart_sql_command = "ora_restart.sql";   # Restart
my $bRunSerial = $false;  # Run driver file in serial mode
my $pid_cnt = 0;  # No of pids active
my $EndTag = "\n========== PROCESS ENDED ==========\n"; # End Tag
my $SERIAL   = "Serial  ";
my $PARALLEL = "Parallel";
my $RESTART  = "Restart ";
my $DspPhase = $SERIAL;
my $DSPSECS = "s";
my $LASTRDBMSJOB = "cmpupstr.sql";
my $LASTJOB = "catupend.sql";
my $CDIRSCRIPT = "utlucdir.sql";
my $MSGFCATCTLREADFILE = "Cannot Read File";

my $MSGWPMSG1 = "\n\n*** WARNING: ERRORS FOUND DURING UPGRADE ***\n\n";
my $MSGWPMSG2 = "Due to errors found during the upgrade process, the post\n";
my $MSGWPMSG3 = "upgrade actions in $POSTUPGRADEJOB have not been ";
my $MSGWPMSG4 = "automatically run.\n\n *** THEREFORE THE DATABASE UPGRADE";
my $MSGWPMSG5 = " IS NOT YET COMPLETE ***\n\n 1. Evaluate the errors found";
my $MSGWPMSG6 = " in the upgrade logs ($SpoolLog*.log) and determine ";
my $MSGWPMSG7 = "the proper action.\n 2. Execute the post upgrade script ";
my $MSGWPMSG8 = "as described in Chapter 3 of the Database Upgrade Guide.\n\n";
my $MSGWPMSG9  = "$MSGWPMSG1$MSGWPMSG2$MSGWPMSG3$MSGWPMSG4";
my $MSGWPMSG10 = "$MSGWPMSG5$MSGWPMSG6$MSGWPMSG7$MSGWPMSG8";
my $MSGWMANUALPSTUPGRD = "$MSGWPMSG9$MSGWPMSG10";
my $ERRORTAG = "CATCTL ERROR COUNT=";    # Error Tag
my $PFILETAG = "CATCTL PFILE=";          # PFile Tag
my $READSP21519 = 1;
my $READPFILE   = 2;

# Intialize Args
$opt_u = 0;
$opt_n = 4;
$opt_d = 0;
$opt_t = 0;
$opt_l = 0;
$opt_s = 0;
$opt_f = 0;
$opt_p = 0;
$opt_e = 0;
$opt_a = 0;
$opt_x = 0;
$opt_r = 0;
$opt_o = 0;
$opt_i = 0;
$opt_y = 0;

#  Parse command line arguments obsolete a
getopts("u:n:d:t:l:s:f:p:i:eaxroy");
my $User    = $opt_u;
my $SrcDir  = $opt_d;
my $LogDir  = $opt_l;
my $Table   = $opt_t;
my $Script  = $opt_s;
my $PwdFile = $opt_f;
my $DBUA    = $opt_a; # DBUA 
my $DisplayPhases   = $opt_y; # Display phases only
my $SpoolLogId      = $opt_i; # Spool log identifier (ie Oracle Sid)
my $DoNotRunPostUpg = $opt_x; # Do not run Post upgrade
my $bOpenModeNormal = $opt_o; # Open Mode normal and leave database open
my $PostUpgMessage = 0;       # Post Upgrade Message
my $SessionStart    = "sqlsessstart.sql";
my $SessionEnd      = "sqlsessend.sql";
my $StartPhase = 0;
my $UseDir  = 0;
my $Path;
my $SpoolLog;
my $ph = 0;
my $Connect;
my $Process = 4;
my $pcnt = 0;
my $LastPhase = 0;
my $Lastnumofphasefiles = 0;
my $LastTime = time();
my $stime  =  0;
my $sdifsec = 0;
my $ttotsec = 0;
my $PadChr = $SPACE;
my $pFileName = 0;       # Pfile Name 
my $File = $ARGV[0] or die "No filename specified";

####################################################################
#  Scan input files building phase execution tables
####################################################################

# initialize phases and scripts array
my @phase_type;               # multi or single
my @phase_compile;            # Load with or without compile 
my @phase_files;              # references to file name array
my @sql_process_in_use;       # Sql processes in use
my @session_files;            # Sql Session Files
my @session_phase;            # Sql Session Phase to start runing in
my @pfh;                      # array of file handle references;
my @pid;                      # array of process ids
my $depth = 0;                # degree of nested scripts (-X control)

push (@phase_compile,$bLoadWithComp);  # default first phase to compile on
push (@phase_type,$single);   # default first phase to single threaded
push (@phase_files,[]);       # default a reference to an empty array

set_process_no();     # Setup Process Number
init($Process);       # Initialize

if (!$bRunSerial) 
{
    scan_file ($Path);    # recursively scan through control files
    display_phases();     # Display Phases
}

#
# Display phases only
#
if ($DisplayPhases)
{
    exit (0); # Success
}

######################################################################
# Start parallel sqlplus session, based on number of processes
######################################################################
start_sql_process(0, $Process, $SpoolLog, $Connect, $opt_e, $true);

#
# Run driver input file serial and skip everything else
#
if ($bRunSerial)
{
    my $mPath = $Path;     # Set to Path File
    my $mFile = lc($File); # Lower Case File
    $StartPhase = @phase_type;
    $LastTime = time();

    #
    # Add in additional argument for an upgrade
    #
    if (rindex($mFile, "catupgrd") != -1)
    {
        #
        # If not present then add in argument
        #
        if (rindex($mFile, "parallel=no") == -1)
        {
            $mPath = $Path." PARALLEL=NO";
        }
    }

    print STDERR "Running $Path In Serial Order Please Wait...\n";
    run_file (0, $mPath);
    #
    # Wait for phase to complete
    #
    $pcnt = $Process;
    end_sql_process(0, $pcnt);
    $pcnt = 0;

}

######################################################################
# loop through phases running scripts in available processes
######################################################################
for ($ph = $StartPhase; $ph < @phase_type; $ph++) 
{

   # set number of process and display for this phase
   $pcnt = ($phase_type[$ph] == $single) || $opt_r ? 1 : $Process;
   $DspPhase = ($pcnt > 1) ? $PARALLEL : $SERIAL;
   if ($phase_files[$ph][0] eq $restart_sql_command)
   {
       $DspPhase = $RESTART;
   }

   #
   #
   # Set number of files in phase
   #
   $numofphasefiles = @{$phase_files[$ph]};

   #
   # Do Nothing
   #
   if ($phase_files[$ph][0] eq $NOTHINGJOB)
   {
       next;
   }

   #
   # Post Upgrade job
   #
   if ($phase_files[$ph][0] eq $POSTUPGRADEJOB)
   {

       #
       # Setup for the Post Upgrade
       #
       if ($DoNotRunPostUpg == $false)
       {
           setup_for_post_upgrade();
       }

       #
       # Don't Run the Post Upgrade We are done
       #
       if ($DoNotRunPostUpg == $true)
       {
           print STDERR "$MSGWMANUALPSTUPGRD";
           $ph = @phase_type;
           next;
       }
   }

   #
   # Set flag to indicate this is a shutdown
   # phase.
   #
   if (($SHUTDOWNJOB eq $phase_files[$ph][0]) ||
       ($LASTJOB     eq $phase_files[$ph][0]))
   {
     $shutdownphase = $true;
   }
   else
   {
     $shutdownphase = $false;
   }

   if ($ph != 0)
   {
       #
       # Turn on and off Compile switch as script dictates
       #
       if ($phase_compile[$ph] == $bLoadWithComp)
       {
           $LoadComp = $LoadWithComp;
       }
       else
       {
           $LoadComp = $LoadWithoutComp;
       }

       #
       # Load without Compile is turn on in phase 1
       # We will get errors if we turn on before phase 1.
       #
       send_sql_to_all_process($Process, $LoadComp);

       $stime  = time();
       $sdifsec = $stime - $LastTime;
       $ttotsec = $ttotsec + $sdifsec;

       $PadChr = $SPACE;
       if ($Lastnumofphasefiles < 10)
       {
           $PadChr = $SPACE.$SPACE.$SPACE;
       }
       else
       {
           if ($Lastnumofphasefiles < 100)
           {
               $PadChr = $SPACE.$SPACE;
           }
       }
       print STDERR " $PadChr Time: $sdifsec$DSPSECS\n";
   }
   $LastTime = time();
   $LastPhase = $ph;
   $Lastnumofphasefiles = $numofphasefiles;

   $PadChr = $ph < 10 ? $SPACE : $NOSPACE;
   print STDERR "$DspPhase Phase #:$PadChr$ph Files: $numofphasefiles ";

   # Restart Sql Process
   if ($phase_files[$ph][0] eq $restart_sql_command)
   {
       end_sql_process(0, $Process);
       start_sql_process(0, $Process, $SpoolLog, $Connect, $opt_e, $false);
       next;
   }

   # No Opts for us just send the alter session command
   if (($phase_files[$ph][0] eq $LoadWithoutComp_sql_command) ||
       ($phase_files[$ph][0] eq $LoadWithComp_sql_command))
   {
       next;
   }

   # loop through processes 
   for ($ps=0; $ps < $pcnt; $ps++)
   {
       # create log table in phase 0 if specified
       if ($Table && $ph == 0 && $ps == 0)
       {
           send_sql_to_process($ps,
           "\nSELECT 'Log table is $Table Created at ' || TO_CHAR(SYSTIMESTAMP,'YY-MM-DD HH:MI:SS') AS catctl_timestamp FROM SYS.DUAL");
           send_sql_to_process($ps, "DROP TABLE $Table");
           send_sql_to_process($ps,
           "CREATE TABLE $Table (script VARCHAR2 (1024), phase NUMBER, process NUMBER, start_time TIMESTAMP, end_time TIMESTAMP)");
       }

       send_sql_to_process($ps,
       "SELECT 'PHASE_TIME___START $ph ' || TO_CHAR(SYSTIMESTAMP,'YY-MM-DD HH:MI:SS') AS catctl_timestamp FROM SYS.DUAL");

       if ($Table)
       {
           send_sql_to_process($ps,
           "INSERT INTO $Table (script, phase, process, start_time) VALUES ('$File','$ph','$ps',SYSTIMESTAMP)");
       }

       #
       # run session initialization script for process except for phase 0
       # phase 0 is single process by default
       #
       if ($ph != 0)
       {
           # Run Session Scripts
           for ($x = 0; $x < @session_files; $x++) 
           {
               # Run only in the phase found and beyond
               if ($session_phase[$x] <= $ph)
               { 
                   run_file ($ps,$session_files[$x]);
               }
           }
       }
   }

   # check if the phase has some files to process
   if ($numofphasefiles > 0)
   {
       if ($phase_type[$ph] == $single)
       {
           #
           # If shutting down database exit other processors
           #
           if ($shutdownphase)
           {
               end_sql_process(1, $Process);
           }

           #
           # Last RDBMS Job Check Logs for SP2-1519 Errors
           #
           if ($LASTRDBMSJOB eq $phase_files[$ph][0])
           {
                catctlReadLogFiles($READSP21519);
           }

           #
           # Create the Pfile before last job is run
           #
           if ($LASTJOB eq $phase_files[$ph][0])
           {
               create_pfile();
           }

           #
           # Send Sql files to process 0
           #
           for ($pf = 0; $pf < $numofphasefiles; $pf++)
           {
               # print the filename to process 0
               run_file (0,$phase_files[$ph][$pf]);
           } # end single threaded file loop
       }
       else
       {
           if ($opt_r)
           {
               for ($pf = @{$phase_files[$ph]}-1; $pf >= 0; $pf--)
               {
                   # print the filename to process 0
                   run_file (0,$phase_files[$ph][$pf]);
               } # end single reverse file loop
           }
           else
           {   # use multiple sqlplus processes

               #
               # Flood the processors with jobs
               #
               $ps = 0;
               for ($pf = 0; $pf < $numofphasefiles; $pf++)
               {
                   #
                   # Get Sql Process Available and
                   # submit the filename to the process
                   #
                   $ps = find_sql_process($ps, $pcnt);
                   run_file ($ps,$phase_files[$ph][$pf]);

                   #
                   # On long running jobs don't put any jobs behind
                   # the one that takes a long time to run.
                   # Set the current sql process in use and load
                   # up the remainng sql processes.  We only do this
                   # if we have enough processes to use.
                   #
                   if (index($longsqljobs,$phase_files[$ph][$pf]) != -1)
                   {
                       #
                       # If we have sql processors to do the work the
                       # rest of the work than we can't set
                       # this CPU as inuse, otherwise don't peg
                       # any CPU as in use we need them all to do
                       # the work.
                       #
                       $tmp_pcnt = $ps + 1;
                       if ($tmp_pcnt < $pcnt)
                       {
                           set_sql_process($true, $ps);
                       }
                   }
                   $ps++;
                   if ($ps == $pcnt) { $ps = 0; }
               }
               # Reset sql processes to available
               init_sql_process($pcnt);
           } # End Else Multiprocess
       } # End Else 
   } # End if files to run in phase

   #
   # Wait for phase to complete
   #
   wait_for_phase_completion($pcnt);

}  # end of phase loop

$stime  = time();
$sdifsec = $stime - $LastTime;
$ttotsec = $ttotsec + $sdifsec;
print STDERR "     Time: $sdifsec$DSPSECS\n";
print STDERR "Grand Total Time: $ttotsec$DSPSECS\n";

end_sql_process(0, $pcnt);
clean_up( $Process, $false );
write_text(0, $Process, $SpoolLog, $EndTag);
exit (0); # Success

######################################################################
# Wait for sql files or commands to complete within a phase
######################################################################
sub wait_for_phase_completion {

    my $cpus      = $_[0];   # Sql Process
    my $ps        = 0;
    my $LogEnd    = 0;
    my $pcomplete = 0;

    # complete phase in all of the sqlplus processes
    for ($ps=0; $ps < $cpus; $ps++)
    {
       #
       # If shutting down database don't send any more SQL
       #
       if ($shutdownphase == $false)
       {
           if ($Table)
           {
               send_sql_to_process($ps,
               "UPDATE $Table SET end_time = SYSTIMESTAMP WHERE script='$File' AND phase='$ph' AND process = '$ps'");
           }
           send_sql_to_process($ps,
           "SELECT 'PHASE_TIME___END $ph ' || TO_CHAR(SYSTIMESTAMP,'YY-MM-DD HH:MI:SS') AS catctl_timestamp FROM SYS.DUAL");
       }

       #
       # This is the last work done for each Sql Process
       # When this files appears we know our work is done
       # for that SQL Process.  Don't use send_sql_to_process.
       # These statements don't like to terminated with
       # the backslash
       #
       $LogEnd = $SpoolLog.$ps.$SpoolLogId.".done";
       print {$pfh[$ps]} "$HostCmd $LogEnd\n";
       $pfh[$ps]->flush;     # flush the buffer
    } # End For

    #
    # Wait until all phases have completed.  Look for
    # the completed file(s).  When all are present we
    # are done for that phase.
    #
    $pcomplete = 0;
    while ($pcomplete < $pcnt)
    {

       #
       # Number of running Sql Processors should match our counter
       # if not one of Sql Processors has died. We must terminate this run.
       #
       if ($pid_cnt != no_running_sql_processes( @pid ))
       {
           kill_running_sql_processes( @pid );
           handle_sigchld();
       }

       for ($ps=0; $ps < $pcnt; $ps++)
       {
           $LogEnd = $SpoolLog.$ps.$SpoolLogId.".done";
           #
           # Is file present increment counter and
           # remove the file.  Wait until all completion
           # files appear for each sql process.
           #
           if (-e $LogEnd)
           {
               $pcomplete++;
               unlink($LogEnd);
               $pfh[$ps]->flush;     # flush the buffer
           }
       } # End For
       select (undef, undef, undef, 0.01);
    }  # End While
}
######################################################################
# Cleanup files
######################################################################
sub clean_up {

    my $cpus           = $_[0];   # Sql Process
    my $bCleanLogFiles = $_[1];   # Clean Out log files
    my $ps;

    #
    # Clean up old log files
    #
    for ($ps=0; $ps < $cpus; $ps++)
    {
        $LogEnd = $SpoolLog.$ps.$SpoolLogId.".done";
        if (-e $LogEnd)
        {
            unlink($LogEnd);
        }

        #
        # Only delete log files greater than or
        # equal to process count
        #
        if ($ps < $Process)
        {
            next;
        }

        #
        # Delete old log file
        #
        if ($bCleanLogFiles == $true)
        {
            $LogEnd = $SpoolLog.$ps.$SpoolLogId.".log";
            if (-e $LogEnd)
            {
                unlink($LogEnd);
            }
        }
    }

    #
    # Delete the pfile
    #
    if ($bFoundPfile && -e $pFileName)
    {
        unlink($pFileName);
        $pFileName = 0;
        $bFoundPfile = $false;
    }

}

######################################################################
# Set Process Number
######################################################################
sub set_process_no {


    $Process = 4;   # use 4 processes as default if no value specified
    if ($opt_n == 0 || $opt_n)
    {
       $Process = $opt_n;
       # Serial Run
       if ($Process == 0)
       {
           $Process = 1; 
           $bRunSerial = $true;
       }

       #
       # Make sure process is valid
       #
       if ($Process > 8)
       {
           $Process = 8;  # Max
       }
       if ($Process < 0)
       {
           $Process = 4;  # Default
       }
    }
}

######################################################################
# Set Connect String
######################################################################
sub set_connect_string {


   if ($User)
   {
       my $password;
       if ($PwdFile)
       {
           open (FileIn, "<$PwdFile") or 
               die "$MSGFCATCTLREADFILE - $PwdFile $!\n";
           $password = <FileIn>;
           close (FileIn);
           chomp $password;
       }
       else
       {
           # prompt for password
           print "Enter Password: ";
           if (!$DBUA || !$Windows)
           {
               ReadMode 'noecho';
           }
           $password = ReadLine 0;
           chomp $password;
           if (!$DBUA || !$Windows)
           {
               ReadMode 'normal';
           }
           print "\n";
       }
       $Connect = $User."/"."\"$password\""." AS SYSDBA";
   }
   else
   {
       # default to OS authentication 
       $Connect = "/ AS SYSDBA"; 
   }

}

######################################################################
# Set Filename to execute
######################################################################
sub set_file_name_to_execute {

    if ($SrcDir)
    {
        if (-r "${SrcDir}")
        {
            $UseDir = 1;
            $Path = $SrcDir."/".$File;
        }
        else
        {
            die "Source file directory does not exist";
        }
    }
    else
    {
        $Path = $File;
    }
}

######################################################################
# Set Log files
######################################################################
sub set_log_files {

    $SpoolLog = index($File,".") > 0 ? substr($File,0,index($File,'.')) : $File;
    if ($LogDir)
    {
        if (-r "${LogDir}")
        {
            $SpoolLog = $LogDir."/".$SpoolLog;
        }
        else
        {
            die "Log file directory does not exist";
        }
    }

    print "\nAnalyzing file $Path\n";

    if ($LogDir)
    {
        print "Log files in $LogDir\n";
    }
}

######################################################################
# Initialization
######################################################################
sub init {

    my $cpus =  $_[0];   # Sql Process
    my $ps;

    #
    # Init Identifier if not present
    #
    if (!$SpoolLogId)
    {
        $SpoolLogId = $NOSPACE;
    }

    #
    # Init pid's
    #
    for ($ps=0; $ps < $cpus; $ps++)
    {
        push(@pid,0);
    }

    #
    # sqlplus plus on windows would not issue
    # the output to a file unless I gave it an
    # invalid command.  On Unix everything works
    # as expected.  Set the platform
    #
    if (substr($OSNAME,0,5) eq "MSWin")
    {
        $HostCmd  = $WindowsDoneCmd;
        $Windows  = $true;
    }


    #
    # Set up connect string
    #
    set_connect_string();

    #
    # Set up sql file to execute
    #
    set_file_name_to_execute();

    #
    # Set Log file directory
    #
    set_log_files();

    #
    # Clean up any old .done files
    #
    clean_up( 500, $true );

    #
    # Initialize the Sql Processors
    #
    init_sql_process( $cpus );

    # Set Start Phase
    if ($opt_p)
    {
        $StartPhase = $opt_p;
    }

    # Set Post Upgrade Message
    if ($DoNotRunPostUpg)
    {
       $PostUpgMessage = 
           "$POSTUPGRADEJOB unable to run -x option has been turned on";
       $DoNotRunPostUpg = $true;
    }
    else
    {
       $PostUpgMessage  = "$POSTUPGRADEJOB is Running";
       $DoNotRunPostUpg = $false;
    }

    #
    # Intialize Session Files
    #
    if ($Script)
    {
       push (@session_files, $Script);  # Session Files
       push (@session_phase, 0);        # Session Phase
    }

} # end of init


######################################################################
# Send Sql to all sql process
######################################################################
sub send_sql_to_all_process { 

   my $cpus =  $_[0];    # Number of Sql Process
   my $Sql  =  $_[1];    # Sql to execute
   my $ps;

    for ($ps=0; $ps < $cpus; $ps++)
    {
        send_sql_to_process($ps, $Sql);
    }

}

######################################################################
# Send Sql to a sql process
######################################################################
sub send_sql_to_process { 

   my $process =  $_[0];   # Sql Process
   my $Sql     =  $_[1];   # Sql to execute

   print {$pfh[$process]} "$Sql $SqlTerm";  # send the Sql to the specified process

}

######################################################################
# Initialization sql process
######################################################################
sub init_sql_process {

    my $cpus =  $_[0];   # Sql Process
    my $ps;

    for ($ps=0; $ps < $cpus; $ps++)
    {
        set_sql_process($false, $ps);
    }
} # end of init_sql_process

######################################################################
# Find sql process not in use
######################################################################
sub find_sql_process
{
    my  $startcpu  =  $_[0];   # Start Cpu Count
    my  $endcpucnt =  $_[1];   # End Cpu Count
    my  $i;

    for ($i=$startcpu; $i < $endcpucnt; $i++)
    {
        if ($sql_process_in_use[$i] == $false)
        {
            return $i;
        }
    }
    die "Find Sql Process has failed Internal error";

} # end of find_sql_process

######################################################################
# Given an array of pids how many sql processors are running
######################################################################
sub no_running_sql_processes
{
    my  @pids  =  @_;   # Array of Pids
    my  $cnt;
    my  @pidsinuse;

    #
    # Take only pids in use
    #
    for ($ps=0; $ps < $Process; $ps++)
    {
        if ($pids[$ps] != 0)
        {
            push (@pidsinuse, $pids[$ps]);
        }
    }

    # 
    # This will return the count of
    # the number of pids running
    #
    $cnt = kill(0, @pidsinuse);
    return $cnt;

} # end of no_running_sql_processes

######################################################################
# Given an array of pids how many sql processors are running
######################################################################
sub kill_running_sql_processes
{
    my  @pids  =  @_;   # Array of Pids
    my  $cnt;
    my  @pidsinuse;

    #
    # Take only pids in use
    #
    for ($ps=0; $ps < $Process; $ps++)
    {
        if ($pids[$ps] != 0)
        {
            push (@pidsinuse, $pids[$ps]);
        }
    }

    # 
    # This will return the count of
    # the number of pids running
    #
    $cnt = kill(-9, @pidsinuse);

} # end of kill_running_sql_processes



######################################################################
# Set Sql Process to true(Inuse) or false(available to do work) 
######################################################################
sub set_sql_process
{
    my  $flag  =  $_[0];       # True(In use) or False(Available)
    my  $cpuno =  $_[1];       # Cpu sql process to set

    $sql_process_in_use[$cpuno] = $flag;


} # end of set_sql_process

######################################################################
# Start Sql Process
######################################################################

sub start_sql_process {

    my  $mstartcnt =  $_[0];   # start count
    my  $mendcnt   =  $_[1];   # end count
    my  $mSpoolLog  =  $_[2];  # Spool log filename
    my  $mConnect   =  $_[3];  # Connect String
    my  $mopt_e     =  $_[4];  # Set Echo on
    my  $mfirstTime =  $_[5];  # First Time
    my  $id;                   # process id

    # 
    # set up signal in case Sql process crashes
    # before it completes it's work
    #
    $SIG{CHLD} = \&handle_sigchld;

    for ($ps=$mstartcnt; $ps < $mendcnt; $ps++)
    {
        $log = $mSpoolLog.$ps.$SpoolLogId.".log";
        # Open for write or append
        if ($mfirstTime == $true)
        {
            open (STDOUT,">", "$log") or die "Failed to Create $log $!\n";
        } else
        {
            close (STDOUT);
            open (STDOUT,"+>>", "$log") or die "Failed to Append to $log $!\n";
        }
        $id = open ($pfh[$ps], "|-", "sqlplus /nolog") or 
            die "sqlplus Process Creation Failed $!\n";
        $pid[$ps] = $id; # Set Pid Id
        $pid_cnt++;      # Increment Pid Counter

        #
        # Send initial commands to the sqlplus session
        # Don't use send_sql_to_process.  These
        # statements don't like to terminated with
        # the backslash
        #
        print {$pfh[$ps]} "connect $mConnect\n";
        #
        # Set echo on by default
        #
        if ($mopt_e)
        {
            print {$pfh[$ps]} "SET ECHO OFF\n";
        }
        else
        {
            print {$pfh[$ps]} "SET ECHO ON\n";
        }

    }

} # end of start_sql_process

######################################################################
# End Sql Process
######################################################################

sub end_sql_process {

    my  $mstartcnt =  $_[0];   # start count
    my  $mendcnt   =  $_[1];   # end count

    $SIG{CHLD} = 'IGNORE';
    for ($ps=$mstartcnt; $ps < $mendcnt; $ps++)
    {
        send_sql_to_process($ps, "EXIT");
        $pid[$ps] = 0;  # Zero out Pid Id
        $pid_cnt--;     # Decrement Pid Counter     
        close ($pfh[$ps]);
    }

} # end of end_sql_process

######################################################################
# Setup procedure to run post upgrade
######################################################################
sub setup_for_post_upgrade {

    #
    # No opt get out
    #
    if ($DoNotRunPostUpg == $true)
    {
        return;
    }

    #
    # Startup the database
    #
    startup_the_database();

} # end setup_for_post_upgrade

######################################################################
# Create the pfile from memory and write to the spool log
######################################################################
sub create_pfile {

    my $PfileCmd = 0;
    my $Cdirline = ($UseDir) ?  "@".$SrcDir."/".$CDIRSCRIPT : "@".$CDIRSCRIPT;

    #
    # Don't Create Pfile if we are not running post upgrade
    #
    if ($DoNotRunPostUpg == $true)
    {
        return;
    }

    #
    # Don't Create Pfile if we already created it.
    #
    if ($bCreatePfile == $true)
    {
        return;
    }

    #
    # Build the PL/SQL Block to create and display the pfile
    #
    $PfileCmd = "set linesize 500;\n set serveroutput on;\n";
    #
    # Call script to create the pfile directory
    #
    $PfileCmd = $PfileCmd.$Cdirline."\n";
    #
    # Start the begin/end block now.
    #
    $PfileCmd = $PfileCmd."declare cnt number:=0;\n";
    $PfileCmd = $PfileCmd.
        "begin sys.dbms_output.put_line ('****************************');\n";
    $PfileCmd = $PfileCmd."select count(*) into cnt from sys.registry\$error;\n";
    $PfileCmd = $PfileCmd."if cnt > 0 then sys.dbms_output.put_line ".
        "('$ERRORTAG' || cnt); end if;\n";
    $PfileCmd = $PfileCmd.
        "sys.dbms_output.put_line ('****************************');\n";
    $PfileCmd = $PfileCmd."sys.dbms_output.put_line ".
        "('$PFILETAG' || sys.dbms_registry_sys.gen_pfile_from_memory());\n";
    $PfileCmd = $PfileCmd.
        "sys.dbms_output.put_line ('****************************');\n";
    $PfileCmd = $PfileCmd."end;\n";

    #
    # Execute command to create the pfile and display in the log file
    #
    send_sql_to_process(0, $PfileCmd);
    $bCreatePfile = $true;

} # end create_pfile


######################################################################
# Starting up database in restricted mode in preperation for the post
# upgrade procedure.
######################################################################
sub startup_the_database {

    my $OpenCommand = "";

    #
    # End the SQL Process that is left over from the first shutdown
    # and read the log file for the pfile.
    #
    end_sql_process(0, 1);
    catctlReadLogFiles($READPFILE);

    #
    # Can't find the pfile get out
    #
    if ($DoNotRunPostUpg == $true)
    {
        start_sql_process(0, 1, $SpoolLog, $Connect, $opt_e, $false);
        return;
    }

    #
    # Start Sql Process and set the startup command
    #
    start_sql_process(0, 1, $SpoolLog, $Connect, $opt_e, $false);

    #
    # Open Mode Normal or Restricted
    #
    if ($bOpenModeNormal)
    {
        $OpenCommand = "startup pfile=$pFileName\n";
    }
    else
    {
        $OpenCommand = "startup restrict pfile=$pFileName\n";
    }

    #
    # Send the startup command
    #
    print {$pfh[0]} "$OpenCommand\n";

    #
    # Wait for the command to complete
    #
    wait_for_phase_completion(1);

    #
    # Restart the SQL Processors this will wait until the open
    # completes. After this we can pick up the upgrade where
    # we left off like the first shutdown never happened.
    #
    end_sql_process(0, 1);
    start_sql_process(0, $Process, $SpoolLog, $Connect, $opt_e, $false);

} # end startup_the_database

######################################################################
# Write Text to log files
######################################################################
sub write_text {

    my  $mstartcnt =  $_[0];   # start count
    my  $mendcnt   =  $_[1];   # end count
    my  $mSpoolLog =  $_[2];   # Spool log filename
    my  $mText     =  $_[3];   # Text To Write to log file

    for ($ps=$mstartcnt; $ps < $mendcnt; $ps++)
    {
        $log = $mSpoolLog.$ps.$SpoolLogId.".log";
        # Open for write or append
        open (FileOut, "+>>$log") or die "Failed to Append Text to $log $!\n";
        print FileOut ("$mText");
        close (FileOut);
    }

} # end of write_text


######################################################################
# Display phases
######################################################################
sub display_phases
{
    my $numofphasefiles = 0;

    for ($i = 0; $i < @phase_type; $i++) 
    {
        $numofphasefiles = @{$phase_files[$i]};
        print STDERR 
            "\n[Phase $i] type is $phase_type[$i] with $numofphasefiles Files\n";

        if ($numofphasefiles > 0)
        {
            $x = 0;
            $fldsize = 16;

            for ($j = 0; $j < $numofphasefiles; $j++)
            {
                $strsize = length($phase_files[$i][$j]);
                print STDERR "$phase_files[$i][$j] ";
                $x++;
                if ($x == 4)
                {
                    print STDERR "\n";
                    $x = 0;
                }
                else
                {
                    # Space out fields
                    for ($z = $strsize; $z < $fldsize; $z++)
                    {
                        print STDERR "$SPACE";
                    }
                }
            }
            print STDERR "\n";
        }
    }

    if($opt_r)
    {
        print STDERR "\nRunning multiprocess phases in reverse order.\n";
    }
    else
    {
        print STDERR "\nUsing $Process processes.\n";
    }

} # end of display_phases

######################################################################
# 
# catctlReadLogFiles - Read Log Files
#
# Description: Read Log Files looking for SP2-1519.  This means we
#              cannot write to error log table.  Mark CATALOG and CATPROC
#              as invalid since we had an error that we could not log.
#
#              Also Parse Log Files looking for Pfile and upgrade errors.
#
# Parameters:
#   - None
######################################################################
sub catctlReadLogFiles
{

  my $ReadFlag =  $_[0];   # Read Flag sp21519 or pfile
  my $LogFile;
  my $SP21519TAG = "SP2-1519";                 # Can't write to error log table
  my $SP21519TAGLEN = length($SP21519TAG);     # Tag Len
  my $PFILETAGLEN = length($PFILETAG);         # Tag Len
  my $ERRORTAGLEN = length($ERRORTAG);         # Error Tag Len
  my $CompareStr = "";                         # Compare String 
  my $bValid     = $true;                      # Assume Valid
  my $INVCATPROC = "execute sys.dbms_registry.invalid('CATPROC')";
  my $INVCATALOG = "execute sys.dbms_registry.invalid('CATALOG')";
  my $RESETDBUA  = "SELECT dbms_registry_sys.time_stamp('rdbms_end') as timestamp from sys.dual";
  my $line       = 0;

  #
  # No opts
  #
  if ($ReadFlag == $READPFILE)
  {
      # Don't run post upgrade
      if ($DoNotRunPostUpg == $true)
      {
          return;
      }
      # Already have pfile name
      if ($bFoundPfile == $true)
      {
          return;
      }
  }

  #
  # Loop on Each Log File
  #
  for ($x = 0; $x < $Process; $x++)
  {
     #
     # Open and Read in Log Files
     #
     $LogFile = $SpoolLog.$x.$SpoolLogId.".log";
     open (FileIn, "<$LogFile") or die "$MSGFCATCTLREADFILE - $LogFile $!\n";

     #
     # Search for SP2-1519 error in the RDBMS component of the
     # upgrade.  If found we need to mark RDBMS component as Invalid.
     #
     while (<FileIn>)
     {
         #
         # Store the line
         #
         $line = $_;
         chomp($line);

         #
         # Scan the first few characters of the line
         #
         if ( ($ReadFlag == $READSP21519)     &&
              (length($line) > $SP21519TAGLEN) )
         {  
             #
             # Look for Tag at the beginning of line only
             #
             $CompareStr = substr($line, 0, $SP21519TAGLEN);

             #
             # If we find the Can't write to error log table
             # then RDBMS is invalid some really bad error
             # occurred that we could not log
             #
             if ($CompareStr eq $SP21519TAG)
             {
                 $bValid = $false;
                 $DoNotRunPostUpg = $true;
                 $PostUpgMessage  = 
                     "$POSTUPGRADEJOB unable to run. Error is $SP21519TAG.";
                 last;
             }
         }

         #
         # Scan the first few characters of the line 
         # Only if we are scanning for a pfile.
         #
         if ($ReadFlag == $READPFILE)
         {
             #
             # Look for Errors
             #
             if (length($line) > $ERRORTAGLEN)
             {
                 #
                 # Look for Tag at the beginning of line only
                 #
                 $CompareStr = substr($line, 0, $ERRORTAGLEN);

                 #
                 # If we find the Error Tag
                 # then stop the running of the post upgrade.
                 #
                 if ($CompareStr eq $ERRORTAG)
                 {
                     $DoNotRunPostUpg = $true;
                     $bFoundPfile = $false;
                     $PostUpgMessage  = 
              "$POSTUPGRADEJOB unable to run. Errors found during upgrade";
                     last;
                 }
             }

             #
             # Look for PFile
             #
             if (length($line) > $PFILETAGLEN)
             {
                 #
                 # Look for Tag at the beginning of line only
                 #
                 $CompareStr = substr($line, 0, $PFILETAGLEN);

                 #
                 # If we find the Pfile Tag
                 # then we can parse out the pfile name
                 #
                 if ($CompareStr eq $PFILETAG)
                 {
                     $pFileName = substr($line, $PFILETAGLEN, 
                                         length($line)-$PFILETAGLEN);
                     $bFoundPfile = $true;
                     last;
                 }
             }
         }

     } # End while loop

     # Close file
     close (FileIn);

     #
     # Check pfile
     #
     if ($ReadFlag == $READPFILE)
     {
         #
         # Did not find the pfile and no errors found
         #
         if (($bFoundPfile == $false) && ($DoNotRunPostUpg == $false)) 
         {
             $DoNotRunPostUpg = $true;
             $PostUpgMessage  = 
  "$POSTUPGRADEJOB unable to run. Can't locate Pfile to startup the database";
         }
         last;  # Only process catupgrd0.log
     }
     else
     {
         #
         # If No Longer Valid then we are done get out
         #
         if (!$bValid)
         {
             #
             # We are going to invalidate both components since
             # I can't tell for sure who is invalid
             #
             print STDERR "\nFound Tag $SP21519TAG Invalidating Oracle Server Check Logs\n";
             send_sql_to_process(0, $INVCATPROC);
             send_sql_to_process(0, $INVCATALOG);
             send_sql_to_process(0, $RESETDBUA);
             last;

         } # End if not valid
     } # End if Check Pfile
 } # End looping on log files

} # End of catctlReadLogFiles

######################################################################
# Subroutine to recursive scan through the control files
######################################################################

sub scan_file {

   # global $depth variable tracks recursion of -X scripts
 
   my $path = $_[0];   # first argument is file name
   my $thread;

   # open the file to read from and load lines starting with @
   open($path, "<", $path) or die "Failed to open $path: $!\n";
   @{$scripts[$depth]}= grep { /^@/ | /--CAT/ }<$path>;
   close ($path);

   if ( $#{$scripts[$depth]} < 0) {
      print STDERR "No scripts found in file $path\n";
      return;  ## no scripts found
   } else {
      print STDERR "$#{$scripts[$depth]} scripts found in file $path\n";
   }

   # analyze scripts for control commands and populate control arrays
   for ($x[$depth] = 0; $x[$depth] < @{$scripts[$depth]}; $x[$depth]++) 
   {
     $i = $x[$depth];
     my $thread = $single;
     my $new_path;
     my $file;
     my $noofcompile;

     #
     # Ignore sqlsessstart.sql and sqlsessend.sql for our driver files.
     # We don't want them involved in the processing of the SQL.
     # ORACLE_SCRIPT is set in our session scripts for these files.
     #
     if (index($scripts[$depth][$i], $SessionStart) != -1 ||
         index($scripts[$depth][$i], $SessionEnd)   != -1)
     {
         next;
     }

     #
     # only @@ scripts or --CATCTL lines can be in catctl files
     #
     if (substr($scripts[$depth][$i],0,8) eq "--CATCTL") {
         # new phase
         if (index($scripts[$depth][$i], "-M") > 0) {
              $thread = $multi;
         }
         else {
              $thread = $single;  # default to Single
         }
         push(@phase_type,$thread);
         push(@phase_files,[]);
         push (@phase_compile, $bLoadWithComp);
         $noofcompile = $#phase_compile;

         #
         # Want to look at the last entry to
         # carry it forward to the next phase
         #
         if ($noofcompile != 0)
         {
             $phase_compile[$noofcompile] = $phase_compile[$noofcompile-1];
         }

         #
         # Start Load without Compile
         #
         if (index($scripts[$depth][$i], "-CS") > 0) {
             push(@{$phase_files[$#phase_files]}, $LoadWithoutComp_sql_command);
             $phase_compile[$noofcompile] = $bLoadWithoutComp;
         }

         #
         # End Load without Compile
         #
         if (index($scripts[$depth][$i], "-CE") > 0) {
             push(@{$phase_files[$#phase_files]}, $LoadWithComp_sql_command);
             $phase_compile[$noofcompile] = $bLoadWithComp;
         }

         #
         # Run This file in catctl.pl only. Does not run in catupgrd.sql
         # Format is --CATCTL -SE filename.sql filename.sql ... EOL
         # We execute an -SE tag the same as we execute the -S tag the
         # only difference is that the sql found under this tag is the form of
         # a comment and does not get executed if the script is run at the
         # SQL command prompt.
         #
         if ((index($scripts[$depth][$i], "-SE") > 0))
         {

             if ($DoNotRunPostUpg == $false)
             {
                 # Skip tag info and parse out first file
                 my $tag = "--CATCTL -SE ";
                 my $files = substr($scripts[$depth][$i], length($tag));
                 my $idx   = index($files, " ");
                 $file     = substr($files, 0, $idx);

                 #
                 # While not at end of line add files into phase
                 #
                 while ($file ne "EOL")
                 {
                     #
                     # Don't shutdown the database if open mode is normal
                     #
                     if (($bOpenModeNormal) && ($file eq $SHUTDOWNJOB))
                     {
                         $file = $NOTHINGJOB;
                     }
                     #
                     # Add File to the phase
                     #
                     push(@{$phase_files[$#phase_files]}, $file);
                     #
                     # Move to the next file
                     #
                     $files = substr($files, $idx+1);
                     $idx   = index($files, " ");
                     $file  = substr($files, 0, $idx);
                 }
             }
             else
             {
                 $file = $NOTHINGJOB;
                 push(@{$phase_files[$#phase_files]}, $file);
             }
         }


         # This tells us to restart the sql process
         if (index($scripts[$depth][$i], "-R") > 0) {
             push(@{$phase_files[$#phase_files]}, $restart_sql_command);
         }
     } else {
        # new script
        if (substr($scripts[$depth][$i],0,2) eq "@@") {
           # process script
           my $ctl = index($scripts[$depth][$i],"--CATFILE");
           my $idx = index($scripts[$depth][$i]," ");
           $idx = $idx > 0 ? $idx : index($scripts[$depth][$i],"\n");
           $file = substr($scripts[$depth][$i],2,$idx-2);

           # add .sql if no suffix
           $idx = index($scripts[$depth][$i],".");
           $file = $idx > 0 ? $file : $file . ".sql";

           if ($ctl > 0)
           {
              # Only process -X files, -I files are ignored
              if (index($scripts[$depth][$i], "-X", $ctl) > 0)
              {
                  $new_path = $UseDir ? $SrcDir."/".$file : $file;
                  print STDERR "Next path: $new_path\n";
                  $depth++;
                  scan_file($new_path);
                  $depth--;
              }

              # Process Session Files
              if (index($scripts[$depth][$i], "-SES", $ctl) > 0)
              {
                  push (@session_files, $file);
                  push (@session_phase, $#phase_type);
              }


           } else {  # add file to array of files for current phase
              # store script to run
              push(@{$phase_files[$#phase_files]}, $file);   
           }
         } else {  # not @@ or --CATCTL
            print STDERR "Line not processed: $scripts[$depth][$i]\n";
         }
     }  # end of line check
   }  # end of for loop
}  # end of scan_file


######################################################################
# Send a line to a sqlplus sprocess (arg 0) to run a file name (arg 1)
######################################################################

sub run_file {

my $process =  $_[0];    # first argument is process subscript
my $file    =  $_[1];    # second argument is file name to run
my $line = ($UseDir) ?  "@".$SrcDir."/".$file : "@".$file;

   #
   # Log start of SQL file run
   #
   send_sql_to_process($process,
   "\nSELECT ' Started: $file on CPU: $process at ' || TO_CHAR(SYSTIMESTAMP,'YY-MM-DD HH:MI:SS') AS catctl_timestamp FROM SYS.DUAL");

   #print {$pfh[$process]} "set serveroutput on;\n";  # Debug
   #print {$pfh[$process]} "set echo on;\n";  # Debug
   #print {$pfh[$process]} "select '????? $file ?????', count(*) from obj\$ where status > 1;\n";  # Debug

   #
   # Send @@commands to sql process.
   # Don't use send_sql_to_process.  These
   # statements don't like to terminated with
   # the backslash
   #
   print {$pfh[$process]} "$line\n"; 

   #print {$pfh[$process]} "show errors;\n";

   #
   # Log end of SQL file run if not the last job
   #
   if ($shutdownphase == $false)
   {
       send_sql_to_process($process,
       "\nSELECT 'Finished: $file on CPU: $process at ' || TO_CHAR(SYSTIMESTAMP,'YY-MM-DD HH:MI:SS') AS catctl_timestamp FROM SYS.DUAL");
   }
}

######################################################################
#  If one of the child process terminates, it is a fatal error
######################################################################

sub handle_sigchld {
    print STDERR "\nA process terminated prior to $File completion.\n";
    print STDERR "Review the $SpoolLog*.log files to identify the failure.\n";
    $SIG{CHLD} = 'IGNORE';  # now ignore any child processes
    clean_up($Process, $false); # Cleanup
    for ($ps=0; $ps < $Process; $ps++)
    {   
        close ($pfh[$ps]);  # Close any opened process
    }
    write_text(0, $Process, $SpoolLog, $EndTag); # Write End Tag
    die;  # Exits with 255 status
}
