Rem
Rem $Header: rdbms/admin/catuposb.sql /st_rdbms_12.1.0.1/3 2013/03/15 10:08:28 cmlim Exp $
Rem
Rem catuposb.sql
Rem
Rem Copyright (c) 2012, 2013, Oracle and/or its affiliates. 
Rem All rights reserved. 
Rem
Rem    NAME
Rem      catuposb.sql - CAT UPgrade update Oracle-Supplied Bits
Rem
Rem    DESCRIPTION
Rem      Reads external tables objxt and userxt.
Rem      Updates obj$, user$ with oracle-supplied bits as listed in external
Rem      tables.
Rem
Rem    NOTES
Rem      Add oracle-supplied bits in dictionary after upgrading a pre-12.1
Rem      db to 12c.
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    cmlim       03/13/13 - XbranchMerge cmlim_bug-16306200 from main
Rem    cmlim       03/02/13 - bug 16306200: flush shared_pool after updating
Rem                           data dict
Rem                         - mark partition indexes and table partitions as
Rem                           oracle supplied
Rem    cmlim       03/01/13 - XbranchMerge cmlim_bug-16085743 from
Rem                           st_rdbms_12.1.0.1
Rem    cmlim       01/08/13 - bug 16085743: mark system internal objects as
Rem                           oracle supplied if base obj is oracle supplied
Rem    cdilling    12/30/12 - XbranchMerge cdilling_bug-16031506 from
Rem    cdilling    12/19/12 - set oracle supplied bit for ordsys
Rem    cmlim       11/04/12 - bug 14763826 - add oracle-supplied bits in
Rem                           dictionary
Rem    cmlim       11/04/12 - Created
Rem


set serveroutput on

Rem ************************************************************************
Rem Create external tables with oracle-supplied bit info
Rem ************************************************************************

@@catupcox.sql


Rem ************************************************************************
Rem Update USER$'s SPARE1 with oracle_supplied bit using external table
Rem userxt as reference
Rem
Rem Note: Must do users first because some of the bit markings for system
Rem internally generated objects depend on them being owned by oracle
Rem supplied users.
Rem ************************************************************************

declare
 rc           sys_refcursor;
 name         varchar2(128);   -- username
 sqlstr       varchar2(5000);  -- to build the sql stmt for execute
 rows_queried number;          -- # of rows returned in a query
 rows_updated number := 0;     -- total # of rows updated

begin

  EXECUTE IMMEDIATE 'select count(*) from sys.userxt' into rows_queried;
  dbms_output.put_line('catuposb.sql : ' || rows_queried ||
                        ' rows in userxt');

  EXECUTE IMMEDIATE
    'select count(*) from sys.user$ where bitand(spare1, 256) = 256'
    into rows_queried;
  dbms_output.put_line('catuposb.sql : before update - ' || rows_queried ||
                       ' oracle-supplied user$ rows');

  -- begin of update in user$
  BEGIN
    OPEN rc FOR select name
                from sys.userxt;

    LOOP
      FETCH rc INTO name;
      EXIT WHEN rc%NOTFOUND;
    
      sqlstr := 'update sys.user$ ' ||
                'set spare1 = spare1 + 256 ' ||
                'where bitand(spare1, 256) = 0 ' || 
                ' and name = ''' || name || '''';

      execute immediate sqlstr;

      rows_updated := rows_updated + SQL%ROWCOUNT;
    END LOOP;
    commit;
    CLOSE rc;
  END;  -- end of update

  dbms_output.put_line('catuposb.sql : ' || rows_updated ||
                       ' user$ rows updated with oracle-supplied bit');

  EXECUTE IMMEDIATE
    'select count(*) from sys.user$ where bitand(spare1, 256) = 256'
    into rows_queried;
  dbms_output.put_line('catuposb.sql : after update - ' || rows_queried ||
                       ' oracle-supplied user$ rows');
end;
/


Rem ************************************************************************
Rem Update OBJ$'s FLAGS with oracle_supplied bit using external table
Rem objxt as reference
Rem ************************************************************************

declare
  rc       sys_refcursor;
  sqlstr   varchar2(5000);  -- to build the sql stmt for execute
  objid    number;          -- object id

  -- create a record of arrays to store object info
  type objRec is record
  (
    name     dbms_sql.varchar2_table,
    subname  dbms_sql.varchar2_table,
    owner    dbms_sql.varchar2_table,
    type#    dbms_sql.number_table
  );

  -- variables associated with objRec
  o_rec         objRec;           -- obj record
  o_array_sz_1  number := 10000;  -- max array size for for null obj subnames
  o_array_sz_2  number := 1000;   -- max array size for nonnull obj subnames
  rows_queried  number;           -- # of rows returned in a query
  rows_updated  number := 0;      -- # of rows updated      

begin

  EXECUTE IMMEDIATE 'select count(*) from sys.objxt' into rows_queried;
  dbms_output.put_line('catuposb.sql : ' || rows_queried ||
                        ' rows in objxt');

  EXECUTE IMMEDIATE
    'select count(*) from sys.obj$ where bitand(flags, 4194304) = 4194304'
    into rows_queried;
  dbms_output.put_line('catuposb.sql : before update - ' || rows_queried ||
                       ' oracle-supplied obj$ rows');

  --
  -- begin of update in obj$ where obj$.subname IS null
  --
  BEGIN
    -- select from external table objxt where subname IS null
    OPEN rc FOR select owner, name, type#
                from sys.objxt
                where subname is null;

    -- bulk collect results into obj record of arrays
    LOOP
      FETCH rc BULK COLLECT INTO
        o_rec.owner, o_rec.name, o_rec.type#
      LIMIT o_array_sz_1;

      EXIT WHEN rc%NOTFOUND;

      -- update obj$ using the data from objxt
      forall i in 1 .. o_rec.owner.count
          update sys.obj$
           set flags = flags + 4194304
           where  bitand(flags, 4194304) = 0
            and name = o_rec.name(i)
            and subname is null
            and "OWNER#" =
                ( select "USER#" from sys.user$
                  where name = o_rec.owner(i) )
            and "TYPE#" = o_rec.type#(i) ;

      -- rows updated in obj$ so far
      rows_updated := rows_updated + SQL%ROWCOUNT;

      commit;
    END LOOP;
    CLOSE rc;
  END;  -- end of update where obj$.subname is null

  --
  -- begin of update in obj$ where obj$.subname IS NOT null
  --
  BEGIN
    -- select from external table objxt where subname IS NOT null
    OPEN rc FOR select owner, name, subname, type#
                from sys.objxt
                where subname is not null;
    LOOP
      -- bulk collect results into obj record of arrays
      FETCH rc BULK COLLECT INTO
        o_rec.owner, o_rec.name, o_rec.subname, o_rec.type#
      LIMIT o_array_sz_2;

      -- update obj$ using the data from objxt
      forall i in 1 .. o_rec.owner.count
          update sys.obj$
           set flags = flags + 4194304
           where  bitand(flags, 4194304) = 0
            and name = o_rec.name(i)
            and subname is not null
            and subname = o_rec.subname(i)
            and "OWNER#" =
                ( select "USER#" from sys.user$
                  where name = o_rec.owner(i) )
            and "TYPE#" = o_rec.type#(i) ;

      EXIT WHEN rc%NOTFOUND;

      -- total rows updated in obj$
      rows_updated := rows_updated + SQL%ROWCOUNT;
      
      commit;
    END LOOP;
    CLOSE rc;
  END;  -- end of update where obj$.subname is not null

  --
  -- begin of update in obj$ where owner is ordsys and type# is 13
  -- note: There are types in an upgraded db that belong to ORDSYS and
  --       generated as a side effect of registering Oracle supplied schemas.
  --       Since the names are not static, it was advised by ORDIM that it
  --       is fine to mark all types owned by ORDSYS as oracle supplied, as
  --       ORDSYS will only own oracle-supplied data anyway.
  --
  BEGIN 
    sqlstr :=  'update sys.obj$ ' ||
               'set flags = flags + 4194304 ' ||
               'where  bitand(flags, 4194304) = 0 ' ||
               'and subname is null ' ||
               'and "OWNER#" = ' ||
               '( select "USER#" from sys.user$ ' ||
               ' where name = ''ORDSYS'') ' ||
               ' and "TYPE#" = 13';
              
    execute immediate sqlstr;

    rows_updated := rows_updated + SQL%ROWCOUNT;
    commit;
  END;  -- end of update where owner is ordsys and type# is 13


  --
  -- begin of update in obj$ for these system internal generated objects:
  -- SYS_IOT_OVER_%, SYS_IOT_TOP_%, SYS_LOB%, SYS_IL%, SYS_YOID%
  --
  BEGIN
    -- for these system internally generated objects that are (1) not marked
    -- oracle supplied and (2) are owned by oracle supplied users, then
    -- parse their system generated names for base obj#s 
    OPEN rc FOR select unique(regexp_substr(o.name, '([[:digit:]]+)'))
                from sys.obj$ o, sys.user$ u
                where (o.name like 'SYS\_IL%' ESCAPE '\'
                       or o.name like 'SYS\_IOT\_OVER_%' ESCAPE '\'
                       or o.name like 'SYS\_IOT\_TOP\_%' ESCAPE '\'
                       or o.name like 'SYS\_LOB%' ESCAPE '\'
                       or o.name like 'SYS\_YOID%' ESCAPE '\'
                      )
                      and bitand(o.flags, 4194304) = 0
                      and o.owner# = u.user#
                      and bitand(u.spare1, 256) = 256;
 
    LOOP
      FETCH rc INTO objid;
      EXIT WHEN rc%NOTFOUND;

      -- mark system internally generated objs with oracle supplied bit
      -- if base object is oracle supplied and system internal obj had not
      -- been marked yet
      EXECUTE IMMEDIATE
        'update sys.obj$ set flags = flags + 4194304 ' ||
        'where ' ||
        ' (name like :1 ' ||
        '  or name like :2 ' ||
        '  or name like :3 ' ||
        '  or name like :4 ' ||
        '  or name like :5) ' ||
        ' and 1 = ' ||
        '  (select decode(count(*),0,0,1) from obj$ ' ||
        '   where obj# = ' || objid ||
        '     and bitand(flags, 4194304) = 4194304) ' ||
        ' and bitand(flags, 4194304) = 0'
      USING
        'SYS_IL%' || objid || '%',
        'SYS_IOT_OVER_%' || objid || '%',
        'SYS_IOT_TOP_%' || objid || '%',
        'SYS_LOB%' || objid || '%',
        'SYS_YOID%' || objid || '%';

      -- rows updated in obj$ so far
      rows_updated := rows_updated + SQL%ROWCOUNT;

      commit;
    END LOOP;
    CLOSE rc;
  END;  -- end of update for system internally generated objs


  --
  -- begin of update in obj$ for these system internal generated objects:
  -- SYS_C% (like SYS_C00698) and SYS_FK%
  --
  BEGIN
    -- find obj#s for SYS_C% and SYS_FK% where (1) the index had not been
    -- marked as oracle supplied yet and (2) where table is oracle supplied
    -- and (3) index owner is oracle supplied
    --
    -- note: iu for index-user, io for index-obj, ibo for index's-base-obj
    OPEN rc FOR select io.obj#
                from sys.ind$ i, sys.user$ iu,
                     sys.obj$ io, sys.obj$ ibo
                where (io.name like 'SYS_C%' or io.name like 'SYS_FK%')
                      and io.type# = 1
                      and io.obj# = i.obj# 
                      and i.bo# = ibo.obj#
                      and bitand(io.flags, 4194304) = 0
                      and bitand(ibo.flags, 4194304) = 4194304
                      and io.owner# = iu.user#
                      and bitand(iu.spare1, 256) = 256;
 
    LOOP
      FETCH rc INTO objid;
      EXIT WHEN rc%NOTFOUND;

      -- mark SYS_C% and SYS_FK% objs with oracle supplied bit if objects
      -- had not been marked yet
      EXECUTE IMMEDIATE
        'update sys.obj$ set flags = flags + 4194304 ' ||
        'where obj# = :1 and bitand(flags, 4194304) = 0'
      USING
        objid;

      -- rows updated in obj$ so far
      rows_updated := rows_updated + SQL%ROWCOUNT;

      commit;
    END LOOP;
    CLOSE rc;
  END;  -- end of update for system internally generated objs

  --
  -- begin of update in obj$ for PARTITION TABLES
  --
  -- (1) will mark unmarked partition table (type# 19) if base table (type# 2)
  --     had already been marked as oracle supplied
  -- USERNAME  OBJECTNAME                     TYPE# SUBNAME
  -- SYS       WRH$_ACTIVE_SESSION_HISTORY    19    WRH$_ACTIVE_1060409768_76 
  -- SYS       WRH$_ACTIVE_SESSION_HISTORY    2
  --
  -- (2) will mark unmarked partition index (type# 20) if base index (type# 1)
  --     had been marked as oracle supplied
  -- USERNAME  OBJECTNAME                     TYPE# SUBNAME
  -- SYS       WRH$_REPORTS_DETAILS_IDX01     20    SYS_P222
  -- SYS       WRH$_REPORTS_DETAILS_IDX01     1
  --
  BEGIN
    OPEN rc FOR select p.obj#
                from sys.obj$ p, sys.obj$ t
                where
                  p.type# in (19, 20) and p.subname is not null
                     and bitand(p.flags, 4194304)=0
                  and p.name = t.name and p.owner# = t.owner#
                  and t.type# in (2, 1) and t.subname is null
                     and bitand(t.flags, 4194304)=4194304;

    LOOP
      FETCH rc INTO objid;
      EXIT WHEN rc%NOTFOUND;
    
      EXECUTE IMMEDIATE
        'update sys.obj$ set flags = flags + 4194304 ' ||
        'where obj# = :1'
      USING
        objid;

      rows_updated := rows_updated + SQL%ROWCOUNT;

      commit;
    END LOOP;
    CLOSE rc;
  END;  -- end of update for system internally generated objs


  --
  -- end of updating oracle supplied bit in obj$
  -- now determine # of rows that had been updated
  --
  dbms_output.put_line('catuposb.sql : ' || rows_updated ||
                       ' obj$ rows updated with oracle-supplied bit');

  EXECUTE IMMEDIATE
    'select count(*) from sys.obj$ where bitand(flags, 4194304) = 4194304'
    into rows_queried;
  dbms_output.put_line('catuposb.sql : after update - ' || rows_queried ||
                       ' oracle-supplied obj$ rows');

  

end;
/

alter system flush shared_pool;
alter system flush shared_pool;
alter system flush shared_pool;

Rem ************************************************************************
Rem drop external tables and associated directory object
Rem ************************************************************************

drop table sys.objxt;
drop table sys.userxt;
drop directory upg_xt_dir;


set serveroutput off
