Rem
Rem $Header: rdbms/admin/dbmsgwmcl.sql /main/10 2012/08/17 14:29:54 sdball Exp $
Rem
Rem dbmsgwmcl.sql
Rem
Rem Copyright (c) 2011, 2012, Oracle and/or its affiliates. 
Rem All rights reserved. 
Rem
Rem    NAME
Rem      dbmsgwmcl.sql - Global Workload Management Cloud administration
Rem
Rem    DESCRIPTION
Rem      Defines the interfaces for dbms_gsm_cloudadmin package that is
Rem      used for cloud administration performed by GSM and GSMCTL.
Rem
Rem    NOTES
Rem      This package is only loaded on the GSM cloud catalog database.
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    sdball      08/14/12 - Add oracle_home and hostname parameters to addGSM
Rem    sdball      08/06/12 - Add MaskPolicy
Rem    sdball      06/13/12 - Support for number of instances
Rem    sdball      05/18/12 - Add VerifyCatalog
Rem    sdball      02/15/12 - Variables for GSM limits
Rem    sdball      01/23/12 - Add rougeGSM
Rem    sdball      01/12/12 - Add syncParameters
Rem    sdball      12/05/11 - change gsm_pooladmin_role to gsm_gsm_pooladmin_role
Rem    sdball      11/30/11 - Add auto VNCR functionality
Rem    sdball      11/28/11 - auto VNCR functionality
Rem    lenovak     11/08/11 - remove database pool admin
Rem    lenovak     07/22/11 - vncr support
Rem    mjstewar    07/22/11 - Allow buddy region to be NULL for modifyRegion
Rem    mjstewar    04/27/11 - Second GSM transaction
Rem    mjstewar    02/02/11 - Created
Rem

-- SET ECHO ON
-- SPOOL dbmsgwmcl.log

--*****************************************************************************
-- Database package for GSM cloud administrator functions.
--*****************************************************************************

CREATE OR REPLACE PACKAGE dbms_gsm_cloudadmin AS


--*****************************************************************************
-- Package Public Types
--*****************************************************************************


--*****************************************************************************
-- Package Public Constants
--*****************************************************************************

gsm_master_lock_name    constant    varchar2(19) := 'ORA$GSM_MASTER_LOCK';
rogueGSM                constant    number := 99;
MaxGSM                  constant    number := 5;

--*****************************************************************************
-- Package Public Exceptions
--*****************************************************************************



--*****************************************************************************
-- Package Public Procedures
--*****************************************************************************

maxwait     constant    integer := 32767;            -- Wait forever

-------------------------------------------------------------------------------
--
-- PROCEDURE     getMasterLock
--
-- Description:
--     Request the GSM catalog master lock in exclusive mode.
--
-- Parameters:
--     timeout:  the number of seconds to wait for the lock
--               "maxwait" means to wait forever
--     lock_handle: handle used to identify the lock
--                  should be passed to releaseMasterLock to release the lock
--                  size can be up to 128
--
-- Returns:
--      0 - success
--      1 - timeout
--      2 - deadlock
--      3 - parameter error
--      4 - already own lock
--      5 - illegal lock handle
--
-- Notes:
--     The routine uses dbms_lock.allocate_unique, so will always do a
--     transaction commit.
--
--     Lock is held until it is explicitly released or session terminates.
-------------------------------------------------------------------------------  

FUNCTION getMasterLock( timeout     IN  integer default maxwait,
                        lock_handle OUT varchar2,
                        gsm_name    IN  varchar2 default NULL )
 RETURN integer;

-------------------------------------------------------------------------------
--
-- PROCEDURE     releaseMasterLock
--
-- Description:
--     Release the GSM catalog master lock acquired previously by getMasterLock.
--
-- Parameters:
--     lock_handle: handle returned by getMasterLock
--
-- Returns:
--      0 - success
--      3 - parameter error
--      4 - don't own lock
--      5 - illegal lock handle
--
-- Notes:
--
-------------------------------------------------------------------------------  

FUNCTION releaseMasterLock( lock_handle IN varchar2 )
 RETURN integer;


-------------------------------------------------------------------------------
--
-- PROCEDURE     createCloud
--
-- Description:
--     Creates a cloud entry in the cloud catalog.
--
-- Parameters:
--     cloud_name: name to give the cloud.
--
-- Notes:
--     Currently the catalog only supports one cloud.
-------------------------------------------------------------------------------      
PROCEDURE createCatalog(cloud_name IN varchar2 default NULL,
                       autoVNCR IN number default  dbms_gsm_utility.updateTrue,
                       instances IN number default NULL);
PROCEDURE createCloud( cloud_name IN varchar2 default NULL,
                       autoVNCR IN number default  dbms_gsm_utility.updateTrue,
                       instances IN number default NULL);

-------------------------------------------------------------------------------
--
-- PROCEDURE     modifyCatalog
--
-- Description:
--     Modifies information in the cloud catalog
--
-- Parameters:
--     autoVNCR: boolean - updateTrue = Turn on autoVNCR
--                         updateFalse = Turn off autoVNCR
--
-- Notes:
--     Currently the catalog only supports one cloud.
-------------------------------------------------------------------------------
PROCEDURE modifyCatalog(autoVNCR IN number default NULL);
-------------------------------------------------------------------------------
--
-- PROCEDURE     removeCloud
--
-- Description:
--     Removes the cloud entry from the cloud catalog.
--
-- Parameters:
--     cloud_name: name of the cloud to remove.      
--
-- Notes:
--     Currently the catalog only supports one cloud.
------------------------------------------------------------------------------- 
PROCEDURE removeCatalog( cloud_name IN varchar2 default NULL );  -- TODO: remove
PROCEDURE removeCloud( cloud_name IN varchar2 default NULL );


-------------------------------------------------------------------------------
--
-- PROCEDURE     addGSM
--
-- Description:
--     Adds a GSM to the cloud.        
--
-- Parameters:
--     gsm_name:             GSM alias name
--     gsm_endpoint1:        Listener endpoint
--     gsm_endpoint2:        TODO:?
--     local_ons_port:       Local ONS port for ONS server process
--     remote_ons_port:      Remote ONS port of ONS server process
--     region_name:          The GSM region, if NULL will use the default.
--     gsm_number:           Unique number assigned to the GSM
--
-- Notes:
--     Updates the "_remote_gsm" parameter on the catalog database to point
--     to the new GSM.
--
--     region_name can be NULL if there is only one region in the cloud.
--     In which case the GSM will be added to that region.
--
--     No assumptions should be made about gsm_number other than it is 
--     unique for the cloud.  For example, the caller should not assume that it
--     is a monotonically increasing number.
------------------------------------------------------------------------------- 
PROCEDURE addGSM( gsm_name        IN  varchar2,
                  gsm_endpoint1   IN  varchar2,
                  gsm_endpoint2   IN  varchar2,
                  local_ons_port  IN  number,
                  remote_ons_port IN  number,
                  region_name     IN  varchar2 default NULL,
                  gsm_number      OUT number,
                  gsm_oracle_home IN  varchar2 default NULL,
                  gsm_hostname    IN  varchar2 default NULL );

-------------------------------------------------------------------------------
--
-- PROCEDURE     syncParameters
--
-- Description:
--     Syncronize spfile parameter values using database information        
--
-- Parameters:
--    NONE
--
-- Notes:
--    Currently updates the _gsm and _cloud_name parameters. These values are
--    required for the catalog database instance and will not be set on
--    data-guard standby databases (since createCatalog is never run there). 
--    This function will be executed as part of the database open notifier
--    callback on any primary database that is a catalog database
--
------------------------------------------------------------------------------
PROCEDURE syncParameters;

-------------------------------------------------------------------------------
--
-- PROCEDURE     modifyGSM
--
-- Description:
--     Changes a GSM attributes.
--
-- Parameters:
--     gsm_name:             GSM alias name
--     gsm_endpoint1:        Listener endpoint
--     gsm_endpoint2:        TODO:?
--     local_ons_port:       Local ONS port for ONS server process
--     remote_ons_port:      Remote ONS port of ONS server process
--     region_name:          The GSM region.
--
-- Notes:
--     One or more of the attributes can be changed on each call.
--
--     If "gsm_endpoint1" is changed, then will update the "_remote_gsm" 
--     parameter on the catalog database with the new endpoint.
------------------------------------------------------------------------------- 
PROCEDURE modifyGSM( gsm_name        IN  varchar2,
                     gsm_endpoint1   IN  varchar2 default NULL,
                     gsm_endpoint2   IN  varchar2 default NULL,
                     local_ons_port  IN  number   default NULL,
                     remote_ons_port IN  number   default NULL,
                     region_name     IN  varchar2 default NULL );


-------------------------------------------------------------------------------
--
-- PROCEDURE     removeGSM
--
-- Description:
--     Removes a GSM from the cloud.        
--
-- Parameters:
--     gsm_name:             GSM alias name
--            
-- Notes:
--    It is up to the caller to insure that the GSM has been stopped.
--
--    Will remove this GSM endpoint from the "_remote_gsm" parameter on the
--    catalog database.
------------------------------------------------------------------------------- 
PROCEDURE removeGSM( gsm_name        IN  varchar2 );

-------------------------------------------------------------------------------
--
-- PROCEDURE     addVNCR
--
-- Description:
--     Adds VNCR to Cloud        
--
-- Parameters:
--     name:             VNCR name
--     group_id:            VNCR group id
--            
-- Notes:
--    Group id could be NULL. If set, it allows group removal of VNCRs
------------------------------------------------------------------------------- 
PROCEDURE addVNCR( name        IN  varchar2, 
                   group_id in varchar2 default NULL,
                   updateRequestTable  IN number 
                     default dbms_gsm_utility.updateTrue);

-------------------------------------------------------------------------------
--
-- PROCEDURE     removeVNCR
--
-- Description:
--     removes VNCR from Cloud        
--
-- Parameters:
--     name:             VNCR name
--     group_id:            VNCR group id
--            
-- Notes:
--    One and only one of either group id or name could be NULL. 
------------------------------------------------------------------------------- 
PROCEDURE removeVNCR( name IN  varchar2 default NULL, 
                      group_id in varchar2 default NULL);

-------------------------------------------------------------------------------
--
-- PROCEDURE     createSubscriber
--
-- Description:
--     Add an AQ subscriber to the change log queue.
--
-- Parameters:
--     gsm_name:    Subscriber name should be a name of one of the GSMs in
--                  the cloud.      
--
-- Notes:
--    
------------------------------------------------------------------------------- 
PROCEDURE createSubscriber( gsm_name IN varchar2 );


-------------------------------------------------------------------------------
--
-- PROCEDURE     removeSubscriber
--
-- Description:
--     Remove an AQ subscriber to the change log queue.        
--
-- Parameters:
--     gsm_name:    The name used originally to subscribe to the queue.
--                  The name should have been a name of one of the GSMs in
--                  the cloud.      
--
-- Notes:
--    
------------------------------------------------------------------------------- 
PROCEDURE removeSubscriber( gsm_name IN varchar2 );


-------------------------------------------------------------------------------
--
-- PROCEDURE     addRegion
--
-- Description:
--     Adds a region to the cloud.        
--
-- Parameters:
--     region_name:      The name to give to the region.
--     buddy_name:       The name of the buddy region.     
--
-- Notes:
--    
------------------------------------------------------------------------------- 
PROCEDURE addRegion( region_name IN varchar2,
                     buddy_name  IN varchar2 default NULL );


-------------------------------------------------------------------------------
--
-- PROCEDURE     modifyRegion
--
-- Description:
--     Modifies a region.        
--
-- Parameters:
--     region_name:      The name of the region to modify.
--     buddy_name:       The name of a buddy region to assign to the region.
--                       Can be NULL.     
--
-- Notes:
--    
------------------------------------------------------------------------------- 
PROCEDURE modifyRegion( region_name IN varchar2,
                        buddy_name  IN varchar2 );


-------------------------------------------------------------------------------
--
-- PROCEDURE     removeRegion
--
-- Description:
--     Removes a region from the cloud.       
--
-- Parameters:
--     region_name:      The name of the region.      
--
-- Notes:
--     The region should be empty of GSMs and databases.
--
--     The last region in the cloud cannot be removed.
------------------------------------------------------------------------------- 
PROCEDURE removeRegion( region_name IN varchar2 );


-------------------------------------------------------------------------------
--
-- PROCEDURE     addDatabasePool
--
-- Description:
--     Adds a database pool to the cloud.        
--
-- Parameters:
--     database_pool_name:    The name to give to the database pool.      
--
-- Notes:
--    
------------------------------------------------------------------------------- 
PROCEDURE addDatabasePool( database_pool_name IN varchar2 );


-------------------------------------------------------------------------------
--
-- PROCEDURE     removeDatabasePool
--
-- Description:
--     Removes a database pool from the cloud.        
--
-- Parameters:
--     database_pool_name:    The name of the database pool.            
--
-- Notes:
--     The pool should be empty, i.e. it should no longer have any
--     databases or services.
------------------------------------------------------------------------------- 
PROCEDURE removeDatabasePool( database_pool_name IN varchar2 );


-------------------------------------------------------------------------------
--
-- PROCEDURE     removeDatabasePoolAdmin
--
-- Description:
--     Adds an administrator for a database pool.        
--
-- Parameters:
--     database_pool_name:    The name of the database pool. 
--     user_name:             The name of user to become administrator for the
--                            pool.
--
-- Notes:
--     database_pool_name can be NULL if there is only one database pool
--     in the cloud.  In which case the command will default to that pool.
--
--     The user is revoked VPD access to the cloud information about the
--     database pool.
--
--     
-------------------------------------------------------------------------------       
PROCEDURE removeDatabasePoolAdmin( database_pool_name IN varchar2 default NULL, 
                                user_name          IN varchar2 );

-------------------------------------------------------------------------------
--
-- PROCEDURE     addDatabasePoolAdmin
--
-- Description:
--     Adds an administrator for a database pool.        
--
-- Parameters:
--     database_pool_name:    The name of the database pool. 
--     user_name:             The name of user to become administrator for the
--                            pool.
--
-- Notes:
--     database_pool_name can be NULL if there is only one database pool
--     in the cloud.  In which case the command will default to that pool.
--
--     The user is granted VPD access to the cloud information about the
--     database pool.
--
--     TODO: the user is also given "gsm_pooladmin_role".
-------------------------------------------------------------------------------       
PROCEDURE addDatabasePoolAdmin( database_pool_name IN varchar2 default NULL, 
                                user_name          IN varchar2 );


-------------------------------------------------------------------------------
--
-- PROCEDURE     poolVpdPredicate
--
-- Description:
--     Enforces VPD read security for database pool tables.      
--
-- Parameters:
--     Standard VPD function parameters.      
--
-- Notes:
--    
------------------------------------------------------------------------------- 
FUNCTION  poolVpdPredicate( obj_schema varchar2,
                            obj_name   varchar2 ) RETURN varchar2;

-------------------------------------------------------------------------------
--
-- PROCEDURE     MaskPolicy
--
-- Description:
--     Enforces VPD masking for select on database table    
--
-- Parameters:
--     Standard VPD function parameters.      
--
-- Notes:
--    
------------------------------------------------------------------------------- 
FUNCTION MaskPolicy ( obj_schema varchar2, 
                      obj_name   varchar2)  RETURN varchar2;
-------------------------------------------------------------------------------
--
-- FUNCTION     VerifyCatalog
--
-- Description:
--     Perform various cross-check verifications on the catalog data      
--
-- Parameters:
--     NONE    
--
-- Notes:
--    
------------------------------------------------------------------------------- 
FUNCTION VerifyCatalog
RETURN NUMBER;

--*****************************************************************************
-- End of Package Public Procedures
--*****************************************************************************

END dbms_gsm_cloudadmin;

/

show errors



