Rem
Rem $Header: rdbms/admin/dbmsgwmut.sql /main/15 2012/12/06 17:24:27 sdball Exp $
Rem
Rem dbmsgwmut.sql
Rem
Rem Copyright (c) 2011, 2012, Oracle and/or its affiliates. 
Rem All rights reserved. 
Rem
Rem    NAME
Rem      dbmsgwmut.sql - Global Workload Management Utility
Rem
Rem    DESCRIPTION
Rem      Defines the dbms_gsm_utility package that is used for utility
Rem      definitions and procedures used for GSM database cloud management.
Rem
Rem    NOTES
Rem      This package is for definitions and functions shared by the 
Rem      dbms_gsm_pooladmin and dbms_gsm_cloudadmin packages on the GSM
Rem      cloud catalog database, and for utility routines used by GSMCTL
Rem      when administering the cloud.
Rem
Rem       It is only loaded on the cloud catalog database.
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    sdball      11/20/12 - Add err_local_exists and err_in_cloud
Rem    lenovak     11/07/12 - update_svc_state
Rem    cechen      09/25/12 - bug-14576320: add err_need_dbp_name 
Rem    sdball      10/10/12 - add err_db_spfile
Rem    sdball      10/04/12 - add err_bad_retention, err_bad_replay,
Rem                           err_db_incompat, err_svc_stopped
Rem    sdball      09/26/12 - Add err_service_stopped
Rem    sdball      09/21/12 - Add sync_database
Rem    sdball      08/16/12 - add err_svc_relocate
Rem    sdball      07/09/12 - add err_loc_failover
Rem    sdball      06/15/12 - Add err_empty_dbpool
Rem    nbenadja    06/06/12 - Add a new warning.
Rem    sdball      05/18/12 - Add cross check messages.
Rem    sdball      04/09/12 - Code hard limits
Rem    sdball      03/26/12 - Add error err_db_incloud, err_nopref_all
Rem    sdball      03/01/12 - Better privilege errors
Rem    sdball      02/15/12 - New error messages
Rem    sdball      01/10/12 - Add message 45500.
Rem    sdball      12/08/11 - verify input lengths.
Rem    sdball      12/02/11 - error message cleanup
Rem    sdball      11/28/11 - Fix error messages.
Rem    nbenadja    11/23/11 - Add a new error
Rem    sdball      11/03/11 - changes for syncBrokerConfig
Rem    sdball      11/01/11 - Expose reserveNextDBNum for direct call by GSM
Rem    sdball      10/27/11 - Add removebk and remove_bk_ph
Rem    sdball      10/24/11 - Changes for recovery of add service
Rem    lenovak     07/22/11 - vncr support
Rem    mjstewar    04/17/11 - Created
Rem

-- SET ECHO ON
-- SPOOL dbmsgwmut.log

--*****************************************************************************
-- Database package for GSM utility functions and definitions.
--*****************************************************************************

CREATE OR REPLACE PACKAGE dbms_gsm_utility AS


--*****************************************************************************
-- Package Public Types
--*****************************************************************************


--*****************************************************************************
-- Package Public Constants
--*****************************************************************************

-------------------------------------------------------------------------------
-- Default Names
-------------------------------------------------------------------------------
default_cloud_name      constant   varchar2(10) := 'oradbcloud';

-------------------------------------------------------------------------------
-- Identifier lengths
-------------------------------------------------------------------------------

-- Must honor max service name lengths defined by database
max_service_name_len     constant  number := 64;
max_net_service_name_len constant  number := 250;

-- Max number of VNCRs
max_vncr_number constant  number :=300;

-- Max number of services
max_services    constant  number := 1000;

-- Maximum number of database pools
max_dbpools       constant  number := 20;

-------------------------------------------------------------------------------
-- Change Log Queue and generateChangeLogEntry() definitions.
-- These constants are defined for use with generateChangeLogEntry().
-- They are also the values stored in a gsm_change_message.
-------------------------------------------------------------------------------

-- "adminId" values for generateChangeLogEntry().
-- These constants define the package that generated the change entry.
-- Also stored in the the change log queue in gsm_change_message.admin_id.

cloud_admin            constant   number := 1;
pool_admin             constant   number := 2;  


-- "changeId" values for generateChangeLogEntry() when adminId is cloud_admin.
-- "target" of command is always the object name (e.g. GSM name).
--  Also stored in the change log queue in gsm_change_message.change_id.

add_gsm                constant   number := 1;
modify_gsm             constant   number := 2;
drop_gsm               constant   number := 3;
start_gsm              constant   number := 4;
stop_gsm               constant   number := 5;

add_region             constant   number := 10;
modify_region          constant   number := 11;
drop_region            constant   number := 12;

add_database_pool      constant   number := 20;
modify_database_pool   constant   number := 21;
drop_database_pool     constant   number := 22;

-- "changeId" values for generateChangeLogEntry() when adminId is pool_admin.
-- "target" of command is always the object name (e.g. database or service name)
-- Also stored in the change log queue in gsm_change_message.change_id.

add_database           constant   number := 30;
modify_database        constant   number := 31;
drop_database          constant   number := 32;
start_database         constant   number := 33;
stop_database          constant   number := 34;
drop_db_phys           constant   number := 35;
add_broker_config      constant   number := 36;
mod_db_status          constant   number := 37;
add_db_done            constant   number := 38;
sync_database          constant   number := 39;

add_service            constant   number := 50;
modify_service         constant   number := 51;
drop_service           constant   number := 52;
relocate_service       constant   number := 53;
start_service          constant   number := 54;
stop_service           constant   number := 55;
enable_service         constant   number := 56;
disable_service        constant   number := 57;
add_service_to_dbs     constant   number := 58;
move_service_to_db     constant   number := 59;
make_dbs_preferred     constant   number := 60;
modify_service_config  constant   number := 61;
modify_service_on_db   constant   number := 62;
update_service_state   constant   number := 63;
add_vncr               constant   number := 70;
remove_vncr            constant   number := 71;
drop_service_ph        constant   number := 72;
drop_broker_config     constant   number := 73;
drop_bc_phys           constant   number := 74;
sync_broker_config     constant   number := 75;

--*****************************************************************************
-- Package Public Exceptions
--*****************************************************************************


--*****************************************************************************
-- Package Public Procedures
--*****************************************************************************
-------------------------------------------------------------------------------
--
-- PROCEDURE     getCatalogLock
--
-- Description:
--       Gets the catalog lock prior to making a change to the cloud catalog.      
--
-- Parameters:
--       currentChangeSeq -    The current value of cloud.change_seq#
--                             This is the sequence # of the last committed 
--                             change.
--    
------------------------------------------------------------------------------- 

PROCEDURE getCatalogLock( currentChangeSeq OUT number);


-------------------------------------------------------------------------------
--
-- PROCEDURE     releaseCatalogLock
--
-- Description:
--      Releases the catalog lock and commits or rolls back the changes
--      made under the lock.       
--
-- Parameters:
--      action:  "releaseLockCommit" -> release lock and commit all
--                             changes
--               "releaseLockRollback" -> release lock and rollback
--                             all changes
--      changeSeq: If "action" = "releaseLockCommit" this is the change
--                 sequence number of the the last change made under this lock.
--                 If "action" = "releaseLockRollback" then will be set to 0.
--            
--
-- Notes:
--    
------------------------------------------------------------------------------- 

releaseLockCommit           constant  number := 1;
releaseLockRollback         constant  number := 2;


PROCEDURE releaseCatalogLock( action    IN number default releaseLockCommit,
                              changeSeq OUT number );


-------------------------------------------------------------------------------
--
-- FUNCTION     regionExists
--
-- Description:
--    Checks if the specified region exists in the cloud catalog.
--
-- Parameters:
--    region_name:   The region to check.
--    region_num:    If the region exists, returns its number
--
-- Returns:
--    TRUE if the region is in the cloud catalog.
--
-- Notes:
--    
------------------------------------------------------------------------------- 
FUNCTION regionExists( region_name IN  varchar2,
                       region_num  OUT number )
  RETURN boolean;

-------------------------------------------------------------------------------
--
-- FUNCTION     databasePoolExists
--
-- Description:
--   Checks if the specified database pool exists in the cloud catalog.
--
-- Parameters:
--   database_pool_name:  The pool to check.
--
-- Returns:
--   TRUE if the database pool is in the cloud catalog.
--
-- Notes:
--    
------------------------------------------------------------------------------- 
FUNCTION databasePoolExists( database_pool_name IN varchar2 )
  RETURN boolean;

-------------------------------------------------------------------------------
--
-- FUNCTION     prepareName
--
-- Description:
--       Verifies the length of a (service, GSM, etc) name and prepares 
--       it for use by the GSM package.
--
--       Trims off leading and trailing spaces and converts it to lower
--       case.        
--
-- Parameters:
--       in_name:     The name to check and and prepare for use.
--       out_name:    The prepared name.
--       max_length:  The maximum allowable length for out_name.
-- 
-- Returns:
--    TRUE if the name is the correct length.
--    FALSE otherwise (out_name will not be set).           
--
-- Notes:
--   Names of GSM objects (services, GSMs, regions, etc) are stored in the
--   catalog in lower case.
--    
-------------------------------------------------------------------------------
FUNCTION prepareName( in_name     IN  varchar2,
                      out_name    OUT varchar2,
                      max_length  IN  number )
  RETURN boolean;

-------------------------------------------------------------------------------
--
-- FUNCTION     prepareRegionName
--
-- Description:
--     Returns the region name to use on a dbms_gsm_* call when the region
--     name in the routine call can be NULL.
--
--     If the supplied name is NULL will determine if a default region name
--     can be used and returns it.  A default only exists if a single
--     region has been defined.  The default is that region.  If more than one
--     region has been defined, then a default cannot be picked and FALSE is
--     returned.
--
--     If the supplied name is not NULL, then verifies that it is the right
--     length and prepares it for use: trims off leading and trailing spaces
--     and converts to upper case.
--          
--
-- Parameters:
--       input_name  (INPUT)  - the supplied name (can be NULL)
--       region_name (OUTPUT) - the region name to use
--
-- Returns:
--       TRUE if a valid region name can be returned
--       FALSE if not
--
-- Notes:
--    
------------------------------------------------------------------------------- 
FUNCTION prepareRegionName( input_name IN varchar2,
                            region_name OUT varchar2 )
  RETURN boolean;

-------------------------------------------------------------------------------
--
-- FUNCTION     prepareDBPoolName
--
-- Description:
--     Returns the database pool name to use on a dbms_gsm_* call when the
--     database pool name in the routine call can be NULL.
--
--     If the supplied name is NULL will determine if a default database pool
--     name can be used and returns it.  A default only exists if a single
--     pool has been defined.  The default is that pool.  If more than one
--     pool has been defined, then a default cannot be picked and FALSE is
--     returned.
--
--     If the supplied name is not NULL, then verifies that it is the right
--     length and prepares it for use: trims off leading and trailing spaces
--     and converts to upper case.
--          
--
-- Parameters:
--       input_name         (INPUT)  - the supplied name (can be NULL)
--       database_pool_name (OUTPUT) - the database pool name to use
--
-- Returns:
--       TRUE if a valid database pool name can be returned
--       FALSE if not
--
-- Notes:
--    
------------------------------------------------------------------------------- 
FUNCTION prepareDBPoolName( input_name         IN  varchar2,
                            database_pool_name OUT varchar2 )
  RETURN boolean;


-------------------------------------------------------------------------------
--
-- PROCEDURE     generateChangeLogEntry
--
-- Description:
--       Generates a description of a change and puts it into the
--       change log queue.     
--
-- Parameters:
--       adminId:   package that is calling this routine
--                  "cloud_admin" - dbms_gsm_cloudadmin package
--                  "pool_admin"  - dbms_gsm_pooladmin package
--       changeId:  the change made (see constant definitions above)
--       target:    command target (e.g. gsm name for "add gsm")
--       poolName:  database pool (only if adminId = pool_admin)
--       params:    additional parameters for the change
--       updateRequestTable: whether or not to also put the change into
--                   gsm_requests table
--            
--
-- Notes:
--   See constant definitions above for "adminId" and "changeId" parameters.
--    
-------------------------------------------------------------------------------

updateFalse  constant  number := 0;
updateTrue   constant  number := 1;

PROCEDURE generateChangeLogEntry( adminId             IN number,
                                  changeId            IN number,
                                  target              IN varchar2,
                                  poolName            IN varchar2 default NULL,
                                  params              IN varchar2 default NULL, 
                                  updateRequestTable  IN number default updateTrue );


-------------------------------------------------------------------------------
--
-- PROCEDURE     removeStaleRequests
--
-- Description:
--       Removes stale entries from gsm_requests
--
-- Parameters:
--       age:    IN    Requests older than this are removed
--
--
-- Notes:
------------------------------------------------------------------------------- 
PROCEDURE removeStaleRequests; --(age IN INTERVAL DAY TO SECOND 
--                                          default '10 minutes');

------------------------------------------------------------------------------
--
-- PROCEDURE isLockedByMaster
--
-- Description:
--      Determines if master locak is already taken
--
-- Parameters:
--      None
--
-- Returns:
--      0 - Master lock is not taken
--      1 - Master Lock is taken
--
------------------------------------------------------------------------------
FUNCTION isLockedByMaster
  RETURN integer;

-------------------------------------------------------------------------------
--
-- FUNCTION     maxDBInstances
--
-- Description:
--       Queries the database parameter setting for the maximum number of
--       instances to reserve for a cloud database.
--
-- Parameters:
--       None
--
-- Returns:
--       The maximum.
--
-- Notes:
--    
-------------------------------------------------------------------------------
FUNCTION maxDBInstances
  RETURN number;

------------------------------------------------------------------------------
--
-- PROCEDURE reserveInstNums
--
-- Description:
--    Reserve reserve_count database numbers from cloud
--
-- Notes:
--   Only useful for PL/SQL calls, current value should be known already
--   otherwise this function has no good use.
--
------------------------------------------------------------------------------
PROCEDURE reserveInstNums( reserve_count IN number );

------------------------------------------------------------------------------
--
-- FUNCTION reserveNextDBNum
--
-- Description:
--   Reserves the next available DB number for use.
--
-- Returns:
--   The next available number
--
-- Notes:
--   This function updates the cloud table, but does not commit. The caller
--   is expected to commit (if the number is used), or rollback (if the 
--   number can be re-used). An update lock on the cloud table will be held
--   until the caller either commits or rolls back.
--
------------------------------------------------------------------------------
FUNCTION reserveNextDBNum( reserve_count   IN number default NULL)
  RETURN number;

------------------------------------------------------------------------------
--
-- FUNCTION getFieldSize
--
-- Description:
--   gets the size of a database field
--
-- Returns:
--   size
--
-- Notes:
--   Used internally by PL/SQL to verify the size of input strings
--
------------------------------------------------------------------------------
FUNCTION getFieldSize( tab_name   IN varchar2,
                       col_name   IN varchar2)
  RETURN number;
--*****************************************************************************
-- End of Package Public Procedures
--*****************************************************************************

  -------------------------
  --  ERRORS AND EXCEPTIONS
  --
  --  When adding errors remember to add a corresponding exception below.
err_generic_gsm      constant number := -44850;
err_bad_db_name      constant number := -44851;
err_region_max       constant number := -44852;
err_vncr_max         constant number := -44853;
err_exist_cld        constant number := -44854;
err_invalid_cld      constant number := -44855;
err_invalid_cldsvc   constant number := -44856;
err_nfound_cld       constant number := -44857;
err_remove_cld       constant number := -44858;
err_exist_GSM        constant number := -44859;
err_nfound_region    constant number := -44860;
err_remove_vncr      constant number := -44861;
err_buddy_region     constant number := -44862;
err_last_region      constant number := -44863;
err_remove_rgn_gsm   constant number := -44864;
err_remove_pool      constant number := -44865;
err_non_broker       constant number := -44866;
err_already_in_pool  constant number := -44867;
err_is_broker        constant number := -44868;
err_net_name         constant number := -44869;
err_svc_non_bc       constant number := -44870;
err_svc_non_pa       constant number := -44871;
err_db_same          constant number := -44872;
err_db_offer         constant number := -44873;
err_db_not_offer     constant number := -44874;
err_invalid_param    constant number := -44875;
err_svc_is_rng       constant number := -44876;
err_svc_is_dis       constant number := -44877;
err_svc_is_lag       constant number := -44878;
err_no_region        constant number := -44879;
err_no_cld           constant number := -44880;
err_nonempty_pool    constant number := -44881;
err_bad_dbp_name     constant number := -44882;
err_bad_region_name  constant number := -44883;
err_bad_svc_name     constant number := -44884;
err_bad_vncr_name    constant number := -44885;
err_bad_vncrgrp_name constant number := -44886;
err_bad_gsm_name     constant number := -44887;
err_bad_gsmu_name    constant number := -44888;
err_exist_region     constant number := -44889;
err_exist_dbpool     constant number := -44890;
err_nfound_gsm       constant number := -44891;
err_nfound_dbpool    constant number := -44892;
err_nfound_database  constant number := -44893;
err_nfound_service   constant number := -44894;
err_remove_rgn_db    constant number := -44895;
err_svc_in_pool      constant number := -44896;
err_svc_lag          constant number := -44897;
err_svc_failover     constant number := -44898;
err_exist_vncr       constant number := -44899;
err_string_size      constant number := -44900;
err_rem_db           constant number := -45500;
err_max_gsm          constant number := -45501;
err_no_priv          constant number := -45502;
err_db_incloud       constant number := -45503;
err_nopref_all       constant number := -45504;
err_max_service      constant number := -45505;
err_max_pools        constant number := -45506;
-- information messages for VerifyCatalog
err_no_buddy         constant number := -45507;
err_no_dbregion      constant number := -45508;
err_bad_dbstatus     constant number := -45509;
err_gsm_request      constant number := -45510;
err_no_preferred     constant number := -45511;
err_no_lcl_pref      constant number := -45512;
err_no_service       constant number := -45513;
err_no_dbreg         constant number := -45514;
err_no_gsm_reg       constant number := -45515;
err_pool_db          constant number := -45516;
err_pool_svc         constant number := -45517;
-- end information messages for VerifyCatalog
err_lag_lgsdby       constant number := -45518;
err_empty_dbpool     constant number := -45519;
err_loc_failover     constant number := -45520;
err_role_failover    constant number := -45521;
err_svc_relocate     constant number := -45522;
err_service_stopped  constant number := -45523;
err_need_dbp_name    constant number := -45524;
err_bad_retention    constant number := -45525;
err_bad_replay       constant number := -45526;
err_db_incompat      constant number := -45527;
err_svc_stopped      constant number := -45528;
err_db_spfile        constant number := -45529;
err_local_exists     constant number := -45530;
err_in_cloud         constant number := -45531;
err_is_cat           constant number := -45532;


generic_gsm       EXCEPTION;
PRAGMA EXCEPTION_INIT(generic_gsm,      -44850); 
bad_db_name      EXCEPTION;
PRAGMA EXCEPTION_INIT(bad_db_name,      -44851);
region_max        EXCEPTION;
PRAGMA EXCEPTION_INIT(region_max,       -44852);
vncr_max          EXCEPTION;
PRAGMA EXCEPTION_INIT(vncr_max,         -44853);
exist_cld         EXCEPTION;
PRAGMA EXCEPTION_INIT(exist_cld,        -44854);
invalid_cld       EXCEPTION;
PRAGMA EXCEPTION_INIT(invalid_cld,      -44855);
invalid_cldsvc    EXCEPTION;
PRAGMA EXCEPTION_INIT(invalid_cldsvc,   -44856);
nfound_cld        EXCEPTION;
PRAGMA EXCEPTION_INIT(nfound_cld,       -44857);
remove_cld        EXCEPTION;
PRAGMA EXCEPTION_INIT(remove_cld,       -44858);
exist_GSM      EXCEPTION;
PRAGMA EXCEPTION_INIT(exist_GSM   ,     -44859);
nfound_region     EXCEPTION;
PRAGMA EXCEPTION_INIT(nfound_region,    -44860);
exremove_vncr     EXCEPTION;
PRAGMA EXCEPTION_INIT(exremove_vncr,    -44861);
buddy_region      EXCEPTION;
PRAGMA EXCEPTION_INIT(buddy_region,     -44862);
last_region       EXCEPTION;
PRAGMA EXCEPTION_INIT(last_region,      -44863);
remove_rgn_gsm        EXCEPTION;
PRAGMA EXCEPTION_INIT(remove_rgn_gsm,   -44864);
remove_pool       EXCEPTION;
PRAGMA EXCEPTION_INIT(remove_pool,      -44865);
non_broker        EXCEPTION;
PRAGMA EXCEPTION_INIT(non_broker,       -44866);
already_in_pool   EXCEPTION;
PRAGMA EXCEPTION_INIT(already_in_pool,  -44867);
is_broker         EXCEPTION;
PRAGMA EXCEPTION_INIT(is_broker,        -44868);
net_name          EXCEPTION;
PRAGMA EXCEPTION_INIT(net_name,         -44869);
svc_non_bc        EXCEPTION;
PRAGMA EXCEPTION_INIT(svc_non_bc,       -44870);
svc_non_pa        EXCEPTION;
PRAGMA EXCEPTION_INIT(svc_non_pa,       -44871);
db_same           EXCEPTION;
PRAGMA EXCEPTION_INIT(db_same,          -44872);
db_offer          EXCEPTION;
PRAGMA EXCEPTION_INIT(db_offer,         -44873);
db_not_offer      EXCEPTION;
PRAGMA EXCEPTION_INIT(db_not_offer,     -44874);
invalid_param     EXCEPTION;
PRAGMA EXCEPTION_INIT(invalid_param,    -44875);
svc_is_rng        EXCEPTION;
PRAGMA EXCEPTION_INIT(svc_is_rng,       -44876);
svc_is_dis        EXCEPTION;
PRAGMA EXCEPTION_INIT(svc_is_dis,       -44877);
svc_is_lag        EXCEPTION;
PRAGMA EXCEPTION_INIT(svc_is_lag,       -44878);
no_region         EXCEPTION;
PRAGMA EXCEPTION_INIT(no_region,        -44879);
no_cld         EXCEPTION;
PRAGMA EXCEPTION_INIT(no_cld   ,        -44880);
nonempty_pool         EXCEPTION;
PRAGMA EXCEPTION_INIT(nonempty_pool ,   -44881);
bad_dbp_name         EXCEPTION;
PRAGMA EXCEPTION_INIT(bad_dbp_name  ,   -44882);
bad_region_name         EXCEPTION;
PRAGMA EXCEPTION_INIT(bad_region_name , -44883);
bad_svc_name         EXCEPTION;
PRAGMA EXCEPTION_INIT(bad_svc_name  ,   -44884);
bad_vncr_name         EXCEPTION;
PRAGMA EXCEPTION_INIT(bad_vncr_name  ,  -44885);
bad_vncrgrp_name         EXCEPTION;
PRAGMA EXCEPTION_INIT(bad_vncrgrp_name, -44886);
bad_gsm_name         EXCEPTION;
PRAGMA EXCEPTION_INIT(bad_gsm_name  ,   -44887);
bad_gsmu_name         EXCEPTION;
PRAGMA EXCEPTION_INIT(bad_gsmu_name  ,  -44888);
exist_region         EXCEPTION;
PRAGMA EXCEPTION_INIT(exist_region   ,  -44889);
exist_dbpool         EXCEPTION;
PRAGMA EXCEPTION_INIT(exist_dbpool   ,  -44890);
nfound_gsm         EXCEPTION;
PRAGMA EXCEPTION_INIT(nfound_gsm     ,  -44891);
nfound_dbpool         EXCEPTION;
PRAGMA EXCEPTION_INIT(nfound_dbpool  ,  -44892);
nfound_database         EXCEPTION;
PRAGMA EXCEPTION_INIT(nfound_database,  -44893);
nfound_service        EXCEPTION;
PRAGMA EXCEPTION_INIT(nfound_service ,  -44894);
remove_rgn_db            EXCEPTION;
PRAGMA EXCEPTION_INIT(remove_rgn_db ,   -44895);
svc_in_pool            EXCEPTION;
PRAGMA EXCEPTION_INIT(svc_in_pool   ,   -44896);
svc_lag            EXCEPTION;
PRAGMA EXCEPTION_INIT(svc_lag       ,   -44897);
svc_failover            EXCEPTION;
PRAGMA EXCEPTION_INIT(svc_failover  ,   -44898);
exist_vncr            EXCEPTION;
PRAGMA EXCEPTION_INIT(exist_vncr    ,   -44899);
string_size            EXCEPTION;
PRAGMA EXCEPTION_INIT(string_size    ,  -44900);
rem_db            EXCEPTION;
PRAGMA EXCEPTION_INIT(rem_db    ,       -45500);
max_gsm            EXCEPTION;
PRAGMA EXCEPTION_INIT(max_gsm    ,      -45501);
no_priv            EXCEPTION;
PRAGMA EXCEPTION_INIT(no_priv    ,      -45502);
db_incloud            EXCEPTION;
PRAGMA EXCEPTION_INIT(db_incloud    ,   -45503);
nopref_all            EXCEPTION;
PRAGMA EXCEPTION_INIT(nopref_all    ,   -45504);
max_service            EXCEPTION;
PRAGMA EXCEPTION_INIT(max_service    ,  -45505);
max_pools          EXCEPTION;
PRAGMA EXCEPTION_INIT(max_pools    ,  -45506);
empty_dbpool          EXCEPTION;
PRAGMA EXCEPTION_INIT(empty_dbpool    ,  -45519);
loc_failover          EXCEPTION;
PRAGMA EXCEPTION_INIT(loc_failover    ,  -45520);
role_failover          EXCEPTION;
PRAGMA EXCEPTION_INIT(role_failover    ,  -45521);
svc_relocate          EXCEPTION;
PRAGMA EXCEPTION_INIT(svc_relocate    ,  -45522);
svc_stopped          EXCEPTION;
PRAGMA EXCEPTION_INIT(svc_stopped    ,  -45523);
need_dbp_name         EXCEPTION;
PRAGMA EXCEPTION_INIT(need_dbp_name   ,  -45524);
bad_retention          EXCEPTION;
PRAGMA EXCEPTION_INIT(bad_retention    ,  -45525);
bad_replay        EXCEPTION;
PRAGMA EXCEPTION_INIT(bad_replay    ,  -45526);
db_incompat        EXCEPTION;
PRAGMA EXCEPTION_INIT(db_incompat    ,  -45527);
svc_stopped        EXCEPTION;
PRAGMA EXCEPTION_INIT(svc_stopped    ,  -45528);
db_spfile        EXCEPTION;
PRAGMA EXCEPTION_INIT(db_spfile    ,  -45529);
local_exists        EXCEPTION;
PRAGMA EXCEPTION_INIT(local_exists    ,  -45530);
in_cloud        EXCEPTION;
PRAGMA EXCEPTION_INIT(in_cloud    ,  -45531);
is_cat        EXCEPTION;
PRAGMA EXCEPTION_INIT(is_cat    ,  -45532);


END dbms_gsm_utility;

/

show errors

--*****************************************************************************
-- Database package for functions that can be executed without GSM privileges
--*****************************************************************************

CREATE OR REPLACE PACKAGE dbms_gsm_nopriv AS

--*****************************************************************************
-- NOTE: This package is executeable by public. We *MUST* ensure that the
-- calling user has the correct catalog privileges at the start of every 
-- procedure before executing any other code with the package.
--*****************************************************************************


--*****************************************************************************
-- Package Public Types
--*****************************************************************************


--*****************************************************************************
-- Package Public Constants
--*****************************************************************************

--*****************************************************************************
-- Package Public Exceptions
--*****************************************************************************


--*****************************************************************************
-- Package Public Procedures
--*****************************************************************************
-------------------------------------------------------------------------------
--
-- PROCEDURE     getCatalogLock
--
-- Description:
--       Gets the catalog lock prior to making a change to the cloud catalog.      
--
-- Parameters:
--       currentChangeSeq -    The current value of cloud.change_seq#
--                             This is the sequence # of the last committed 
--                             change.
--       privs                 Privilege required for this lock operation
--
-- Notes:
--       WARNING: This function is executabble by "public" and runs with
--       gsmadmin_internal privileges. It *MUST* check that the real calling
--       session use has the privilege to peform catalog operations first
--       (before anything else is done). The "privs" and "pool_name"
--       parameters provide the require privileges for the current "lock"
--       operation. The sequence of events is that the user "locks" the catalog,
--       performs the desired operation, and then unlocks the catalog
--    
------------------------------------------------------------------------------- 

GSMAdmin                    constant    number := 1;
GSMPoolAdmin                constant    number := 2;

PROCEDURE getCatalogLock( currentChangeSeq OUT number,
                          privs IN number default GSMAdmin);


-------------------------------------------------------------------------------
--
-- PROCEDURE     releaseCatalogLock
--
-- Description:
--      Releases the catalog lock and commits or rolls back the changes
--      made under the lock.       
--
-- Parameters:
--      action:  "releaseLockCommit" -> release lock and commit all
--                             changes
--               "releaseLockRollback" -> release lock and rollback
--                             all changes
--      changeSeq: If "action" = "releaseLockCommit" this is the change
--                 sequence number of the the last change made under this lock.
--                 If "action" = "releaseLockRollback" then will be set to 0.
--            
--
-- Notes:
--    
------------------------------------------------------------------------------- 

releaseLockCommit           constant  number := 1;
releaseLockRollback         constant  number := 2;


PROCEDURE releaseCatalogLock( action    IN number default releaseLockCommit,
                              changeSeq OUT number );


END dbms_gsm_nopriv;

/

show errors
