Rem
Rem $Header: rdbms/admin/utlucdir.sql /main/2 2012/11/09 14:05:32 bmccarth Exp $
Rem
Rem utlucdir.sql
Rem
Rem Copyright (c) 2012, Oracle and/or its affiliates. All rights reserved. 
Rem
Rem    NAME
Rem      utlucdir.sql - Create directories on disk for Upgrade use
Rem
Rem    DESCRIPTION
Rem      Used by upgrade process to ensure that a directory used to create
Rem      a pfile exists.
Rem
Rem    NOTES
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    bmccarth    11/07/12 - LRG 8481168
Rem    bmccarth    10/23/12 - Create Directory Script for Upgrade process
Rem    bmccarth    10/23/12 - Created
Rem

Rem
Rem Used by HOST command after anonymous pl/sql block attemps to use 
Rem JAVA to create the directory
Rem Use VARIABLE to get info from pl/sql block back out to 
Rem SQL.
Rem
VARIABLE  createDirCmd VARCHAR2(500);
VARIABLE  errtext      VARCHAR2(500);

DECLARE
  javaCreate  VARCHAR2(500) :=
    'CREATE OR REPLACE FUNCTION Upgrdcreatedir(path VARCHAR2)
       RETURN NUMBER AS
       LANGUAGE JAVA
       NAME ''Upgrdcreatedir.create (java.lang.String) return java.lang.int'';';
  dummyCreate   VARCHAR2(500) :=
    'CREATE OR REPLACE FUNCTION Upgrdcreatedir(path VARCHAR2)
       RETURN NUMBER AS
     BEGIN
       RETURN 0;
     END Upgrdcreatedir;';
  dummy       VARCHAR2(2500);
  platform    v$database.platform_name%TYPE;
  dirPath      VARCHAR2(4000);  -- the physical directory
  cdirpath    VARCHAR2(4000);  -- This is dirPath without any trailing slashes
  clearJava   BOOLEAN;
  status      NUMBER   := 0;
  javaOK      NUMBER;
  createString  VARCHAR2(500);

BEGIN

  --
  -- host command needs something to do since it is executed in all cases
  --
  :createDirCmd := 'exit';

  --
  -- Figure out what directory we want to attempt to create
  -- the path returned includes a trailing '/' (or '\') which 
  -- is dealt with below.
  --
  dirPath := dbms_registry_sys.get_pfile_path ('upgrade');

  IF (length(dirPath) = 0 ) THEN
    :errtext := 'ERROR: Could not determine directory path';
  ELSE
    --
    -- got back a directory to try
    --
    EXECUTE IMMEDIATE 'SELECT NLS_UPPER(platform_name) FROM v$database'
       INTO platform;

    IF INSTR(platform, 'WINDOWS') != 0 THEN
      cdirpath := RTRIM(dirPath, '\');
      :createDirCmd := 'mkdir ' || cdirpath;
    ELSIF INSTR(platform, 'VMS') != 0 THEN
      -- Nothing special for VMS 
      cdirpath := dirPath;
      :createDirCmd := 'CREATE/DIR ' || cdirpath;
    ELSE
      cdirpath := RTRIM (dirPath, '/');
      :createDirCmd := 'mkdir -p ' || cdirpath;
    END IF;
    --
    -- Load in the Java piece
    --
    javaOK := 0;

    IF dbms_registry.is_valid('JAVAVM',dbms_registry.release_version) = 1 THEN
      DECLARE
        classInUse  EXCEPTION;
        PRAGMA EXCEPTION_INIT(classInUse, -29553);

      BEGIN
        EXECUTE IMMEDIATE '
          CREATE OR REPLACE AND COMPILE JAVA SOURCE NAMED "Upgrdcreatedir" AS
            import java.lang.*;
            import java.util.*;
            import java.io.*;

            public class Upgrdcreatedir
            {
              public static int create (String path)
              {
                File myFile = new File(path);
                if (myFile.exists())
                {
                  if (myFile.canWrite())
                    return 1;  /* Directory exists and is writable, OK */
                  else
                    return 2;  /* Directory exists and is not writable, NOT OK */
                }
                else
                {
                  if (myFile.mkdirs())
                    return 1;  /* Directory created, OK */
                  else
                    return 4;  /* Directory could not be created, NOT OK */
                }
              }
            }';
          javaOK := 1;
          EXCEPTION
            WHEN classInUse THEN javaOK := 1;  -- already created
            WHEN OTHERS THEN     javaOK := 0;
      END;
    
      IF javaOK = 1 THEN
        createString := javaCreate;
      ELSE
        --  Create dummy version if the java create failed
        createString := dummyCreate;      
      END IF;
    ELSE
      -- JavaVM is not present or in an invalid state
      --
      -- Create dummy version of Upgrdcreatedir so subsequent
      -- blocks will compile
      createString := dummyCreate;
    END IF;
    --
    -- Execute command to create the java package (or the dummy one)
    --
    BEGIN
      EXECUTE IMMEDIATE createString;
    EXCEPTION 
      WHEN OTHERS THEN javaOK := 0;
    END;

   IF javaOK = 1 THEN
      --
      -- Java is around, try and call the new package to create the 
      -- directory.
      --
      DECLARE
        --
        -- This exception is common and is handled by ending the session
        -- inside the loop.
        --
        JavaSessionCleared EXCEPTION;
        PRAGMA EXCEPTION_INIT(JavaSessionCleared, -29549);
      BEGIN
        clearJava := FALSE;

        FOR tries IN 1..2 LOOP
          BEGIN
            EXECUTE IMMEDIATE 'CALL Upgrdcreatedir(:1) into :2'
            USING IN cdirpath, OUT status;
          EXCEPTION
            WHEN JavaSessionCleared THEN  clearJava := TRUE;
            WHEN OTHERS THEN NULL;
          END;

          IF clearJava THEN 
            -- Clear state and try again
            -- Use dynamic SQL since dbms_java may not be installed
            BEGIN
              EXECUTE IMMEDIATE 'BEGIN :1 := dbms_java.endsession; END;'
              USING OUT dummy;
            EXCEPTION WHEN OTHERS THEN NULL;
            END;
          ELSIF status = 1 THEN
            -- Success, exit loop
            EXIT;
          ELSE
            EXIT;   -- loop
          END IF;
        END LOOP;
      END;
      --
      -- done with the java, clean it up
      --
      BEGIN
        EXECUTE IMMEDIATE 'DROP JAVA SOURCE "Upgrdcreatedir"';
        EXCEPTION WHEN OTHERS THEN NULL;
      END;
      --
      -- wipe out the java session
      --
      BEGIN
        EXECUTE IMMEDIATE 'BEGIN :1 := dbms_java.endsession; END;'
        USING OUT dummy;
        EXCEPTION WHEN OTHERS THEN NULL;
      END;
    END IF; -- Javaok check

    IF status != 1 THEN
      --
      -- something went wrong with the java (or it was not even executed)
      --
      IF status = 2 THEN
        -- A two means we can't write to the area.
        :errtext := 'Unable to write to  directory:' || dirPath || '.';
      ELSIF status = 4 THEN
        -- Only other is a 4 which means we couldn't create the dir
        :errtext := 'Unable to create directory:' || dirPath || '.';
      ELSE
        -- 0 ?  Could not verify/create because of JAVA
        :errtext := 'Unable to verify and/or create directory: ' || dirPath 
              || '. Verify JAVA is installed and valid.';
      END IF;
    END IF;
  END IF;  -- dirPath coming back as non-zero
END;
/


Rem
Rem if there is some error generated by the JAVA 
Rem code above, display it using select (in case
Rem set serveroutput is off).
Rem If the issue is JAVA, the host command may create 
Rem the directory.
Rem
DOC

  Any error from the Directory creation are listed below, or 
  the message  'no rows selected' is displayed.

  In the case of a JAVA error, the directory may still be created
  using an operating system specific command.

#
SELECT :errtext AS Error_Text from sys.dual where (length(:errtext) <> 0);

Rem
Rem The above block attempts to create the directory using JAVA
Rem and if it succeeds, the below command ends up being 'exit'
Rem otherwise, its a mkdir or mkdir -p or create/dir which host
Rem will use to attempt to create the directory.
Rem
Rem As mentioned in catbundle.sql, the problem with this method is
Rem the error checking is non-existant.  Errors will end up 
Rem going to the screen or spooled file.
Rem 

COLUMN create_cmd NEW_VALUE create_cmd NOPRINT
SELECT :createDirCmd AS create_cmd FROM SYS.dual;
HOST &create_cmd


Rem
Rem done
Rem