#
# Copyright (c) 2007, 2013, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#       crsdeconfig.pm
#
#    DESCRIPTION
#       Root deconfig script for Oracle Clusterware.
#
#    MODIFIED   (MM/DD/YY)
#    sidshank    03/15/13 - unset ORACLE_HOME environment variable before
#                           OCRDowngrade
#    sidshank    03/04/13 - fix bug 16382128.
#    sidshank    02/24/13 - fix bug 16382128.
#    sidshank    11/15/12 - fix bug 15843792
#    ssprasad    11/13/12 - Disable OKA actions for 12.1
#    xyuan       11/06/12 - Fix bug 14404372
#    xyuan       10/11/12 - Fix bug 14501724 - remove all 12.1-related stuff
#                           from /etc/oratab during downgrade
#    rdasari     10/10/12 - use force to stop excl mode stack
#    xyuan       10/10/12 - Fix bug 14634138
#    rdasari     09/12/12 - disable remote asm during last node deconfig
#    xyuan       08/30/12 - Fix bug 14528838
#    xyuan       08/30/12 - Fix bug 14528781
#    xyuan       08/20/12 - Fix bug 14000167
#    rdasari     08/16/12 - mention ocr backup location in mesg 414
#    xyuan       08/02/12 - Fix bug 14404483
#    rdasari     08/01/12 - stop oc4j before removing the resource
#    shmubeen    07/19/12 - add uninstall of afd
#    xyuan       07/01/12 - Fix bug 14241008
#    shullur     06/21/12 - For ignoring downgrade path failures in case of ora
#                           file
#    xyuan       06/16/12 - Fix bug 14183602
#    shullur     06/15/12 - For checking if bdb location is default
#    rdasari     05/25/12 - remove KA
#    shullur     05/09/12 - For removing exit if no previous CHM data is not
#                           found
#    rtamezd     05/11/12 - Fix bug 14061578
#    sidshank    05/03/12 - remove s_redirect/restore output subroutines
#    rtamezd     03/20/12 - Fix bug 13867352
#    rtamezd     03/12/12 - Fix bug 13700977
#    anjiwaji    02/15/12 - Use chkpoints in removeACFSRoot if not in SIHA.
#    xyuan       02/01/12 - XbranchMerge rdasari_bug-12614809 from
#                           st_has_11.2.0.3.0
#    gsanders    01/30/12 - remove call to removeACFSRegistry()
#    rdasari     01/19/11 - remove OCR on DG in deconfigure_ASM if DG is reused
#    sidshank    01/18/12 - fix for bug 13537946
#    xyuan       01/08/12 - Add 12c to 11.2 downgrade code
#    sidshank    12/15/11 - disabling call to remove_misc_dirs on windows
#    sidshank    11/09/11 - Fix for the bug 13352502
#    shullur     11/04/11 - XbranchMerge shullur_bug_12639640 from
#                           st_has_11.2.0
#    shullur     11/04/11 - XbranchMerge shullur_bug_11852891 from
#                           st_has_11.2.0
#    xesquive    09/19/11 - forward fix from bug-12587677 to main
#    xyuan       07/27/11 - XbranchMerge xyuan_bug-12585291 from
#                           st_has_11.2.0.3.0
#    xyuan       07/27/11 - XbranchMerge xyuan_bug-12727247 from
#    xesquive    07/26/11 - Add functions set_bold and reset_bold instead of
#                           print color
#    dpham       03/31/11 - Creation for 12c 
#
package crsdeconfig;

use strict;
use English;
use File::Copy;
use File::Path;
use File::Spec::Functions;

# root scripts modules
use crsutils;
use crsgpnp;
use oracss;
use oraacfs;
use crska;
use oraafd;
use s_crsutils;

    
my @ns_files = ("CSS","CRS","EVM","PROC","css","crs","evm","proc");

sub new {
   shift;
   crsutils->new(@_);

   # deconfig / downgrade
   if ($CFG->SIHA) {
      HADeconfigure();
   }
   else {
      CRSDeconfig();
   }

}

sub CRSDeconfig
{
   if (! $CFG->DOWNGRADE) { 
      trace ("Deconfiguring Oracle Clusterware on this node");
      prechecksDeconfig();
      remove_checkpoints();
   } else {
      trace ("Downgrading Oracle Clusterware on this node");
      downgrade_GetOldConfig();
   }

   TraceOptions();

   crsDowngrade();

   my @votedisk_list = ();

   if ($CFG->LASTNODE) {
      # Extract voting disks
      @votedisk_list = extractVotedisks();
   }

   if (! $CFG->DOWNGRADE) {
      my $status = VerifyResources();
      if ($status eq FAILED) {
         die(dieformat(311));
      }

      # Only remove resources which were created during 
      # Oracle Clusterware configuration
      RemoveResources ();
   }

   DeconfigStorage();

   #Downgrade the CHM
   if ($CFG->DOWNGRADE) {
      CHMDowngrade();
   }

   #Remove CHM repository
   removeCHMDB();

   s_RemoveInitResources();

   if ($CFG->LASTNODE) {
      s_ResetVotedisks(@votedisk_list);
   }

   if (! $CFG->DOWNGRADE) {
      crsDelete_ResetOCR();
   }

   #Remove olr.loc in case of downgrade
   if ($CFG->DOWNGRADE) {
      s_ResetOLR ();
   }

   if ($CFG->DOWNGRADE && $CFG->LASTNODE && ocrDowngrade()) {
      $CFG->CSS_downgrade_VFs();
   }

   DeleteSCR();

   s_NSCleanUp();
   
   # Temporarely comment out removeGPnPprofile()
   # removeGPnPprofile();

   if ($CFG->platform_family eq 'windows') {
      s_houseCleaning();
   }

   if ($CFG->platform_family ne 'windows') {
      remove_misc_dirs();
   }

   s_CleanTempFiles();

   remove_checkpoints();

   # reset the permissions only when deconfig is called from deinstall or
   # the -deinstall option is specified during de-configuration
   if (($CFG->defined_param('HOME_TYPE')) || ($CFG->DEINSTALL)) {
      trace ("Opening permissions on Oracle clusterware home");
      s_reset_crshome($CFG->params('ORACLE_OWNER'),
                      $CFG->params('ORA_DBA_GROUP'),
                      755, $CFG->params('ORACLE_HOME'));

      trace("Reset Parent dir permissions for Oracle clusterware home");
      s_resetParentDirOwner($CFG->ORA_CRS_HOME);
   }

   # reset permission on OCR locations
   #TODO - move this to s_ResetOCR so that we need not rely
   #on OCR_LOCATIONS
   trace ("reset permissions on OCR locations");
   if (! $CFG->ASM_STORAGE_USED) {
      if ($CFG->defined_param('OCR_LOCATIONS')) {
         my @ocr_locs = split (/\s*,\s*/, $CFG->params('OCR_LOCATIONS'));
         foreach my $loc (@ocr_locs) {
            # reset owner/group of OCR path
            trace ("set owner/group of OCR path");
            s_reset_crshome($CFG->params('ORACLE_OWNER'),
                            $CFG->params('ORA_DBA_GROUP'), 755, $loc);
         }
      }
   }

   if ($CFG->platform_family eq "unix") {
      s_removeCvuRpm();
   }

   if (($CFG->DOWNGRADE) && ($CFG->platform_family eq 'unix'))
   {
     # Remove /etc/oracle/setasmgid
     my $file = catfile($CFG->params('OCRCONFIGDIR'), 'setasmgid');
     if (-f $file)
     {
       trace("Remove $file");
       s_remove_file("$file");
     }

     downgradeOratab();
   }

   if (($CFG->DOWNGRADE) && (! $CFG->LASTNODE)) {
     set_bold();
     print_error(335);
     reset_bold();
   }

   if (! $CFG->DOWNGRADE) {
      set_bold();
      print_error(336);
      reset_bold();
   }
} 

sub CHMDowngrade
{
  my $oldcrfhome  = $CFG->OLD_CRS_HOME;
  my $crfhome = $CFG->ORA_CRS_HOME;
  my $host = $CFG->HOST;
  my $oldorafile = catfile("$oldcrfhome", "crf", "admin", "crf$host.ora");
  my $oldcfgfile = catfile("$oldcrfhome", "crf", "admin", "crf$host.cfg");
  my $orafile = catfile("$crfhome", "crf", "admin", "crf$host.orabkp");
  my $cfgfile = catfile("$crfhome", "crf", "admin", "crf$host.cfgbkp");
  my $bdbloc;
  my $oldbdbdirbkp;
  
  if ($^O =~ /linux/i || $^O =~ /aix/i || $^O =~ /solaris/i || $CFG->platform_family eq "windows")
  {  

    if (!-e $orafile)
    {
      trace("Failed to find Cluster Health Monitor internally-generated file $orafile during downgrade.");
    }

    if (!copy_file($orafile, $oldorafile))
    {
      trace("Failed to copy Cluster Health Monitor internally-generated file $orafile to $oldorafile during downgrade.");
    }

    if (!copy_file($cfgfile, $oldcfgfile))
    {
      trace("Failed to copy Cluster Health Monitor internally-generated file $cfgfile to $oldcfgfile during downgrade.");
    }
  }
  $bdbloc = getCHMAttrib("BDBLOC", $orafile);

  if ($bdbloc eq "default")
  {
    $bdbloc = catfile("$crfhome", "crf", "db", $host);
  }
  
  # Check if the older bdb location exists
  if (-d $bdbloc)
  {
    my $bdbdirbkp = "$orafile"."bkp";
    # Check if there is old bdb files if yes move to old location.
    if (-d $bdbdirbkp)
    {
      my $bdbdirbkp = "$orafile"."bkp";
      # Check if there is old bdb files if yes move to old location.
      if (-d $bdbdirbkp)
      {
        movedir($bdbdirbkp, $bdbloc);
      }
    }
    else
    {
      trace ("Not able to find older bdb location at $bdbloc");
    }
  }
  else
  {
    trace ("Not able to find older bdb location at $bdbloc");
  }
}


sub TraceOptions
{
   my $options;

   if ($CFG->DOWNGRADE) { $options = $options . "-downgrade "; }
   if ($CFG->FORCE)     { $options = $options . "-force "; }
   if ($CFG->LASTNODE)  { $options = $options . "-lastnode "; }
   if ($options) {
      trace ("options=$options");
   }
}

sub ValidateSRVCTL
{
   trace ("Validate srvctl command");
   my $srvctl = crs_exec_path('srvctl');
   my $srvctl_exists = ValidateCommand ($srvctl);

   if (! $srvctl_exists) {
      if ($CFG->FORCE) {
         return SUCCESS;
      } 
      else {
         print_error(36, $srvctl);
         print_error(37);
	 trace ("Use -force option to force deconfiguration");
         return FAILED;
      }
   }

   return SUCCESS;
}

sub ValidateCRSCTL
{
   trace ("Validate crsctl command");
   my $crsctl = crs_exec_path('crsctl');

   if (! ValidateCommand($crsctl)) {
      if ($CFG->FORCE) {
         return SUCCESS;
      } 
      else {
         print_error(36, $crsctl);
         print_error(37);
         trace ("Use -force option to force deconfiguration");
         return FAILED;
      }
   }

   return SUCCESS;
}

sub ValidateCommand
#-------------------------------------------------------------------------------
# Function: Validate system command to ensure command exists and
#           exececutable.
# Args    : 1
#-------------------------------------------------------------------------------
{
   my $cmd = $_[0];

   trace("Validating $cmd");
   if (-x $cmd) {
      return (TRUE);
   } else {
      return (FALSE);
   }
}

sub removeListeners
{
   trace ("Remove listener resource...");
   my $srvctl = crs_exec_path('srvctl');

   # Verify listener resources when deconfiguring last node in the cluster
   # Listener resource are local resources
   my @out = system_cmd_capture ($srvctl, 'config', 'listener');
   my $rc  = shift @out;

   if ($rc == 0) {
      print_error(332);
      #stop the listener resource
      my @cmd = ($srvctl, 'stop', 'listener');
      if ($CFG->FORCE) {
         push @cmd, '-f';
      }

      trace ("@cmd");
      system_cmd(@cmd);

      if ($CHILD_ERROR != 0) {
         if ($CFG->FORCE) {
            print_error(424);
         } 
         else {
            print_error(423);
            return FAILED;
         }
       }

      #remove the listener resource
      @cmd = ($srvctl, 'remove', 'listener', '-a');
      if ($CFG->FORCE) {
         push @cmd, '-f';
      }

      trace ("@cmd");
      system_cmd(@cmd);

      if ($CHILD_ERROR != 0) {
         if ($CFG->FORCE) {
            print_error(334);
         } 
         else {
            print_error(333);
            return FAILED;
         }
      }
    }

   return SUCCESS;
}

sub VerifyASMProxy
{
   return SUCCESS;
}

sub VerifyDatabases
{
  return SUCCESS;
}

sub VerifyResources
#-------------------------------------------------------------------------------
# Function: Verify resources (db, lsnr, asm) 
# Args    : 0
#-------------------------------------------------------------------------------
{
   trace ("Verifying the existence of CRS resources used by Oracle RAC databases");

   my $node_name = $CFG->HOST;

   # Check if CRS is running
   trace ("check_service cluster");
   my $crs_running = check_service ("cluster", 2);

   if (! $crs_running) {
      if ($CFG->FORCE) {
         return SUCCESS;
      } else {
         print_error(38);
	 trace("Restart the clusterware stack (use crsctl start crs) and retry");
         return FAILED;
      }
   }

   # Validate system command
   ValidateSRVCTL || return FAILED;

   if ($CFG->LASTNODE) {
      removeListeners () || return FAILED;
      VerifyASMProxy () || return FAILED;
   }

   VerifyDatabases () || return FAILED;
}

sub VerifyHAResources
#-------------------------------------------------------------------------------
# Function: Verify resources (db, lsnr, asm)
# Args    : 0
#-------------------------------------------------------------------------------
{
   trace ("Verifying the existence of SIHA resources used by Oracle databases");

   my $node_name = $CFG->HOST;

   # Check if SIHA is running
   trace ("check_service SIHA");
   my $siha_running = check_service ("ohasd", 2);

   if (! $siha_running) {
      if ($CFG->FORCE) {
         return SUCCESS;
      } else {
         print_error(39);
         trace("Restart the SIHA stack (use crsctl start has) and retry");
         return FAILED;
      }
   }

   # Validate system command
   ValidateSRVCTL () || return FAILED;

   removeListeners () || return FAILED;

   VerifyDatabases () || return FAILED;
}

sub GetDBInst
#-------------------------------------------------------------------------------
# Function: Get database instances.
# Args    : 2
#-------------------------------------------------------------------------------
{
   my $node_name    = $_[0];
   my $srvctl       = crs_exec_path('srvctl');
   my @db_inst_list = ();

   trace ("Get database instances for node $node_name");

   # Validate system command
   ValidateSRVCTL() || return FAILED;

   trace ("srvctl config database");
   open (SRVCTL_OUT, "$srvctl config database");
   my @config_db_out = <SRVCTL_OUT>;
   close (SRVCTL_OUT);

   foreach my $line (@config_db_out) {
      chomp($line);
      if ($line =~ $node_name) {
         my @word = split(/ /, $line);
         push (@db_inst_list, $word[1]);
      }
   }

   trace ("database instance list = @db_inst_list");
   return(@db_inst_list);
}

sub RemoveResources
#---------------------------------------------------------------------
# Function: Remove nodeapps
# Args    : 0
#---------------------------------------------------------------------
{
   trace ("Remove Resources");

   # Validate system command
   ValidateSRVCTL || return FAILED;

   if ($CFG->LASTNODE) {
      RemoveCVU();
      RemoveOC4J();
      RemoveScan();
   }

   RemoveNodeApps ();
}

sub RemoveNodeApps
#---------------------------------------------------------------------
# Function: Remove nodeapps
# Args    : 0
#---------------------------------------------------------------------
{
   trace ("Removing nodeapps...");

   # check if nodeapps is configured
   my $srvctl = crs_exec_path('srvctl');
   system ("$srvctl config nodeapps");

   if ($CHILD_ERROR != 0) {
      return SUCCESS;
   }

   my $node  = $CFG->HOST;
   my ($cmd, $status, $force);

   if ($CFG->FORCE) { $force = '-f'; }

   # stop nodeapps
   $cmd = "$srvctl stop nodeapps -n $node $force";
   $status = system ($cmd);

   if ($status == 0) {
      trace ("$cmd ... success");
   } else {
      trace ("$cmd ... failed");
   }

   # remove nodeapps if lastnode, otherwise remove VIP
   if ($CFG->LASTNODE) {
      $cmd    = "$srvctl remove nodeapps -y $force";
      $status = system ($cmd);

      if ($status == 0) {
         trace ("$cmd ... success");
      } else {
         trace ("$cmd ... failed");
      }
   }
   else 
   {
     # vip does not exist on leaf nodes
     if (isHubNode())
     {
       $cmd    = "$srvctl remove vip -i $node -y $force";
       $status = system ($cmd);

       if ($status == 0) {
         trace ("$cmd ... success");
       } else {
         trace ("$cmd ... failed");
       }
     }
   }
}

sub RemoveCVU
{
  trace("Removing CVU ...");

  my $ret = TRUE;
  my $force;
  if ($CFG->FORCE)
  {
    $force = '-f';
  }

  my $run_as_owner = TRUE;
  my $status = srvctl($run_as_owner, "stop cvu $force");
  if ($status)
  {
    trace("Stop CVU ... success");
  }
  else
  {
    print_error(180, "srvctl stop cvu $force", ${status});
    $ret = FALSE;
  }

  $status = srvctl($run_as_owner, "remove cvu $force");
  if ($status)
  {
    trace("Remove CVU ... success");
  }
  else
  {
    print_error(180, "srvctl stop cvu $force", ${status});
    $ret = FALSE;
  }

  return $ret;
}

sub RemoveScan
#-------------------------------------------------------------------------------
# Function: Remove scan and scan listener
# Args    : 0
#-------------------------------------------------------------------------------
{
   trace("Removing scan....");

   my $owner = $CFG->params('ORACLE_OWNER');
   my (@cmd, $force);

   if ($CFG->FORCE) {
      $force = '-f';
   }

   my $srvctl = crs_exec_path('srvctl');

   # stop/remove scan listener
   @cmd = ($srvctl, 'stop', 'scan_listener', "$force");
   trace ("@cmd");
   run_as_user($owner, @cmd);

   @cmd = ($srvctl, 'remove', 'scan_listener', '-y', "$force");
   trace ("@cmd");
   run_as_user($owner, @cmd);

   # stop/remove scan vip
   @cmd = ($srvctl, 'stop', 'scan', "$force");
   trace ("@cmd");
   system (@cmd);

   # remove scan vip
   @cmd = ($srvctl, 'remove', 'scan', '-y', "$force");
   trace ("@cmd");
   system(@cmd);
}

sub RemoveHAResources
#---------------------------------------------------------------------
# Function: Remove HA application resources
# Args    : 0
#---------------------------------------------------------------------
{
   trace ("Remove High Availability resources...");
   my $srvctl = crs_exec_path('srvctl');

   # Validate system command
   ValidateSRVCTL () || return FAILED;

   # remove ONS
   my @out = system_cmd_capture ($srvctl, 'config', 'ons');
   my $rc  = shift @out;

   if ($rc == 0) {
      # stop ons
      my @cmd = ($srvctl, 'stop', 'ons');
      if ($CFG->FORCE) {
         push @cmd, '-f';
      }

      trace ("@cmd");
      system_cmd(@cmd);

      # remove ons
      @cmd = ($srvctl, 'remove', 'ons');
      if ($CFG->FORCE) {
         push @cmd, '-f';
      }

      trace ("@cmd");
      system_cmd(@cmd);
   }

   # remove eONS
   @out = system_cmd_capture ($srvctl, 'config', 'eons');
   $rc  = shift @out;

   if ($rc == 0) {
      # stop eons
      my @cmd = ($srvctl, 'stop', 'eons');
      if ($CFG->FORCE) {
         push @cmd, '-f';
      }

      trace ("@cmd");
      system_cmd(@cmd);

      # stop eons
      @cmd = ($srvctl, 'remove', 'eons');
      if ($CFG->FORCE) {
         push @cmd, '-f';
      }

      trace ("@cmd");
      system_cmd(@cmd);
   }
}

sub RemoveOC4J
{
  trace("Removing OC4J....");

   my $owner = $CFG->params('ORACLE_OWNER');

   my $srvctl = crs_exec_path('srvctl');

   # stop/remove oc4j
   my @cmd = ($srvctl, 'stop', 'oc4j', '-f');
   trace ("@cmd");
   run_as_user($owner, @cmd);

   @cmd = ($srvctl, 'remove', 'oc4j', '-f');
   trace ("@cmd");
   run_as_user($owner, @cmd);
}

sub RemoveDrivers
{
   my $chkpoints = $_[0];
   my $ACFS_supported = isACFSSupported();

   my $AFD_supported = isAFDSupported();
  
   if ($CFG->params('CRS_STORAGE_OPTION') == 1) {
 
     if ($ACFS_supported) {
       oraacfs::removeACFSRoot($chkpoints);
     }
   } 

   if($chkpoints eq 1)
   {
       #my $OKA_supported = isOKASupported();
      
       #if ($OKA_supported) {
       #   crska::removeOKARoot();
       #}
   }
   if ($AFD_supported) {
     #TODO: shmubeen remove later
      my $afdInstall = FALSE;

      $afdInstall = uc($ENV{'ORA_ENABLE_AFD_INSTALL'});
      if ( $afdInstall ne "TRUE" )
      {
        trace("AFD disabled because of ENV in test mode");
      }

     s_rm_afdinit_rclevel();
     s_rm_afdinit_init();
     afd::removeAFDRoot($chkpoints);
   }
}

sub DeconfigStorage 
{
   my $success = TRUE;

   trace ("Deconfiguring Oracle ASM storage ...");

   # Validate system command
   ValidateCRSCTL || return FAILED;

   # check if ACFS supported
   my $ACFS_supported = isACFSSupported();
   my $OKA_supported = isOKASupported();

   my $AFD_supported = isAFDSupported();
  
   trace("Stopping Oracle Clusterware ...");
   # Do not start ohasd before CSS exclusive startup
   $success = stopFullStack("force");

   if (($CFG->LASTNODE) && isOCRonASM())
   {
     if (! deconfigure_ASM()) { $success = FALSE; }
     # Here we're stopping CSS started in deconfigure_ASM().
     $success = stopFullStack("force");
   }
   
   sleep 10;  # Allow CRS daemons to shutdown in 10sec

   RemoveDrivers(1);

   if (!$success) {
      print_error(191);
   }
}

sub StopHAStack
#-------------------------------------------------------------------------------
# Function: Stop Oracle Restart
# Args    : 0
#-------------------------------------------------------------------------------
{
   trace ("Stop Oracle Restart");
   my $crsctl = crs_exec_path('crsctl');
   my @cmd    = ();

   # Validate system command
   ValidateCRSCTL || return FAILED;

   # stop cssd
   @cmd = ($crsctl, 'stop', 'resource', 'ora.cssd');
   if ($CFG->FORCE) {
      push @cmd, '-f';
   }

   trace ("@cmd");
   system_cmd(@cmd);

   # remove ohasd resource
   @cmd = ($crsctl, 'delete', 'resource', 'ora.cssd');
   if ($CFG->FORCE) {
      push @cmd, '-f';
   }

   trace ("@cmd");
   system_cmd(@cmd);

   # stop ohasd
   trace ("Stopping Oracle Restart");
   @cmd = ($crsctl, 'stop', 'has');
   if ($CFG->FORCE) {
      push @cmd, '-f';
   }

   trace ("@cmd");
   system_cmd(@cmd);

   sleep 10; # Allow HA daemons to shutdown in 10sec

   # check the status of HA stack
   if (! checkServiceDown("ohasd")) {
      print_error(357);
   }
}

sub DowngradeTo9i
#--------------------------------------------------------------------
# Function: Downgrade to 9i
# Args    :
#--------------------------------------------------------------------
{
   my ($ocr_loc, $srv_loc);

   # Check existence of srvconfig.loc
   if (-f $CFG->params('SRVCONFIG')) { 
      # CASE 1: srvConfig.loc does exist. So lets see the OCR location
      $srv_loc = get_srvdisk(); 
      $ocr_loc = get_ocrdisk(); 

      if ($srv_loc eq "/dev/null") {
         # CASE 1.1: srvconfig_loc = /dev/null
         # srvConfig.loc has /dev/null that was invalidated during 10g install
         # Copying the location from ocr.loc to srvConfig.loc for downgrade
         AddLocation ($CFG->params('SRVCONFIG'), "srvconfig_loc", $ocr_loc)
      }
      elsif ($srv_loc ne $ocr_loc) {
         # CASE 1.2: if srvconfig_loc = ocrconfig_loc, do nothing.
         # Otherwise OCR device in ocr.loc will be wiped out next
         $ocr_loc = $srv_loc;
      }
   } 

   if ($CFG->LASTNODE) {
      # Ensure 9i GSD can read/write 9i OCR
      if (isOwnerGroupValid()) {
         chown ($CFG->params('ORACLE_OWNER'), $CFG->params('ORA_DBA_GROUP'), $ocr_loc);
      }
 
      chmod (0644, $ocr_loc);
   }

   unlink($CFG->params('OCRCONFIG'));

}

sub DowngradeTo10or11i
#----------------------------------------------------------------
# Function:
# Args    :
#----------------------------------------------------------------
{
   # Re-create ocr.loc for version 10 and later
   my $ocr_loc        = get_ocrdisk();
   my $ocr_mirror_loc = get_ocrmirrordisk();
   my $ocr_loc3       = get_ocrloc3disk();
   my $ocr_loc4       = get_ocrloc4disk();
   my $ocr_loc5       = get_ocrloc5disk();
   my $local_only     = get_ocrlocaldisk("ocr", "local_only");

   # for all 10 version and later
   AddLocation ($CFG->params('OCRCONFIG'), "ocrconfig_loc", $ocr_loc);

   # 10.1 doesn't support any mirroring
   if ($CFG->oldcrsver ne "10.1") {
      if (! -z $ocr_mirror_loc) {
         AddLocation ($CFG->params('OCRCONFIG'), "ocrmirrorconfig_loc",
                      $ocr_mirror_loc);
      } 
   } 

   # 10.1, 10.2 and 11.1 only support 2-way mirroring
   if (($CFG->oldcrsver ne "10.1") && 
       ($CFG->oldcrsver ne "10.2") && 
       ($CFG->oldcrsver ne "11.1")) 
   {
      if (! -z $ocr_loc3) {
         AddLocation ($CFG->params('OCRCONFIG'), "ocrconfig_loc3", $ocr_loc3);
      } 

     if (! -z $ocr_loc4) {
         AddLocation ($CFG->params('OCRCONFIG'), "ocrconfig_loc4", $ocr_loc4);
      } 

      if (! -z $ocr_loc5) {
         AddLocation ($CFG->params('OCRCONFIG'), "ocrconfig_loc5", $ocr_loc5);
      }
   }

   # for all 10 version and later; append local_only to the end
   if (! -z $local_only) {
      AddLocation ($CFG->params('OCRCONFIG'), "local_only", $local_only);
   } 

   #ocr.loc will be preserved for downgrading to 10.1
}

sub AddLocation
#---------------------------------------------------------------------
# Function: Add ocrconfig_loc, ocrmirrorconfig_loc, etc...
#           If location already exists it replaces with the new location
# Args    : 3
#---------------------------------------------------------------------
{
   my $infile        = $_[0];
   my $match_pattern = $_[1];
   my $replace_text  = $_[2];
   my $found_pattern = FALSE;

   open (InFile, "<$infile")      || die(dieformat(206, $infile, $!));
   open (OutFile, ">$infile.tmp") || die(dieformat(207, "$infile.tmp", $!));

   while (my $line = <InFile>) {
      if ($line =~ /$match_pattern\b/) {
         $found_pattern = TRUE;
         print OutFile "$match_pattern=$replace_text\n";
      }
      else {
         print OutFile $line;
      }
   }

   if (! $found_pattern) {
      print OutFile "$match_pattern=$replace_text\n";
   }

   close (InFile)  || die(dieformat(272, $infile, $!));
   close (OutFile) || die(dieformat(272, "$infile.tmp", $!));

   move ("$infile.tmp", $infile)
      or die(dieformat(209, "$infile.tmp", $infile, $!));
}

sub DeleteSCR
#-------------------------------------------------------------------------------
# Function: Remove SCR dir/registry
# Args    : none
#-------------------------------------------------------------------------------
{
   if ($CFG->platform_family eq 'windows') {
      s_removeSCR();
   }
   else {
      if (-d $CFG->params('SCRBASE')) {
         trace ("Cleaning up SCR settings in " . $CFG->params('SCRBASE'));
         rmtree ($CFG->params('SCRBASE'));
      }

      if (-d $CFG->params('OPROCDDIR')) {
         trace ("Cleaning oprocd directory, and log files");
         rmtree ($CFG->params('OPROCDDIR'));
      }
   }
}

sub remove_misc_dirs
{
   # Following directories and files are removed
   # only if root script is invoked from deinstall tool
   my $diagdir = catdir ($CFG->params('ORACLE_BASE'), 'oradiag_root');
   my $mapdir = catdir ($CFG->params('OCRCONFIGDIR'), 'maps');
   my $asmgidfile = catfile($CFG->params('OCRCONFIGDIR'), 'setasmgid');

   if ($CFG->defined_param('HOME_TYPE')) {
      # remove oracle_base/oradiag_root,/etc/oracle/maps & /etc/oracle/setasmgid
      if (-e $diagdir) {
         trace ("Remove $diagdir");
         rmtree ($diagdir);
      }
      
      if (-e $mapdir) {
         trace ("Remove $mapdir");
         rmtree ($mapdir);
      }

      if (-e $asmgidfile)
      {
        trace("Remove $asmgidfile");
        s_remove_file("$asmgidfile");
      }
   }
   else {
      trace ("Root script is not invoked as part of deinstall. $diagdir, " .
             "$mapdir, and $asmgidfile are not removed");
   }
}

sub HADeconfigure
{
   trace ("Deconfiguring Oracle Restart on this node");
   TraceOptions ();

   remove_checkpoints();

   if (! $CFG->DOWNGRADE) {
      my $status = VerifyHAResources ();
      if ($status eq FAILED) {
         die(dieformat(312));
      }

      # Only remove resources which were created during
      # Oracle Restart configuration
      RemoveHAResources ();
   }

   StopHAStack();

   RemoveDrivers(0);

   s_RemoveInitResources();

   crsDelete_ResetOCR();

   DeleteSCR();

   if ($CFG->platform_family eq "unix") {
      s_NSCleanUp();
   }

   if ($CFG->platform_family eq 'windows') {
      s_houseCleaning();
   }

   if (($CFG->defined_param('HOME_TYPE')) && ($CFG->platform_family eq 'unix'))
   {
     my $file = catfile($CFG->params('OCRCONFIGDIR'), 'setasmgid');
     if (-f $file)
     {
       trace("Remove $file");
       s_remove_file("$file");
     }

     my $mapdir = catdir($CFG->params('OCRCONFIGDIR'), 'maps');
     if (-e $mapdir)
     {
       trace ("Remove dir $mapdir");
       rmtree ($mapdir);
     }
   }

   s_CleanTempFiles();

   if ($CFG->defined_param('HOME_TYPE')) {
      trace ("Opening permissions on Oracle Restart home");   
      s_reset_crshome($CFG->params('ORACLE_OWNER'), $CFG->params('ORA_DBA_GROUP'),
                      755, $CFG->params('ORACLE_HOME'));
   }

   set_bold();
   print_error(337);
   reset_bold();

}

sub ocrDowngrade
#-------------------------------------------------------------------------------
# Function: Downgrade OCR
# Args    : None
#-------------------------------------------------------------------------------
{
 my $OLD_CRS_HOME  = $CFG->oldcrshome;
 my $ocrconfigbin  = catfile ($OLD_CRS_HOME, 'bin', 'ocrconfig');
 my $oldcrsver     = $CFG->oldcrsver;
 my $ocrbackupfile = "ocr" . "$oldcrsver";
 my $ocrbackuploc  = catfile ($CFG->params('ORACLE_HOME'), 'cdata', $ocrbackupfile);
 my $rc = TRUE;

 if (! -e $OLD_CRS_HOME) {
    print_error(46, $OLD_CRS_HOME);
    trace ("crshome: $OLD_CRS_HOME not found. " .
           "Re-try the command with right path for old crs home");     
    exit 1;
 }

 trace("OCR backup file is $ocrbackuploc");

 if ((! (-f $ocrbackuploc)) || 
     (0 !=  system_cmd($ocrconfigbin, '-import', $ocrbackuploc)))
 {
    my $rootcrs = catfile($CFG->params('ORACLE_HOME'), 'crs', 'install',
                          'rootcrs.pl');
    $rc = FALSE;
    print_error(204, $oldcrsver);
    print "Perform the following to restore to old version Clusterware\n";
    print "1. Run '$rootcrs -force -deconfig' \n";
    print "   on all the remote nodes\n";
    print "2. Run '$rootcrs -force -deconfig -lastnode'\n";
    print "   on the local node\n";
    print "3. Re-install old Clusterware and re-configure all the resources with \n";
    print "   appropriate srvctl commands\n";
 } else {
    my $rootscript = "root.sh";
    if ($CFG->platform_family eq "windows") {
       $rootscript = "gridconfig.bat";
    }

    print_error(338, $oldcrsver);
    if (isOldVersionLT112())
    {
      print_error(339, $rootscript);
    }
 }

 return $rc;
}

sub crsDelete_ResetOCR
{
   if ($CFG->DOWNGRADE) {
      if ($CFG->oldcrsver eq "9.2") {
         DowngradeTo9i ();
      } 
      else {
        DowngradeTo10or11i ();
      }

      return;
   }

   s_ResetOCR();
}

sub deconfigure_ASM {
   trace ("De-configuring ASM...");

   my $owner           = $CFG->params('ORACLE_OWNER');
   my $rc              = FALSE;
   my $status;

   trace("Starting CSS exclusive");
   my $css_rc = CSS_start_exclusive();
   if ($css_rc != CSS_EXCL_SUCCESS)
   {
     trace ("CSS failed to enter exclusive mode to de-configure ASM");
     return FALSE;
   }

   # delete voting disks on ASM
   if (! CSS_delete_vfs ())
   {
     trace ("Unable to delete voting files in exclusive mode");
     return FALSE;
   }

   # start ora.asm resource
   if (! start_resource("ora.asm", "-init"))
   {
     trace ("Unable to start ora.asm resource to deconfigure ASM");
     return FALSE;
   }

   # call asmca -deleteLocalASM to delete diskgroup
   # Do not change the order of these parameters as asmca requires the
   # parameters to be in a specific order or it will fail
   my @runasmca = (catfile ($CFG->ORA_CRS_HOME, "bin", "asmca"),
                   '-silent', '-deleteLocalASM');

   if ($CFG->params('ASM_DISK_GROUP') ){
      my $diskgroup = $CFG->params('ASM_DISK_GROUP');
      if ($diskgroup =~ /\$/) {
         # if diskgroup contains '$', put single-quotes around it
         quoteDiskGroup($diskgroup);
         push @runasmca, '-diskGroups', "'$diskgroup'";
      }
      else {
         push @runasmca, '-diskGroups', $diskgroup;
      }
   }

   if (($CFG->defined_param('ASM_DISKSTRING')) && ($CFG->params('ASM_DISKSTRING'))) {
      my $disktring = $CFG->params('ASM_DISKSTRING');
      push @runasmca, '-diskString', "'$disktring'";
   }

   if ($CFG->params('NODE_NAME_LIST')) {
      push @runasmca, '-nodeList', $CFG->params('NODE_NAME_LIST');
   }

   # ASM_DROP_DISKGROUPS is valid only if the root script is invoked
   # from the deinstall tool, and we pass -reuseDiskgroup to asmca if
   # ASM_DROP_DISKGROUPS is set to false.
   #
   # $CFG->KEEPDG can be set along with -deconfig
   if ((($CFG->defined_param('ASM_DROP_DISKGROUPS')) &&
        ('false' eq $CFG->params('ASM_DROP_DISKGROUPS'))) || $CFG->KEEPDG)
   {
     trace("Re-use existing diskgroups");
     push @runasmca, '-reuseDiskgroup';
     removeOCROnDG();
   }

   $ENV{'PARAM_FILE_NAME'} = $CFG->paramfile;
   my $ASMCA_log = catdir($CFG->params('ORACLE_BASE'), 'cfgtoollogs', 'asmca');
   $status       = run_as_user($owner, @runasmca);

   if ($status == 0)
   {
     # set remote asm to disabled in the gpnp profile
     $rc = disableRemoteASM();
      if (SUCCESS != $rc)
      {
        trace("de-configuration of ASM ... failed with error $rc");
        print_error(170, $rc);
        return $rc;
      }       
      trace ("de-configuration of ASM ... success");
      trace ("see asmca logs at $ASMCA_log for details");
      $rc = TRUE;
   }
   else {
      trace("de-configuration of ASM ... failed with error $status");
      trace("see asmca logs at $ASMCA_log for details");
      print_error(170, $status);
   }

   return $rc;
}

sub downgrade_GetOldConfig
{
  my $ckptName = "ROOTCRS_OLDHOMEINFO";
  my $ckptStatus;
  
  if (isCkptexist($ckptName))
  {
    $ckptStatus = getCkptStatus($ckptName);
  }
  else
  {
    trace("The checkpoint ROOTCRS_OLDHOMEINFO doesn't exist");
    die(dieformat(416));
  }

  if (($ckptStatus eq CKPTSUC) || is_dev_env())
  {
    my $oldHome;
    my $oldVersion;

    # Get old CRS home
    $oldHome = getCkptPropertyValue($ckptName, "OLD_CRS_HOME");
    trace("Old CRS home from ckpt property is [$oldHome]");
    $oldHome = trim($oldHome);

    if ($oldHome)
    {
      $CFG->oldconfig('ORA_CRS_HOME', $oldHome);
      $CFG->oldcrshome($oldHome);
    }
    else
    {
      trace("Failed to retrieve old Grid Infrastructure home location");
      die(dieformat(416));
    }

    # Get old CRS version
    $oldVersion = getCkptPropertyValue($ckptName, "OLD_CRS_VERSION");
    trace("Old crs version from ckpt property is [$oldVersion]");
    my @oldVer = split(/\./, trim($oldVersion));
    $CFG->oldconfig('ORA_CRS_VERSION', \@oldVer);
    $CFG->oldcrsver(trim($oldVersion));
  }
  else
  {
    trace("The checkpoint status of ROOTCRS_OLDHOMEINFO is $ckptStatus");
    die(dieformat(416));
  }
}

sub crsDowngrade
{
  if (! $CFG->DOWNGRADE) { return; } 

  if (isOldVersionLT112())
  {
    trace("Downgrade Oracle Clusterware to a pre-11.2 version");
    return;
  }

  trace("Downgrade Oracle Clusterware to a post-11.2 version");

  my %ref;
  if (isOCRonASM())
  {
    trace("Downgrade with OCR on ASM");
    my @diskgroups = extractDiskgroups();
    if (scalar(@diskgroups) <= 0)
    {
      trace("No diskgroups to store voting file(s) found");
      exit(1);
    }

    %ref =
    (
      USE_ASM       => 1,
      RM_VOTEDISKS  => \&removeVotingDiks,
      ADD_VOTEDISKS => \&addVotingDisks,
      VOTEDISKS     => $diskgroups[0],
    );
  }
  else
  {
    trace("Downgrade with OCR on filesystem");
    my @votefiles = extractVotedisks();
    if (scalar(@votefiles) <= 0)
    {
      trace("No voting file(s) found");
      exit(1);
    }

    %ref =
    (
      USE_ASM       => 0,
      RM_VOTEDISKS  => \&removeVotingfiles,
      ADD_VOTEDISKS => \&addVotingFiles,
      VOTEDISKS     => \@votefiles,
    );
  }

  crsDowngrade_Precheck();

  trace("Stop the current CRS stack");
  stopFullStack("force") || die(dieformat(349));

  my $USE_ASM = $ref{USE_ASM};
  if ($CFG->LASTNODE)
  {
    # Delete voting disks
    trace("Start CRS in exclusive mode from new home");
    start_excl_crs($CFG->ORA_CRS_HOME) || die(dieformat(260));

    trace("Delete voting disks. OCR on ASM: $USE_ASM");
    my $rm_func = $ref{RM_VOTEDISKS};
    if (FAILED == &$rm_func($CFG->ORA_CRS_HOME, $ref{VOTEDISKS}))
    {
      die(dieformat(415));
    }

    stopFullStack("force") || die(dieformat(349));
  }

  my $acfsSupported = isACFSSupported();
  # De-install current ACFS drivers
  if ($acfsSupported)
  {
    removeACFSRoot(1);
  }

  # Copy the OLR backup
  crsDowngrade_RestoreOLR();

  #Delete the OHASD service
  if ($CFG->platform_family eq "windows") {
	  s_deltService("OracleOHService");
          s_delete_ASM_Services_downgrade();
          s_register_service('ohasd');
  }

  if ($CFG->LASTNODE)
  {
    # Start old stack in exclusive mode without CRSD running
    crsDowngrade_StartExclNoCRS();

    # unset ORACLE_HOME environment variable since this var when set to 
    # current home location causes issues with OCRDowngrade.
    # Bug# 16467434
    
    $ENV{'ORACLE_HOME'}  = "";

    # Downgrade OCR
    crsDowngrade_DowngradeOCR();

    # Start CRSD
    trace("CRSD needs to be up for adding vote disks back");
    start_crsd_and_check($CFG->OLD_CRS_HOME) || die(dieformat(249));

    # Add Voting disks back
    trace("Add voting disks. OCR on ASM: $USE_ASM");
    my $add_func = $ref{ADD_VOTEDISKS};
    if (FAILED == &$add_func($CFG->OLD_CRS_HOME, $ref{VOTEDISKS}))
    {
      die(dieformat(261));
    }

    trace("Stopping the old stack");
    stopClusterware($CFG->OLD_CRS_HOME, "crs") || die(dieformat(349));
  }

  if ($acfsSupported)
  {
    if (WARNING == crsDowngrade_reinstallACFSRoot())
    {
      print_error(453);
    }
  }

  crsDowngrade_postTasks();
  
  trace ("Successfully downgraded Oracle Clusterware stack on this node");

  set_bold();
  print "Successfully downgraded Oracle Clusterware stack on this node\n";
  if ($CFG->LASTNODE)
  { 
    my $crshome = $CFG->OLD_CRS_HOME;
    print "Run '$crshome/bin/crsctl start crs' on all nodes ".
          "to complete downgrade\n";
  } 
  reset_bold();

  exit(0); 
}

sub crsDowngrade_reinstallACFSRoot
{
  my $acfsroot;
  my $ret = SUCCESS;
  
  trace("Re-installing ACFS drivers from older home ...");
  if ($CFG->platform_family eq 'windows')
  {
    $acfsroot = catfile($CFG->OLD_CRS_HOME, 'bin', 'acfsroot.bat');
  }
  else
  {
    $acfsroot = catfile($CFG->OLD_CRS_HOME, 'bin', 'acfsroot');
  }

  if (-e $acfsroot)
  {
    my $cmd = "$acfsroot install";
    trace("Executing '$cmd'"); 
    my @output = system_cmd_capture($cmd);
    my $status = shift(@output);
    if (0 == $status)
    {
      trace("Successfully execute 'acfsroot install' from older home");
      $ret = SUCCESS;
    }
    elsif (2 != $status)
    {
      if (scalar(grep(/09394/, @output)) > 0)
      {
        trace("'acfsroot install' succeeded, but need a system reboot");
        $ret = WARNING;
      }
      else
      {
        trace("Failed to execute 'acfsroot install' from older home");
        return FAILED;
      }
    }
  }

  return $ret;
}

sub crsDowngrade_Precheck
{
  trace("Check olr.loc bakcup");
  if (FAILED == s_checkolrbackup())
  {
    die(dieformat(417));
  }

  if ($CFG->LASTNODE)
  {
    my $oldcrsver     = $CFG->oldcrsver;
    my $ocrbackupfile = "ocr" . "$oldcrsver";
    my $ocrbackuploc  = catfile($CFG->ORA_CRS_HOME, 'cdata', $ocrbackupfile);

    trace("Checking OCR backup ...");
    if (! (-f $ocrbackuploc))
    {
      die(dieformat(414, $ocrbackuploc));
    }
  }
}

sub crsDowngrade_StartExclNoCRS
{
  my $res = FALSE;
  trace("Start Oracle Clusterware in exclusive mode from old home");

  my @OLD_CRS_VERSION = @{$CFG->oldconfig('ORA_CRS_VERSION')};
  my $ver = join(".", @OLD_CRS_VERSION);
  trace("The old version is [$ver]");

  if (($OLD_CRS_VERSION[0] == 11) && ($OLD_CRS_VERSION[1] == 2) &&
       ($OLD_CRS_VERSION[2] == 0) && ($OLD_CRS_VERSION[3] > 1))
  {
    trace("Start CRS in exclusive mode with 'nocrs' option");
    $res = startExclNoCRS($CFG->OLD_CRS_HOME);
  }
  else
  {
    trace("Stat CRS in exclusive mode then stop CRSD");
    if ((start_excl_crs($CFG->OLD_CRS_HOME)) &&
         (stop_crsd_and_check($CFG->OLD_CRS_HOME)))
    {
      $res = TRUE;
    }
  }

  if (! $res) { die(dieformat(260)); }

  trace("Successfully start Oracle Clusterware in exclusive mode");
}

sub crsDowngrade_RestoreOLR
{
  # Copy the OLR backup
  trace("Restore olr.loc from backup");
  if (FAILED == s_restoreolrloc())
  {
    die(dieformat(167, "olr.loc"));
  }
}

sub crsDowngrade_DowngradeOCR
{
  trace("Downgrading OCR ...");
  if (FALSE == ocrDowngrade())
  {
    exit(1); 
  }
  else
  {
    trace("Check if the OCR is properly configured");
    if (FAILED == checkOCR($CFG->OLD_CRS_HOME))
    {
      die(dieformat(361));
    }
  }
}

sub crsDowngrade_postTasks 
{
  trace("Restore init files");
  s_restoreInitScripts($CFG->OLD_CRS_HOME);
 
  my @oldCrsVer = @{$CFG->oldconfig('ORA_CRS_VERSION')};
  my $verinfo = join('.', @oldCrsVer);
  if (-1 == versionComparison($verinfo, "11.2.0.3.0"))
  {
    trace("Restore ASM files");
    if (FAILED == s_restoreASMFiles())
    {
      die(dieformat(167, "setasmgid"));
    }
  }

  downgradeOratab();

  remove_checkpoints();
}

sub prechecksDeconfig
{
  trace("Perform prechecks for deconfiguration");

  my %nodeRoles = getActiveNodeRoles($CFG->ORA_CRS_HOME);
  trace("Do not deconfigure the last hub node if there are still leaf nodes");
  if (NODE_ROLE_HUB eq $nodeRoles{($CFG->HOST)})
  {
    my $moreHubNodes = 0;
    my $moreLeafNodes = 0;

    for my $clunode (keys %nodeRoles)
    {
      next if ($CFG->HOST =~ /^$clunode$/i); 
      if (NODE_ROLE_HUB eq $nodeRoles{$clunode})
      {
        trace("Still have more hub nodes");
        $moreHubNodes = 1;
        last;
      }

      if (NODE_ROLE_RIM eq $nodeRoles{$clunode})
      {
        $moreLeafNodes = 1;
      }
    }

    if ((! $moreHubNodes) && ($moreLeafNodes))
    {
      print_error(451);
      exit(1);
    }
  }
}

#-------------------------------------------------------------------------------
# Function: Scan /etc/oratab and remove all entries that have higher version
#           of CRS home
# Args    : 0
#-------------------------------------------------------------------------------
sub downgradeOratab
{
  my $crshome = $CFG->ORA_CRS_HOME;
  my $host    = $CFG->HOST;

  if ($CFG->platform_family ne 'unix') { return; }
  my $oratab = catfile("/etc", "oratab");
  if (! -e $oratab)
  {
    trace("The $oratab file doesn't exist");
    return;
  }

  trace("Remove all new version related stuff from $oratab");
  my $oratabNew = catfile("/etc", "oratab.new.$host");

  my @oratabLines = read_file($oratab);
  open(ORATABNEW, ">${oratabNew}") or die(dieformat(207, $oratabNew, $!));
  my $changed = 0;
  foreach my $line (@oratabLines)
  {
    if ($line =~ /${crshome}/)
    {
      $changed = 1; 
      next;
    }   
    print ORATABNEW "$line";
  }
  close(ORATABNEW);

  if ($changed)
  {
    trace("Copying file $oratabNew to $oratab");
    copy_file($oratabNew, $oratab);
  }
  
  trace("Removing file $oratabNew");
  s_remove_file($oratabNew);
}

# Function: Remove the OCR file in the diskgroup
sub removeOCROnDG()
{
  my $crsctl = catfile($CFG->ORA_CRS_HOME, "bin", "crsctl");
  my @statcmd = ($crsctl, "stat", "res", "ora.asm", "-init", "-p");
  
  open STATOUT, "@statcmd |";
  my @output = <STATOUT>;
  close (STATOUT);

  my @txt = grep (/GEN_USR_ORA_INST_NAME/, @output);
  trace("ASM SID attribute: @txt");
  my ($key, $val) = split(/=/, $txt[0]);

  chomp($val);
  trace("ORACLE_SID = $val");
  
  $ENV{'ORACLE_SID'} = $val;
  $ENV{'ORACLE_HOME'} = $CFG->ORA_CRS_HOME;

  # remove the OCR file
  my $dgname = $CFG->params('ASM_DISK_GROUP');
  my $dg = "+".$dgname;
  my $asmcmd = catfile($CFG->ORA_CRS_HOME, "bin", "asmcmd");
  my $ocrfile = "\%REGISTRY.255.\%";
  my @cmd1 = ($asmcmd, "find", $dg, $ocrfile);
  my @capout1;
  my $rc = run_as_user2($CFG->params('ORACLE_OWNER'), \@capout1, @cmd1);
  
  my $ocrdgfile = $capout1[0];
  chomp($ocrdgfile);
  trace("OCR file on the diskgroup: $ocrdgfile");

  my @cmd2 = ($asmcmd, "rm", "-f",  $ocrdgfile);
  my @capout2;
  $rc = run_as_user2($CFG->params('ORACLE_OWNER'), \@capout2, @cmd2);
  if ($rc)
  {
    die(dieformat(427, $ocrdgfile, "@cmd2", "@capout2"));
  }
}

1;
