#!/usr/local/bin/perl
#
# $Header: has/install/crsconfig/crska.pm /main/3 2012/08/21 13:37:24 ssprasad Exp $
#
# crska.pm
#
# Copyright (c) 2008, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      crska.pm - Library module for Kernel Accelarator root install functions.
#
#    DESCRIPTION
#      crska.pm - Contains initial installation and deinstallation
#                   routines for OKA (Kernel Accelarator).
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#    ssprasad    06/29/12 - 'okaroot install' failures are ignored for root 
#                           scripts. Also, remove temporary disabled OKA code.
#    ssprasad    06/26/12 - Temporarily disable OKA
#    ssprasad    05/08/12 - Initial Creation
#

=head1 NAME

  crska.pm  RAC Kernel Accelarator component configuration/startup package

=head1 DESCRIPTION

   This package contains functions required for initial configuration
   and startup of the KA drivers.

=cut

package crska;
use strict;
use English;
use File::Temp qw/ tempfile /;
use File::Spec::Functions;
use File::Find ();

use crsohasd;
use crsutils;

use Exporter;
use vars qw(@ISA @EXPORT @EXPORT_OK);
@ISA = qw(Exporter);

my @exp_func  = qw(installOKADriver
                   disableOKADriver
                   deleteOKADriver
                   removeOKARoot
                   actionOKADriversResource
                   isOKASupported);

push @EXPORT, @exp_func;


=head2 installOKADriver

   Installs OKA - kernel and user components via okaroot.

=head3 Parameters

   None

=head3 Returns

  TRUE  - Configuration successful
  FALSE - Configuration failed

=head3 Notes


=cut
sub installOKADriver
{
   my $okaroot;
   my $ret = SUCCESS;
   my $crsctl = crs_exec_path('crsctl');
   my ($has) = @_;

   # if we are running in development mode, then limit support to only when
   # the appropriate env variables are set

   trace("installOKAdriver");
   if (is_dev_env())
   {
      my $okaInstall = uc($ENV{'ORA_ENABLE_OKA_INSTALL'});

      # if this ENV is not set then we give up early
      if ( $okaInstall ne "TRUE" )
      {
         trace("OKA disabled because of ENV in test mode");
         return $ret;
      }
   }

   if ($CFG->platform_family eq 'windows') {
      $okaroot = catfile ($CFG->ORA_CRS_HOME, 'bin', 'okaroot.bat');
   }
   else {
      $okaroot = catfile ($CFG->ORA_CRS_HOME, 'bin', 'okaroot');
   }

   if (-e $okaroot) {
      my $cmd = "$okaroot install";
      
      trace("Executing '$cmd'");
      my @output = system_cmd_capture($cmd);
      my $rc = shift @output;

      if ($rc == 0) {
         my $enable = "$crsctl enable $has";
         my @output = system_cmd_capture($enable);
         my $rc = shift @output;
         trace ("$cmd ... success");

         if ($rc != 0) {
           trace("$enable .... unable to enable CRS.");
           print_error(2000);
           $ret = FAILED;
         }
         else {
           trace("$enable .... success.");
         }
      }
      elsif ($rc != 2) {
         # 
         # 629 covers unload failures.
         # 650 covers install and load failures.
         #   (driver currently running, driver in use can't copy file etc.)
         # When we get above mentioned errors, we expect a 
         # reboot will fix them. 
         #
         # But, for now, we consider all non-success 'rc' to be same.
         # For any 'okaroot install' failure, we don't block
         # root scripts from proceeding. We return WARNING
         # and log a message in trace file.
         #
         # If OKA install/load fails, customer needs to manually
         # invoke the respective commands to enable KA.
         # We return WARNING to let root script proceed.
         #
         trace("OKA is supported on this platform, but install failed($rc).");
         $ret = WARNING;
      }
   }
   else {
      trace("$okaroot not found");
      if (isOKASupported())
      {
        # if okaroot not found and OKA supported, we have a problem
        # some required files are not here.
        trace("OKA is supported on this platform, but install files are missing.");
        $ret = FAILED;
      }
      else
      {
        # If okaroot not found and OKA not supported,
        # then assume everything is okay.
        trace("OKA is not supported on this platform.");
        $ret = SUCCESS;
      }
   }

  trace("OKA driver install status is $ret");
  return $ret;
}

sub disableOKADriver
#-------------------------------------------------------------------------------
# Function: Stop OKA drivers
# Args    : 0
#-------------------------------------------------------------------------------
{
   my $crsctl = crs_exec_path('crsctl');
   my $res    = 'ora.drivers.oka';

   # disable OKA drivers
   my @cmd    = ($crsctl, 'modify', 'resource', $res,
                 '-attr', "\"ENABLED=0\"", '-init');
   my $status = system_cmd(@cmd);

   if ($status == 0) {
      trace ("@cmd ... success");
   } else {
      trace ("@cmd ... failed");
   }
}

sub deleteOKADriver
#-------------------------------------------------------------------------------
# Function: Delete OKA resource
# Args    : 0
#-------------------------------------------------------------------------------
{
   my $crsctl = crs_exec_path('crsctl');
   my $res = 'ora.drivers.oka';

   # delete OKA drivers
   my @cmd = ($crsctl, 'delete', 'res', $res, '-init');
   if ($CFG->FORCE) {
      push @cmd, '-f';
   }

   trace ("@cmd");
   my $status = system_cmd(@cmd);
   if ($status == 0) {
      trace ("@cmd ... success");
   } else {
      trace ("@cmd ... failed");
   }
}

sub removeOKARoot
#-------------------------------------------------------------------------------
# Function: remove OKA drivers
#-------------------------------------------------------------------------------
{
   my $okaroot;
   my $has       =  "crs";
   my $crsctl    =  catfile ($CFG->ORA_CRS_HOME, "bin", "crsctl");

   if ($CFG->platform_family eq 'windows') {
      $okaroot = catfile ($CFG->ORA_CRS_HOME, 'bin', 'okaroot.bat');
   }
   else {
      $okaroot = catfile ($CFG->ORA_CRS_HOME, 'bin', 'okaroot');
   }

   if (! (-e $okaroot)) {
      trace ("OKA is not configured");
      return;
   }

   if (!isCkptexist("ROOTCRS_OKAUNINST")) 
   {
      trace("Writing checkpoint for OKA driver uninstall");
      writeCkpt("ROOTCRS_OKAUNINST", CKPTSTART);
      $CFG->wipCkptName("ROOTCRS_OKAUNINST");
   }

   if (!isCkptSuccess("ROOTCRS_OKAUNINST"))
   {
      $CFG->wipCkptName("ROOTCRS_OKAUNINST");

      my @cmd    = ($okaroot, 'uninstall');
      trace ("Executing @cmd");
      my @output = system_cmd_capture(@cmd);
      my $rc = shift @output;

      if ( $rc == 0 ) {
         trace ("@cmd ... success");
         trace("OKA drivers uninstall completed");
                 
         # Only try a re-enable if we had previously failed.
         # AND we are in the DOWNGRADE process
         # This is because we only require a reboot in the downgrade process. 
         # See below. So we disable crs before reboot and enable it here.

         if ( (getCkptStatus("ROOTCRS_OKAUNINST") == CKPTFAIL)  &&
              ($CFG->DOWNGRADE) )
         { 
            my $enable = "$crsctl enable $has";
            @output = system_cmd_capture($enable);
            $rc = shift @output;
            trace ("@cmd ... success");
            
            if ($rc != 0) {
               trace("$enable .... unable to enable CRS.");
               print_error(2002);
               exit 1;
            }
            else {
               trace("$enable .... success.");
            }
         }

         writeCkpt("ROOTCRS_OKAUNINST", CKPTSUC);

      }
      elsif ( $rc != 2 )
      {
         if ((scalar(grep(/9118/, @output))) || 
             (scalar(grep(/9119/, @output))) ||
             (scalar(grep(/9348/, @output))) > 0)
         {
            # We couldn't unload the old drivers or load the new ones.
            trace("OKA drivers unable to be uninstalled.");
            if ($CFG->DOWNGRADE)
            {    
               print color 'bold';
               print_error(2003);
               print color 'reset';
               
               my $disable = "$crsctl disable $has";
               trace("$disable ... disabling CRS in preparation for reboot.");
               @output = system_cmd_capture($disable);
               $rc = shift @output;

               if ($rc != 0)
               {
                   trace("$disable ... unable to disable CRS for reboot.");
                   # Don't fail here - the user should correct
                   # the disable, by running it by hand, and reboot.
                   print_error(2004);
               }
               else
               {
                   trace("$disable ... CRS disabled, ready for reboot.");
                   # At this point, CRS is disabled.  The user will need
                   # to reboot, rerun the operation.  This will rely on 
                   # OPatch and OUI to print the appropriate error message.
               }
            }

            writeCkpt("ROOTCRS_OKAUNINST", CKPTFAIL);
            exit 1;
         }   
      }
      else 
      {
         trace ("@cmd ... failed");
         writeCkpt("ROOTCRS_OKAUNINST", CKPTFAIL);
         exit 1;
      }
   }
}

=head2 actionOKADriversResource

   Create/Start/Delete the OKA Drivers Resource. 

=head3 Parameters

   $action - "add"/"start"/"delete" actions for the drivers resource

=head3 Returns

  TRUE  - Creation/Start/Delete successful
  FALSE - Creation/Start/Delete failed

=head3 Notes


=cut
sub actionOKADriversResource
{
  my $action  = shift;
  my $crsctl  = catfile ($CFG->ORA_CRS_HOME, "bin", "crsctl");
  my $ret     = FAILED;

  if (isOKASupported()) 
  { 
    trace("Doing KA resource actions");

    my @out  = system_cmd_capture($crsctl, "stat", "res", "ora.drivers.oka",
                                  "-init");
    my $notExists = grep (/CRS-2613/i, @out);
    my $status    = grep (/TARGET/i, @out);

    if (scalar($notExists) > 0 && 
        ($action eq "add" || $action eq "start"))
    {
      my $user          = $CFG->params('ORACLE_OWNER');;
      my $grp           = $CFG->params('ORA_DBA_GROUP');
      my $owner         = $user;
      
      if ($CFG->platform_family ne "windows")
      {
        $owner = $CFG->SUPERUSER;
      }
      
      # NOTE: oka_attr will be used only for 'add' command.
      my @oka_attr = 
          ("ACL='owner:$owner:rwx,pgrp:$grp:r-x,other::r--," .
           "user:$user:r-x'","ENABLED=0");
      $ret = crsohasd::crsctlResource("add", 
                                      "ora.drivers.oka", 
                                      "ora.oka.type", \@oka_attr);
    }
    elsif (scalar($status) > 0  && $action eq "delete")
    {
      $ret = crsohasd::crsctlResource($action, "ora.drivers.oka");
    }
    else
    {
      # Make sure 'crsctl stat res' command ran properly
      if (scalar($status) > 0 || scalar($notExists) > 0) 
      {
        # The resource already exists or was already deleted  
        $ret = SUCCESS; 
      }
      else
      { 
        # There was an error running the command
        $ret = FAILED; 
      }
    }

    if ($action eq "start")
    {
      # Get the status again just to be sure  
      @out  = system_cmd_capture($crsctl, "stat", "res", "ora.drivers.oka",
                                 "-init");
      my $statusOnline  = grep (/TARGET=ONLINE/i,  @out);
      my $statusOffline = grep (/TARGET=OFFLINE/i, @out);
      
      if (scalar($statusOnline) > 0) 
      {
        # The resource has been started already  
        $ret = SUCCESS; 
      }
      elsif (scalar($statusOffline) > 0 )
      { 
        # The resource has not been started 
        $ret = crsohasd::crsctlResource($action, "ora.drivers.oka");
      }
      else
      {
        # The command failed or resource was not created
        $ret = FAILED;
      }
    }
  }
  else
  {
    # If OKA not supported,
    # then assume everything is okay.
    trace("OKA is not supported on this platform.");
    $ret = SUCCESS;
  }

  return $ret;
}

=head2 isOKASupported

   Determines if this platform is an OKA supported platform
   by calling 'okadriverstate supported'.

=head3 Parameters

   None

=head3 Returns

  TRUE  - OKA Supported
  FALSE - OKA Not Supported

=head3 Notes


=cut
sub isOKASupported
{
   my $OKA_supported = FALSE;
   my $okadriverstate;

   if ($CFG->platform_family eq 'windows') {
      $okadriverstate = catfile($CFG->ORA_CRS_HOME, 'bin', 'okadriverstate.bat');
   }
   else {
      $okadriverstate = catfile($CFG->ORA_CRS_HOME, 'bin', 'okadriverstate');
   }

   # check if OKA is supported
   if (! (-e $okadriverstate)) {
      trace ("$okadriverstate not found");
      return FALSE;
   }

   my @cmd = ($okadriverstate, "supported");
   my @out = system_cmd_capture(@cmd);
   my $rc  = shift @out;

   if ($rc == 0) {

      # In dev env, we disable KA.
      if (!is_dev_env() || ($ENV{'ORA_ENABLE_OKA_INSTALL'} eq "true")) {
         $OKA_supported = TRUE;
         trace ("OKA is supported");
      }
      else {
         $OKA_supported = FALSE;
         trace ("OKA is not supported");
      }
   }
   else {
      $OKA_supported = FALSE;
      trace ("OKA is not supported");
   }

  return $OKA_supported;
}

