#!/usr/local/bin/perl
# 
# $Header: has/install/crsconfig/crsoc4j.pm /st_has_12.1.0.1/1 2013/01/09 02:29:32 jgrout Exp $
#
# crsoc4j.pm
# 
# Copyright (c) 2008, 2013, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      crsoc4j.pm - Library module for OC4J root install/upgrade functions.
#
#    DESCRIPTION
#      crsoc4j.pm - Contains installation and upgrade
#                   routines for OC4J.
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#    jgrout      01/08/13 - Fix bug 16075946
#    jgrout      10/24/12 - Fix bug 14741962
#    jmunozn     07/12/12 - Use qosctl -autogenerate to add QoS users
#    gmaldona    07/29/11 - Initial Creation
# 

package crsoc4j;

use strict;
use English;
use File::Temp qw/ tempfile /;
use File::Spec::Functions;
use File::Find ();

use crsutils;

use Exporter;

use constant WALLET_NAME  => "APPQOSADMIN";
use constant USR_OC4J     => "oc4jadmin";
use constant USR_QOS      => "qosadmin";

use vars qw(@ISA @EXPORT @EXPORT_OK);
@ISA = qw(Exporter);

my @exp_func  = qw(add_J2EEContainer addWallet_J2EEContainer 
                   start_J2EEContainer enable_J2EEContainer
                   stop_J2EEContainer disable_J2EEContainer);

push @EXPORT, @exp_func; 

# OSD API definitions
use s_crsutils;

sub add_J2EEContainer
#---------------------------------------------------------------------
# Function: Add the OC4J Container.
#           This function has a checkpoint that wrap the procedure of 
#           add oc4j.
# Args    : none 
#---------------------------------------------------------------------
{
   my $oc4j_status   = FAILED;
   my $srvctl = crs_exec_path('srvctl');
   my $user = $CFG->params('ORACLE_OWNER');
   my @cmd  = ($srvctl, 'add', 'oc4j');
   my @out  = ();
   my $rf   = run_as_user2($user, \@out, @cmd);
   
   #We have success with these conditions:
   # - ($rf == 0) : The oc4j container was added with success.
   # - PRCR-1086 : resource ora.oc4j is already registered
   #now proceed to add the wallet.
   if (($rf == 0) || (scalar(grep(m/PRCR-1086/i, @out)) > 0)) {
      if ($rf == 0) {
         trace("J2EE (OC4J) Container Resource Add ... passed  ...");
      } else {
         trace("J2EE (OC4J) Container Resource Add ... container exists ...");
         # Stop the container if it is running
         if (stop_J2EEContainer() == FAILED) {
           # FIXME - When we can add a message for OC4J stop
           # failing, use that message instead of 1001
           print_error(1001);
           return $oc4j_status;
         }
      }
      $oc4j_status = addWallet_J2EEContainer(); 
   } else {
      trace(join(">  ", ("Error encountered in the command @cmd \n", 
            @out, "End Command output")));
      trace("J2EE (OC4J) Container Resource Add ... failed ...");
      print_error(1001);
   }

   return $oc4j_status;
}

sub addWallet_J2EEContainer
#---------------------------------------------------------------------
# Function: Add the wallet for OC4J Container
# Args    : none
#---------------------------------------------------------------------
{   
   # Status variables
   my $oc4j_addwallet_status = FAILED;
   my $wallet_exist          = FALSE;
   my $query_wallet_error    = FALSE;
   # Verify the existence of the wallet
   my $user   = $CFG->params('ORACLE_OWNER');
   my $crsctl = crs_exec_path('crsctl');
   my @cmd    = ($crsctl, 'query', 'wallet', '-type', WALLET_NAME);
   my @out    = ();
   my $rc     = run_as_user2($user, \@out, @cmd);
   
   # search in the out if the wallet not exists in output query
   if (scalar(grep(m/CRS-4030/i, @out)) > 0) {
      # This code means the wallet exists
      $wallet_exist = TRUE;
   } elsif (scalar(grep(m/CRS-4031/i, @out)) > 0) {
      # This code means the wallet not exists
      $wallet_exist = FALSE;
   } else {
      $query_wallet_error = TRUE;
   }
   
   if ($query_wallet_error == TRUE) {
      trace(join(">  ", ("Error encountered in the command @cmd \n", 
            @out, "End Command output")));
      trace ("J2EE (OC4J) Container Resource Query Wallet ... failed ...");
   } elsif ($wallet_exist == FALSE) {
      # The wallet does not exist, we must now add the wallet
      my @cmd2 = ($crsctl, 'add', 'wallet', '-type', WALLET_NAME);
      my @out2 = ();
      my $rc2  = run_as_user2($user, \@out2, @cmd2);

      if ($rc2 == 0) {
         trace ("J2EE (OC4J) Container Resource Add Wallet ... passed ...");
         $oc4j_addwallet_status = addQosUsers();
      } else {
         trace(join(">  ", ("Error encountered in the command @cmd2 \n", 
               @out2, "End Command output")));
         trace ("J2EE (OC4J) Container Resource Add Wallet ... failed ...");
      }
   } elsif ($wallet_exist == TRUE) {
      # the wallet exists
      trace ("J2EE (OC4J) Container Resource Add Wallet ... already exists ...");
      $oc4j_addwallet_status =  addQosUsers();
   }
   
   if ($oc4j_addwallet_status == FAILED) {
      print_error(1002);
   }

   return $oc4j_addwallet_status;
}

sub addQosUsers
#------------------------------------------------------------------------------
# Function: Adds the QoS users (oc4jadmin and qosadmin).
# Args    : none.
# Returns : SUCCESS in case the users are added succesfully (or already exist).
#           FAILED in case of any error.
#------------------------------------------------------------------------------
{
   my $rc = FAILED;
   my $user = $CFG->params('ORACLE_OWNER');
   my @cmd = (get_qosctl_path(), '-autogenerate');
   my @out = ();
   my $rf = run_as_user2($user, \@out, @cmd);

   if ( $rf == 0 ) {
      trace("qosctl output: " . join("\n", @out));
      if ( verifyQosUser(USR_OC4J) == SUCCESS && 
           verifyQosUser(USR_QOS) == SUCCESS ) {
         $rc = SUCCESS;      
      }
   } else {
      trace(join(">  ", ("Error encountered in the command @cmd \n", 
            @out, "End Command output")));
   }
   return $rc;
}

sub verifyQosUser
#------------------------------------------------------------------------------
# Function: Verifies existence of a QoS users.
# Args    : [0] user name to be verified.
# Returns : SUCCESS in case the user exists, FAILED otherwise
#------------------------------------------------------------------------------
{
   my $rc = FAILED;
   my $user = $CFG->params('ORACLE_OWNER');
   my $userName = $_[0];
   my @cmd = (crs_exec_path('crsctl'), 'query', 'wallet', '-type',
              WALLET_NAME,  '-user', $userName);
   my @out = ();
   my $rf = run_as_user2($user, \@out, @cmd);
   
   if (scalar(grep(m/CRS-4028/i, @out)) > 0) {
      $rc = SUCCESS;
   } else {
      trace(join(">  ", ("Error encountered in the command @cmd \n", 
            @out, "End Command output")));
      trace("Verify QoS user $userName failed");
   }
   return $rc;
}

sub start_J2EEContainer
#---------------------------------------------------------------------
# Function: Start the OC4J Container
# Args    : none
#---------------------------------------------------------------------
{
   my $oc4j_status = FAILED;
   my $oc4j_enable_start = TRUE;
   
   # Start oc4j on Windows (bug 9900672)
   # Disable oc4j on HP-UX (bug 12648797)
   # Disable oc4j on shiphome without management DB (bug 14741962)
   if (($OSNAME eq 'hpux') ||
       ((is_dev_env() == FALSE) &&
        !(($CFG->defined_param('MGMT_DB')) &&
          ($CFG->params('MGMT_DB') =~ m/true/i))))
   {
      $oc4j_enable_start = FALSE;
   }

   # Start the OC4J resource if oc4j_enable_start=TRUE
   if ($oc4j_enable_start) {
      # Enable oc4j resource
      $oc4j_status = enable_J2EEContainer();
      if ($oc4j_status == SUCCESS) {
         # Start oc4j resource
         my $user = $CFG->params('ORACLE_OWNER');
         my $srvctl = crs_exec_path('srvctl');
         my @cmd = ($srvctl, 'start', 'oc4j');
         my @out  = ();
         my $status = run_as_user2($user, \@out, @cmd);
      
         #We have success with these conditions:
         # - ($status == 0) : OC4J started 
         # - PRCC-1014 : OC4J is already running
         if (($status == 0) || (scalar(grep(m/PRCC-1014/i, @out)) > 0)) {
            $oc4j_status = SUCCESS;
            trace ("J2EE (OC4J) Container Resource Start ... passed ...");
         } else {
            $oc4j_status = FAILED;
            trace(join(">  ", ("Error encountered in the command @cmd \n", 
                  @out, "End Command output")));
            trace("J2EE (OC4J) Container Resource Start ... failed ...");
            print_error(1003);
         }
      }  
   } 
   else {
      # Disable the OC4J resource if oc4j_enable_start = FALSE
      $oc4j_status = disable_J2EEContainer();
   }

   return $oc4j_status;
}

sub enable_J2EEContainer
#---------------------------------------------------------------------
# Function: Enable OC4J Container
# Args    : none
#---------------------------------------------------------------------
{
   my $oc4j_status   = FAILED;
   
   # Enable OC4J resource during upgrade 
   # - From 11.2.0.1 or 11.2.0.2 to 11.2.0.3
   # - From 11.2.0.x to 12.1
   my $srvctl = crs_exec_path('srvctl');
   my $user = $CFG->params('ORACLE_OWNER');
   my @cmd  = ($srvctl, 'enable', 'oc4j');
   my @out  = ();
   my $rf   = run_as_user2($user, \@out, @cmd);
   
   # We have success with these conditions:
   # - ($rf == 0) : OC4J was successfully enabled
   # - PRKO-2116 : OC4J is already enabled
   if (($rf == 0) || (scalar(grep(m/PRKO-2116/i, @out)) > 0)) {
      $oc4j_status = SUCCESS;
      trace("J2EE (OC4J) Container Resource enable ... passed");
   } else {
      trace(join(">  ", ("Error encountered in the command @cmd \n", 
            @out, "End Command output")));
      trace("J2EE (OC4J) Container Resource enable ... failed");
      print_error(1004);
   }

   return $oc4j_status;
}

sub disable_J2EEContainer
#---------------------------------------------------------------------
# Function: Disable OC4J Container
# Args    : none
#---------------------------------------------------------------------
{
   my $oc4j_status   = FAILED;
   #  Disable OC4J resource.
   my $srvctl = crs_exec_path('srvctl');
   my $user = $CFG->params('ORACLE_OWNER');
   my @cmd  = ($srvctl, 'disable', 'oc4j');
   my @out  = ();
   my $rf   = run_as_user2($user, \@out, @cmd);
   
   # We have success with these conditions:
   # - ($rf == 0) : This means the OC4J was successfully disabled.
   # - PRKO-2115 : OC4J is already disabled
   if (($rf == 0) || (scalar(grep(m/PRKO-2115/i, @out)) > 0)) {
      $oc4j_status = SUCCESS;
      trace("J2EE (OC4J) Container Resource Disable ... passed");
   } else {
      trace(join(">  ", ("Error encountered in the command @cmd \n", 
            @out, "End Command output")));
      trace("J2EE (OC4J) Container Resource Disable ... failed ...");
      print_error(1005);
   }

   return $oc4j_status;
}

sub stop_J2EEContainer
#---------------------------------------------------------------------
# Function: Stop the OC4J Container
# Args    : none
#---------------------------------------------------------------------
{
   my $oc4j_status = FAILED;
   
   # Stop oc4j resource
   my $user = $CFG->params('ORACLE_OWNER');
   my $srvctl = crs_exec_path('srvctl');
   my @cmd = ($srvctl, 'stop', 'oc4j');
   my @out  = ();
   my $status = run_as_user2($user, \@out, @cmd);
      
   #We have success with these conditions:
   # - ($status == 0) : OC4J stopped
   # - PRCC-1016 : OC4J is already stopped
   if (($status == 0) || (scalar(grep(m/PRCC-1016/i, @out)) > 0)) {
      $oc4j_status = SUCCESS;
      trace ("J2EE (OC4J) Container Resource Stop ... passed ...");
   } else {
      $oc4j_status = FAILED;
      trace(join(">  ", ("Error encountered in the command @cmd \n", 
            @out, "End Command output")));
      trace("J2EE (OC4J) Container Resource Stop ... failed ...");
      # FIXME - When we can add a message for OC4J stop
      # failing, uncomment the print_error and issue the message

      # print_error(nnnn);
   } 

   return $oc4j_status;
}
