# Copyright (c) 2011, 2013, Oracle and/or its affiliates. All rights reserved. 
#
#   NAME
#      crsohasd.pm
#
#   DESCRIPTION
#      This module contains ohasd related functions
#
#   NOTES
#      <other useful comments, qualifications, etc.>
#
#   MODIFIED   (MM/DD/YY)
#   minzhu      04/17/13 - 16444109, workaround for only 12.1.0.1
#   ssprasad    11/13/12 - Disable OKA actions for 12.1
#   xyuan       09/04/12 - Remove bounce_ohasd() from sub
#                          perform_configure_hasd
#   xyuan       08/03/12 - Fix bug 14392140
#   xyuan       07/16/12 - Fix bug 14328234
#   rdasari     06/07/12 - add KA resource
#   rdasari     05/30/12 - delete configuration on rerun
#   rdasari     05/09/12 - add/delete the hub category based on $action
#   anjiwaji    05/08/12 - Pass in the action parameter to 
#                          createACFSDriversResource and rename to 
#                          actionACFSDriversResource
#   hkanchar    04/11/12 - Update stop dependencies for storage in remote asm
#   hkanchar    04/27/12 - Update dependencies
#   rdasari     03/21/12 - create ora.hub.category
#   rtamezd     03/13/12 - Fix bug 13840985
#   rtamezd     03/01/12 - Fix bug 13699947
#   hkanchar    01/11/12 - Update action entries for ohasd asm
#   xyuan       01/05/12 - Call s_install_initd if it's SuSE Linux
#   anjiwaji    01/04/12 - Add start action to crsctlResource and export it
#   anjiwaji    12/14/11 - Call createACFSDriversResource in configure_hasd
#   minzhu      12/01/11 - bug 12730228 - asm/haip hard dep
#   ysharoni    11/30/11 - gpnpd intermediate, start/stop depends
#   xyuan       11/03/11 - Fix bug 13328777
#   xyuan       10/20/11 - XbranchMerge ksviswan_bug-12580496 from
#                          st_has_11.2.0.3.0
#   xyuan       10/20/11 - XbranchMerge ksviswan_bug-12630162 from
#                          st_has_11.2.0.3.0
#   rdasari     09/07/11 - change ocr storage dependencies
#   xyuan       06/10/11 - add ohasdbase type, ocrstorage resource
#   lmortime    05/25/11 - Making EVMD start first
#   dpham       03/28/11 - New for 12c
#
package crsohasd;

use strict;
use English;
use Exporter;
use File::Copy;
use File::Path;
use File::Find;
use File::Basename;
use File::Spec::Functions;
use Sys::Hostname;
use POSIX qw(tmpnam);
use Carp;
use Socket;
use Env qw(NLS_LANG);
use Term::ANSIColor;

# root scripts modules
use oraacfs;
use crska;
use crsutils;
use s_crsutils;

use constant CMDSUC               =>  '0';
use constant CMDFAIL              =>  '1';

use constant ROOTCRS_INITRES      =>  "ROOTCRS_INITRES";
use constant ROOTCRS_OHASD        =>  "ROOTCRS_OHASD";

use Exporter;
use vars qw(@ISA @EXPORT @EXPORT_OK);

@ISA = qw(Exporter);

my @exp_func  = qw(perform_register_ohasd register_ohasd perform_start_ohasd
                   perform_configure_hasd configure_hasd crsctlResource
                  );

push @EXPORT, @exp_func;

####---------------------------------------------------------
#### Function for registering ohasd service
# ARGS: 0
sub perform_register_ohasd
{
   trace ("Registering ohasd");
   my $srv      = 'ohasd';
   my $ckptName = ROOTCRS_OHASD;
   my $ckptStatus;

   if (isCkptexist($ckptName)) {
      $ckptStatus = getCkptStatus($ckptName);
      trace("'$ckptName' state is $ckptStatus");

      if (($ckptStatus eq CKPTSTART) || ($ckptStatus eq CKPTFAIL))
      {
         trace("Unregistering OHASD service configirations");
         s_unregister_service('ohasd');
         $CFG->isNodeCrashed(FALSE);
      }
      elsif ($ckptStatus eq CKPTSUC) {
         trace("ohasd already registered");
         $CFG->wipCkptName("ROOTCRS_STACK");
         return SUCCESS;
      }
   }

   writeCkpt($ckptName, CKPTSTART);
   $CFG->wipCkptName($ckptName);

   register_ohasd();
   trace("ohasd is now registered");
}

sub register_ohasd
{
   my $rc = s_register_service('ohasd');

   if ($rc == SUCCESS) {
      s_install_initd() or die(dieformat(317));
      writeCkpt(ROOTCRS_OHASD, CKPTSUC);
   }
   else {
      writeCkpt(ROOTCRS_OHASD, CKPTFAIL);
   }

   $CFG->wipCkptName("ROOTCRS_STACK");
   return $rc;
}

sub perform_start_ohasd
{
   trace ("Starting ohasd");
   my $srv = 'ohasd';

   # Check if the service/daemon has started
   trace ("Checking the status of $srv");
   my $ohasdStatus = check_service ($srv, 3, FALSE);

   if ($ohasdStatus) {
      trace ("$srv is already running");
   }
   else {
      trace ("$srv is not already running.. will start it now");
      start_service($srv);
   }

   # Check if the service/daemon has started
   trace ("Checking the status of $srv");
   $ohasdStatus = check_service ($srv, 360);

   if ($ohasdStatus) {
      trace ("$srv started successfully");
   }
   else {
      trace ("$srv has failed to start");
      exit 1;
   }
}

sub perform_configure_hasd
{
  my $mode     = $_[0];
  my $hostname = $_[1];
  my $owner    = $_[2]; # the owner of the software bits aka CRS USER
  my $pusr     = $_[3]; # the privileged user 
  my $grp      = $_[4];

  my $ckptName = "ROOTCRS_INITRES";
  my $ckptStatus;

  if (isCkptexist($ckptName))
  {
    $ckptStatus = getCkptStatus($ckptName);

    trace("'$ckptName' state is $ckptStatus");

    if (($ckptStatus eq CKPTSTART) || ($ckptStatus eq CKPTFAIL))
    {
       trace("Removing OHASD init resources and types");
       configure_hasd('crs', $hostname, $owner, $pusr, $grp, "delete");
       $CFG->isNodeCrashed(FALSE);
    } elsif ($ckptStatus eq CKPTSUC) {
       trace("OHASD Resources are already configured");
       $CFG->wipCkptName("ROOTCRS_STACK");
       return SUCCESS;
    }
  }
  else
  {
    trace("The checkpoint '$ckptName' does not exist");
    if (($CFG->isRerun) || areResTypeRegistered())
    {
      trace("Removing OHASD init resources and types");
      configure_hasd('crs', $hostname, $owner, $pusr, $grp, "delete");
    }
  }

  writeCkpt($ckptName, CKPTSTART);
  $CFG->wipCkptName($ckptName);

  configure_hasd('crs', $hostname, $owner, $pusr, $grp);

  if (FAILED == setCurrentNodeRole())
  {
    writeCkpt($ckptName, CKPTFAIL);
    exit(1);
  }
  else
  {
    trace("Successfully set the role for the local node");
  }

  writeCkpt($ckptName, CKPTSUC);
  $CFG->wipCkptName("ROOTCRS_STACK");

  return SUCCESS;
}

####----------------------------------------------------------
#### Function for checking if any OHASD resources have existed
# ARGS: None 
# Returns: TRUE or FALSE 
sub areResTypeRegistered
{
  my @output;
  my $rc = -1;
  my $ckptName = "ROOTCRS_INITRES";
  my $crsctl = crs_exec_path('crsctl');

  trace("Check if OHASD resource types have been registered");
  @output = system_cmd_capture($crsctl, "stat", "type", "-init");
  $rc = shift @output;

  if ($rc != 0)
  {
    error("Failed to check status of init resource types");
    return FALSE;
  }

  my @typeNm = grep (/TYPE_NAME/, @output);
  foreach my $type (@typeNm)
  {
    my @val = split(/=/, $type);

    if ($val[1] =~ /ora\..+\.type/)
    {
      my $res = $val[1];
      trace("A resource type with the name '$res' is registered");
      return TRUE;
    }
  }

  trace("Not found any registered resource type");
  return FALSE;
}


sub crsctlResource
{
  my ($action, $resName, $resType, $resAttr, $force) = @_;

  trace("$action '$resName' ..."); 

  my $crsctl = crs_exec_path('crsctl'); 
  if ($action eq "add")
  {
    my @cmd = ($crsctl, "add", "resource", $resName,
                "-attr", '"', join(',', @{$resAttr}), '"',
                "-type", $resType, "-init", $force);

    if (! is_dev_env() && ($CFG->platform_family eq "windows"))
    {
      push @cmd, '-buildowner';
    }

    my $status = system_cmd(@cmd);
    if ($status != CMDSUC)
    {
      writeCkpt(ROOTCRS_INITRES, CKPTFAIL);
      die(dieformat(271, $resName)); 
    }
  }
  elsif ($action eq "start")
  {
    system_cmd_capture ($crsctl, "start", "resource", $resName, "-init");
  }
  elsif ($action eq "delete")
  {
    system_cmd_capture($crsctl, "delete", "resource", $resName, "-f",
      "-init");
  }
  else
  {
    trace("unknown action '$action' for '$resName'");
    return FAILED;
  }

  return SUCCESS;
}

sub addResourceType
{
  my ($type, $baseType, $templateFile, $ohasdlog) = @_; 

  trace("Registering resource type '$type'");

  my $ORACLE_HOME = $ENV{'ORACLE_HOME'};
  my $CRS_HOME_TEMPLATE = catdir($ORACLE_HOME, "crs", "template");
  my $infile  = catfile($CRS_HOME_TEMPLATE, $templateFile);
  my $outfile = catfile($ohasdlog, $type);
  instantiateTemplate($infile, $outfile);
  
  my $crsctl = crs_exec_path('crsctl');
  my $status = system_cmd($crsctl, "add", "type", $type, "-basetype",
                 $baseType, "-file", "$outfile", "-init");
  if ($status != CMDSUC)
  {
    trace("Failed to add type '$type'");
    writeCkpt(ROOTCRS_INITRES, CKPTFAIL);
    die(dieformat(270, $type));    
  }
  else
  {
    trace("Type '$type' added successfully");
  }

  unlink($outfile); 
}

sub configure_hasd {
  my $mode     = $_[0];
  my $hostname = $_[1];
  my $owner    = $_[2]; # the owner of the software bits aka CRS USER
  my $pusr     = $_[3]; # the privileged user 
  my $grp      = $_[4];
  my $action   = $_[5];
  my @out      = ();
  my $status;

  # Register these resources for SIHA only
  my @registerTypesHAS = ("cssd", "evm");

  # Register these resources for clusterware only
  my @registerTypesCRS = ("evm", "mdns", "gpnp", "gipc", "cssd", "cssdmonitor",
                          "crs", "ctss", "crf", "asm", "drivers.acfs");
 
  if ($CFG->platform_family eq "windows") {
     if (! $owner) {
        $pusr = '';
        $grp  = '';
     }
  } else {
     push(@registerTypesHAS, "diskmon");
     push(@registerTypesCRS, "diskmon");
  }
      
  my $ORACLE_HOME = $ENV{'ORACLE_HOME'};

  # Make sure CRS_HOME is set
  if ( ! $ORACLE_HOME ) {
    writeCkpt(ROOTCRS_INITRES, CKPTFAIL);
    die(dieformat(268));
  }

  # Set Homes
  my $CRS_HOME_BIN = catdir($ORACLE_HOME,"bin");
  my $crsctl = catfile( $CRS_HOME_BIN, "crsctl");

  my @types;
  if ($mode eq "has") {
    @types = @registerTypesHAS;
  } elsif ($mode eq "crs") {
    @types = @registerTypesCRS;
  }

  my $ohasdbaseType = 'ora.ohasdbase.type';
  my $daemonType = 'ora.daemon.type';
  my $ocrstorageType = 'ora.storage.type';
  my $haipType = 'ora.haip.type';
  my $file;
  my $name;
  my $logdir = catdir($CFG->ORA_CRS_HOME, "log", $hostname);
  my $ohasdlog = catdir($logdir, "ohasd");

  #set default action to add
  if (! $action) {
     $action = "add";
  }

  if ($action eq "add")
  {
    addResourceType($ohasdbaseType, "cluster_resource", "ohasdbase.type", $ohasdlog);
    addResourceType($daemonType, $ohasdbaseType, "daemon.type", $ohasdlog);

    if ($mode eq "crs")
    {
      addResourceType($ocrstorageType, $ohasdbaseType, "storage.type", $ohasdlog);
    }

    if (isHAIPsupported())
    {
      addResourceType($haipType, "cluster_resource", "haip.type", $ohasdlog);
      # as part of the resource addition, also fixup any system config files
      s_CheckNetworkConfig();
    }
  }

  # register the infrastructure resources
  foreach my $type (@types)
  {
    $file = $type . '.type';
    $name = 'ora.' . $type . '.type';

    if ($action eq "add")
    {
      addResourceType($name, $daemonType, $file, $ohasdlog);
    }
  }

  # acting on resources
  if ($mode eq "crs")
  {
    configure_ohasd_CRS($owner, $pusr, $grp, $action);
  } 
  elsif ($mode eq "has")
  {
    configure_ohasd_SIHA($owner, $pusr, $grp, $action);
  }

  if (($action eq "delete") && isHAIPsupported())
  {
    trace("Unregistering type '$haipType'");
    @out = system_cmd_capture($crsctl, "delete","type", $haipType, "-init");
  }

  if ($action eq "delete")
  {
     trace("Deleting base types");

     foreach my $type (@types)
     {
       $name = 'ora.' . $type . '.type';
       @out =  system_cmd_capture($crsctl, "delete","type", $name, "-init");
     }

     @out = system_cmd_capture($crsctl, "delete","type", $daemonType, "-init");

     if ($mode eq "crs")
     {
       trace("Unregistering type '$ocrstorageType'");
       system_cmd_capture($crsctl, "delete", "type", $ocrstorageType, "-init");
     }

     trace("Unregistering type '$ohasdbaseType'");
     system_cmd_capture($crsctl, "delete", "type", $ohasdbaseType, "-init");
  } 

  return TRUE;
}

sub configure_ohasd_SIHA
{
  my ($owner, $pusr, $grp, $action) = @_;

  # set the owners : user IDs to spawn agents as
  my $CSSOWNER  = $pusr;
  my $EVMOWNER  = $owner;

  # set the users : explit execution rules (may or may not be equal to owner)
  my $CSSUSER  = $owner;
  my $EVMUSER  = $owner;

  my @css_attr = ("CSS_USER=$CSSUSER");
  my @diskmon_attr = ("USR_ORA_ENV=ORACLE_USER=$CSSUSER");
  my @evm_attr = ("ACL='owner:$EVMOWNER:rw-,pgrp:$grp:rw-," .
                       "other::r--,user:$EVMUSER:rwx'");
  if ($ENV{'CSSDAGENT_ATTR'}) { push @css_attr, $ENV{'CSSDAGENT_ATTR'}; }

  # SI-HA cssd does not depend on mdnsd/gpnpd
  push @css_attr, "ACL='owner:$CSSOWNER:rwx,pgrp:$grp:rwx,other::r--'";
  push @css_attr, "RESTART_ATTEMPTS=5";
  push @diskmon_attr, "ACL='owner:$CSSOWNER:rwx,pgrp:$grp:rwx,other::r--'";

  if ($CFG->platform_family ne "windows")
  {
    push @css_attr, "START_DEPENDENCIES='weak(concurrent:ora.diskmon)'";
    push @css_attr, "STOP_DEPENDENCIES='hard(shutdown:ora.diskmon)'";
    push @diskmon_attr, "START_DEPENDENCIES='weak(concurrent:ora.cssd)" .
                        "pullup:always(ora.cssd)'";

    crsctlResource($action, "ora.diskmon", "ora.diskmon.type", \@diskmon_attr);
  }

  crsctlResource($action, "ora.cssd", "ora.cssd.type", \@css_attr);
  crsctlResource($action, "ora.evmd", "ora.evm.type", \@evm_attr);
}

sub configure_ohasd_CRS
{
  my ($owner, $pusr, $grp, $action) = @_;

  # set the owners : user IDs to spawn agents as
  my $MDNSOWNER = $owner;
  my $GPNPOWNER = $owner;
  my $GIPCOWNER = $owner;
  my $CSSOWNER  = $pusr;
  my $EVMOWNER  = $owner;
  my $CRSOWNER  = $pusr;
  my $OCRSTORAGEOWNER = $pusr;

  # set the users : explit execution rules (may or may not be equal to owner)
  my $MDNSUSER = $owner;
  my $GPNPUSER = $owner;
  my $GIPCUSER = $owner;
  my $CSSUSER  = $owner;
  my $EVMUSER  = $owner;
  my $CRSDUSER = $owner;
  my $OCRSTORAGEUSER = $owner;

  my @evm_attr = ("ACL='owner:$EVMOWNER:rw-,pgrp:$grp:rw-," .
                       "other::r--,user:$EVMUSER:rwx'"); 
  my @asm_attr = ("ACL='owner:$CSSUSER:rw-,pgrp:$grp:rw-," .
                       "other::r--,user:$CSSUSER:rwx'");
  my @ocrstorage_attr = ("ACL='owner:$OCRSTORAGEOWNER:rw-,pgrp:$grp:r--," .
                              "other::r--,user:$OCRSTORAGEUSER:r-x'");
  my @css_attr    = ("CSS_USER=$CSSUSER");
  my @crsd_attr   = ("ACL='owner:$CRSOWNER:rw-,pgrp:$grp:rw-," .
                          "other::r--,user:$CRSDUSER:r-x'");
  my @css_attrmon = ("CSS_USER=$CSSUSER",
                     "ACL='owner:$CSSOWNER:rw-,pgrp:$grp:rw-," .
                          "other::r--,user:$CSSUSER:r-x'");
  my @diskmon_attr = ("USR_ORA_ENV=ORACLE_USER=$CSSUSER");

  if ($ENV{'CSSDAGENT_ATTR'}) { push @css_attr, $ENV{'CSSDAGENT_ATTR'}; }
  if ($ENV{'CSSDAGENT_ATTR'}) { push @css_attrmon, $ENV{'CSSDAGENT_ATTR'}; }
  
  push @css_attr, "ACL='owner:$CSSOWNER:rw-,pgrp:$grp:rw-," .
                       "other::r--,user:$CSSUSER:r-x'"; 
  push @css_attr, "AUTO_START=always";
  push @asm_attr, "ACTIONS=CRSDASM_START";

  if ($CFG->platform_family eq "windows")
  {
    push @css_attr, "START_DEPENDENCIES='hard(ora.gpnpd,ora.gipcd,ora.cssdmonitor)" .
                                        "pullup(ora.gpnpd,ora.gipcd)'";
    push @css_attr, "STOP_DEPENDENCIES='hard(intermediate:ora.gipcd,intermediate:ora.cssdmonitor)'";
  }
  else
  {
    push @css_attr, "START_DEPENDENCIES='weak(concurrent:ora.diskmon)" .
                                        "hard(ora.cssdmonitor,ora.gpnpd,ora.gipcd)" .
                                        "pullup(ora.gpnpd,ora.gipcd)'";
    push @css_attr, "STOP_DEPENDENCIES='hard(intermediate:ora.gipcd,shutdown:ora.diskmon,intermediate:ora.cssdmonitor)'";

    push @diskmon_attr, "START_DEPENDENCIES='weak(concurrent:ora.cssd)" .
                                            "pullup:always(ora.cssd)'";
  }

  push @diskmon_attr, "ACL='owner:$CSSOWNER:rw-,pgrp:$grp:rw-," .
                           "other::r--,user:$CSSUSER:r-x'";

  my @mdns_attr = ("ACL='owner:$MDNSOWNER:rw-,pgrp:$grp:rw-,other::r--,user:$MDNSUSER:rwx'");
  crsctlResource($action, "ora.mdnsd", "ora.mdns.type", \@mdns_attr);

  my @gpnp_attr = ("ACL='owner:$GPNPOWNER:rw-,pgrp:$grp:rw-,other::r--,user:$GPNPUSER:rwx'");
  crsctlResource($action, "ora.gpnpd", "ora.gpnp.type", \@gpnp_attr, "-f");

  my @gipc_attr = ("ACL='owner:$GIPCOWNER:rw-,pgrp:$grp:rw-,other::r--,user:$GIPCUSER:rwx'");
  crsctlResource($action, "ora.gipcd", "ora.gipc.type", \@gipc_attr);

  if ($CFG->platform_family ne "windows") 
  {
    crsctlResource($action, "ora.diskmon", "ora.diskmon.type", \@diskmon_attr);
  }
  
  crsctlResource($action, "ora.cssdmonitor", "ora.cssdmonitor.type", \@css_attrmon, "-f");
  crsctlResource($action, "ora.cssd", "ora.cssd.type", \@css_attr); 
  crsctlResource($action, "ora.evmd", "ora.evm.type", \@evm_attr); 

  my @ctss_attr = ("ACL='owner:$CRSOWNER:rw-,pgrp:$grp:rw-,other::r--,user:$CRSDUSER:r-x'");
  crsctlResource($action, "ora.ctssd", "ora.ctss.type", \@ctss_attr);

  if (isHAIPsupported())
  {
    my @haip_attr = ("ACL='owner:$CRSOWNER:rw-,pgrp:$grp:rw-,other::r--,user:$CRSDUSER:r-x'");
    crsctlResource($action, "ora.cluster_interconnect.haip", "ora.haip.type", \@haip_attr);
  }

  if (isCRFSupported())
  {
    my @crf_attr = ("ACL='owner:$CRSOWNER:rw-,pgrp:$grp:rw-,other::r--,user:$CRSDUSER:r-x'");
    crsctlResource($action, "ora.crf", "ora.crf.type", \@crf_attr);
  }

  # ora.ctssd dependency only needed for cluster and not for siha
  if (isHAIPsupported() )
  {
    if( ($OSNAME eq "hpux") || ($OSNAME eq "aix") )
    {
        push @asm_attr, "START_DEPENDENCIES='hard(ora.cssd,ora.ctssd)" .
          "pullup(ora.cssd,ora.cluster_interconnect.haip,ora.ctssd)" .
            "weak(ora.cluster_interconnect.haip,ora.drivers.acfs)'";
    }
    else
    {
        push @asm_attr, "START_DEPENDENCIES='hard(ora.cssd,ora.cluster_interconnect.haip,ora.ctssd)" .
          "pullup(ora.cssd,ora.cluster_interconnect.haip,ora.ctssd)" .
            "weak(ora.drivers.acfs)'";
    }
    push @asm_attr, "STOP_DEPENDENCIES='hard(intermediate:ora.cssd," .
                                    "shutdown:ora.cluster_interconnect.haip)'";
  }
  else
  {
    push @asm_attr, "START_DEPENDENCIES='hard(ora.cssd,ora.ctssd)" .
                                        "pullup(ora.cssd,ora.ctssd)" .
                                        "weak(ora.drivers.acfs)'";
  }

  my $crsctl = crs_exec_path('crsctl');
  my $rc;
  if ($action eq "add")
  {
    $rc = system_cmd($crsctl, 'add', 'category', 'ora.hub.category', '-init',
                         '-attr', '"ACTIVE_CSS_ROLE=hub"');
  }

  if ($rc != 0)
  {
    trace("Failed to '$action' ora.hub.category");
  }

  # When OCR is on ASM, add ora.asm as a HARD and PULLUP start dependency
  # These need to be consistent with :
  # has/crs/template/crs.type
  # prou.c
  if (($CFG->ASM_STORAGE_USED) || ( $CFG->UPGRADE && isOCRonASM() ))
  {
    if (isLegacyASM())
    {
      push @ocrstorage_attr, "START_DEPENDENCIES='hard(ora.asm,ora.cssd)pullup(ora.asm,ora.cssd)'";
      push @ocrstorage_attr, "STOP_DEPENDENCIES='hard(shutdown:ora.asm,intermediate:ora.cssd)'";
    }
    elsif (isNearASM())#near ASM only, for far ASM use default
    {
      push @ocrstorage_attr, "START_DEPENDENCIES='hard(ora.cssd)weak(ora.asm)pullup(ora.asm,ora.cssd)pullup(ora.ctssd)'";
      push @ocrstorage_attr, "STOP_DEPENDENCIES='hard(shutdown:ora.asm,intermediate:ora.cssd)'";
    }
  }

  crsctlResource($action, "ora.asm", "ora.asm.type", \@asm_attr);
  crsctlResource($action, "ora.storage", "ora.storage.type", \@ocrstorage_attr);
  crsctlResource($action, "ora.crsd", "ora.crs.type", \@crsd_attr);

  # Unregister 'ora.hub.category' after deleting ora.asm as it is referenced by resource 'ora.asm'
  if ($action eq "delete")
  {
    system_cmd_capture($crsctl, 'delete', 'category', 'ora.hub.category', '-init');
  }


  # add or delete the ACFS resource
  if ( FAILED == actionACFSDriversResource($action) && $action eq "add")
  { 
    print_error(425);
  }

  #trace("KA resources action");
  # add or delete the KA resource
  #if ( FAILED == actionOKADriversResource($action) && $action eq "add")
  #{ 
  #  print_error(2006);
  #}
}

#----------------------( instantiateTemplate )--------------------------#
#                                                                       #
#                                                                       #
#  FUNCTION: instantiateTemplate                                        #
#                                                                       #
#  PURPOSE: Instantiates the cap file with the CRS HOME location        #
#                                                                       #
#-----------------------------------------------------------------------#
sub instantiateTemplate
{
  my $ORACLE_HOME = $CFG->params('ORACLE_HOME');

  my ($inFile, $outFile) = @_;

  #TODO Define this based on platforms
  my $FSEP = '/';

  # If I can read the template or cap, instantiate the file replacing any 
  # special values
  if ( -r $inFile) 
  {
    open (INF, "<", "$inFile") or
        fatal("Unable to open $inFile, $!, ");

    # Make sure to open output file safely
    if ( -r $outFile ) 
    {
      trace("Removing pre existing $outFile from a previous run.");
      unlink($outFile) or 
          die(dieformat(168, $outFile, $!));
    }
    open (OUTF, ">", "$outFile") or
        die(dieformat(207, $outFile, $!));

    # 
    # Filter Transformations
    #
    while (<INF>) 
    {
      # Modify CRS Home
      s/%ORA_CRS_HOME%/$ORACLE_HOME/;

      # Modify File Separators for NT
      s/\/\//$FSEP/g;
      print OUTF $_;
    }
    close(INF) or
        die(dieformat(272, $inFile, $!));

    close(OUTF) or
        die(dieformat(272, $outFile, $!));

  } 
  else 
  {
    print_error(32, $inFile);
    trace("$inFile is not readable.",
          "Verify the value of ORACLE_HOME, ");
    die(dieformat(273, $inFile));
  }

}

1;
