# Copyright (c) 2011, 2013, Oracle and/or its affiliates. All rights reserved. 
#
#   NAME
#      crsupgrade.pm
#
#   DESCRIPTION
#      This module contains upgrade related functions for CRS/SIHA
#
#   NOTES
#      <other useful comments, qualifications, etc.>
#
#   MODIFIED   (MM/DD/YY)
#   sidshank    02/18/13 - fix bug 16169568
#   sidshank    02/04/13 - fix bug 16275273.
#   sidshank    01/16/13 - fix bug 16175611
#   jgrout      01/08/13 - Fix bug 16075946
#   xyuan       12/25/12 - Fix bug 16014163 - disable listener changes for
#                          Windows
#   xyuan       12/20/12 - Fix bug 16026674
#   xyuan       12/20/12 - XbranchMerge xyuan_bug-15969912 from main
#   xyuan       12/20/12 - XbranchMerge xyuan_bug-15994179 from main
#   xyuan       12/13/12 - Fix bug 15994179
#   xyuan       12/09/12 - Start listener for joining node
#   rdasari     11/14/12 - Fail if crsctl startupgrade fails
#   ssprasad    11/13/12 - Disable OKA actions for 12.1
#   spavan      11/09/12 - fix bug15848256 - add checkpoints for CVU commands
#                          on last node
#   xyuan       10/30/12 - Fix bug 14804296 - Call asmca -joinCluster on
#                          joining node
#   xyuan       10/29/12 - Wait for CRSD to come up during the rerun of 'crsctl
#                          startupgrade'
#   xyuan       10/17/12 - Fix bug 14648503
#   spavan      10/17/12 - fix bug14771917 - store config info in check point
#   rdasari     10/17/12 - modify upgCreateCredDomain
#   spavan      10/09/12 - fix bug14739163 - isCVUConfigured should use old crs
#                          home
#   xyuan       10/09/12 - Fix bug 14683232
#   spavan      09/27/12 - fix bug14218734 - check cvu config
#   epineda     09/27/12 - Bugfix 13539130
#   xyuan       09/18/12 - Fix bug 14603264
#   xyuan       09/18/12 - Fix bug 14621340 - make error in
#                          removing CVU fatal
#   shullur     09/17/12 - For adding check of version greater than 12 in case
#                          of bdb
#   xyuan       09/13/12 - Fix bug 14603434
#   jmunozn     09/11/12 - Execute addWallet_J2EEContainer() in first node when
#                          not isOldVersionLT112()
#   xyuan       09/11/12 - Fix bug 14592705
#   ssprasad    09/05/12 - Fix bug 14576870 - invoke OKA install
#                          action in upgrade scripts.
#   dpham       08/21/12 - Add XAG component
#   rdasari     08/16/12 - write OLD_CSSD_LOGLEVEL in ckpt
#   xyuan       08/14/12 - Fix bug 14487626
#   xyuan       08/08/12 - Fix bug 9584563
#   xyuan       08/03/12 - Fix bug 14392140
#   xyuan       08/01/12 - Fix bug 13730408
#   xyuan       07/25/12 - Make few trace messages in preForceUpgChecks go to
#                          the console
#   rdasari     07/20/12 - wait for stack start after set active version
#   rdasari     07/10/12 - do not add asm on a asm client cluster
#   rtamezd     07/02/12 - Use trace_lines()
#   xyuan       06/20/12 - Fix bug 14202360
#   xyuan       06/20/12 - Fix bug 14026786
#   shullur     06/15/12 - For handling default case of bdb location
#   rdasari     06/02/12 - change default to root automation for Windows
#   sidshank    05/23/12 - fix bug 14106919
#   sidshank    05/22/12 - fix bug 14099962
#   xyuan       05/16/12 - Cache the value for
#                          isFirstNodeToUpgrade/isLastNodeToUpgrade
#   shullur     05/09/12 - For fixing the exists in case of failure of copy of
#                          orafiles
#   sidshank    05/03/12 - remove s_redirect/restore output subroutines
#   xyuan       05/01/12 - Remove modifyClusterName because the installer will
#                          instantiate the CLUSTER_NAME in crsconfig_params
#                          after the bug 12681659 is fixed 
#   sidshank    04/25/12 - fix for bug 14004260
#   xyuan       04/24/12 - Implementation for adding the local node to an
#                          upgraded cluster
#   xyuan       04/24/12 - Remove asmca invocation on the last node
#   rdasari     04/19/12 - bounce ohasd after creating ohasd resources
#   gmaldona    03/16/12 - adding oc4j setup
#   rtamezd     03/05/12 - Fix bug 13803068
#   rtamezd     02/28/12 - Fix bug 13401742
#   akhaladk    02/28/12 - Bug 13785751
#   xyuan       02/24/12 - Fix bug 13770508
#   xyuan       02/14/12 - Do not change permissions on EXTERNAL_ORACLE if .ssh
#                          exists under it during upgrade
#   xyuan       02/09/12 - Fix bug 13612321 & 12710059
#   sidshank    02/09/12 - Remove call to first_node_tasks()
#   rtamezd     02/07/12 - Move upgradeModelResType to crsutils.pm
#   xyuan       02/02/12 - Force upgrade support
#   xyuan       02/01/12 - Remove old ASM files when upgrading from
#                          11.2.0.1/11.2.0.2
#   xyuan       01/30/12 - Create the credential domains for ASM and Gridhome
#                          during upgrade
#   rsirigan    01/27/12 - export upgrade_OCR for pcw_upgrade_nodes macro 
#   xyuan       01/12/12 - Fix bug 13426221
#   shullur     01/06/12 - XbranchMerge shullur_bug-12976590 from st_has_11.2.0
#   spavan      01/06/12 - fix bug 13401742
#   xyuan       12/25/11 - Fix bug 13532044
#   gsanders    12/23/11 - Eliminate ACFS registry. Check upgrade_acfs_registry ret
#   rtamezd     12/21/11 - Updated call to add_GNS
#   shullur     12/12/11 - XbranchMerge shullur_bug-12673460 from st_has_11.2.0
#   xyuan       12/13/11 - Removed unnecessary comments
#   xyuan       12/10/11 - Fix bug 13435658
#   xyuan       12/06/11 - Fix bug 13435843
#   xyuan       12/04/11 - Fix bug 13091719
#   agraves     11/30/11 - Update reboot necessary mechanism to be a failure,
#                          not a continuation and a suggestion.
#   shullur     11/03/11 - XbranchMerge shullur_bug-12614795 from st_has_11.2.0
#   shullur     11/03/11 - XbranchMerge shullur_bug_11852891 from st_has_11.2.0
#   rvadraha    10/25/11 - Bug13247694, Fix ocfs upgrade
#   xyuan       11/14/11 - Fix bug 13340630
#   xyuan       11/07/11 - Forward merge fixes for bug 12344535 & 12640551
#   shullur     11/03/11 - XbranchMerge shullur_bug_11852891 from st_has_11.2.0
#   xyuan       10/31/11 - Fix bug 13328777
#   xyuan       09/29/11 - Forward merge fix for bug 12640884 
#   xyuan       09/20/11 - Fix bug 12737227
#   priagraw    08/19/11 - remove call to clsfmt
#   xyuan       08/04/11 - XbranchMerge xyuan_bug-12795595 from
#                          st_has_11.2.0.3.0
#   xyuan       08/03/11 - XbranchMerge xyuan_bug-12698968 from
#                          st_has_11.2.0.3.0
#   xesquive    07/26/11 - Add functions set_bold and reset_bold instead of
#                          print color
#   xyuan       07/25/11 - Rename create_crs_resources to 
#                          create_ohasd_resources
#   rdasari     07/15/11 - do not use avlookup for siha
#   lmortime    06/27/11 - Make EVMD start first
#   spavan      06/19/11 - remove and add cvu in upgrade
#   dpham       05/23/11 - Modulize upgrade function
#   rdasari     05/20/11 - remove ocr check from preUpgradeChecks
#   dpham       03/28/11 - New for 12c
#
package crsupgrade;

use strict;
use English;
use File::Copy;
use File::Path;
use File::Find;
use File::Basename;
use File::Spec::Functions;
use Sys::Hostname;
use POSIX qw(tmpnam);
use Carp;
use Socket;
use Env qw(NLS_LANG);
use Env qw(SRVM_TRACE);
use Env qw(ORACLE_HOME);

use constant FIRST_NODE           => "first";
use constant LAST_NODE            => "last";

# root script modules
use crsinstall;
use crsutils;
use crsgpnp;
use crsoc4j;
use crsohasd;
use oracss;
use oraacfs;
use s_crsutils;

use Exporter;
use vars qw(@ISA @EXPORT @EXPORT_OK);

@ISA = qw(Exporter);

my @exp_func = qw(get_oldconfig_info upgrade_olr perform_upgrade_config upgrade_OCR
                  perform_CHM_upgrade);

push @EXPORT, @exp_func;

my @force_upgrade_nodes;

sub new {
   shift;
   crsutils->new(@_);

   if ($ENV{'ORA_AUTO_WIN'} eq "false") {
      # redirect stdout/stderr as appropriate for Windows only
      s_redirect_souterr($CFG->crscfg_trace_file . "_OUT");
   }

   rscPreChecks();

   # upgrade
   if ($CFG->SIHA) {
      HAUpgrade();
   }
   else {
      CRSUpgrade();
   }

   if ($ENV{'ORA_AUTO_WIN'} eq "false") {
      # restore stdout/stderr as appropriate
      s_restore_souterr();
   }
}

sub CRSUpgrade
{
   my $success = TRUE;

   # instantiate scripts
   instantiate_scripts();

   # run directory creation, files creation/permissions modules
   setup_env();

   create_starting_ckpt_entry();

   get_oldconfig_info();

   precheck();

   preUpgChecks();

   if (($CFG->FORCE) && (FAILED == preForceupgChecks()))
   {
     die(dieformat(429));
   }

   save_param_file();

   $ENV{'ORA_PCW_AGENTS'} = "true"; # FIXME: This should not be required

   if ($CFG->platform_family eq "windows") {
	   s_osd_setup();
   }

   trace("Querying previous clusterware configuration");
   queryClusterConfig();

   if (!preUpgradeChecks()) {
      print_error(362);
      exit 1;
   }

   prepare_to_upgrade_clusterware();

   upgrade_clscfg();

   # configure node
   $success = upgrade_node();

   if (isOldVersionLT112())
   {
      # set permissions on voting disks
      if (! setperm_vdisks()) {
         trace ("Failed to set permission on voting disks during upgrade");
      }

      if ($CFG->FORCE)
      {
        trace("Removing the ASM node resources of un-upgraded nodes ...");
        if ( ! removeASMNodeRes()) { exit(1); }
      }

      if ($CFG->JOIN)
      {
        trace("Calling asmca on node that is joining the cluster ...");
        if (! asmcaJoinCluster()) { die(dieformat(452)); } 
      }
   }

   upgCreateCredDomains();

   backupOLR();

   configureCvuRpm();

   install_xag();

   cleanASMFiles();

   createMgmtdbDir();

   if (($CFG->platform_family eq "unix") && ($CFG->JOIN))
   {
     trace("Starting listeners on node that is joining the cluster ...");
     $success = startListeners(TRUE);
   }

   if ($success) {
      set_bold();
      print_error(325);
      reset_bold();
      writeCkpt("ROOTCRS_STACK", CKPTSUC);

      # Environment variable to be set only on Windows, and only for
      # temporary backward compatibility of using crssetup for remote nodes.
      if ($ENV{'ORA_AUTO_WIN'} eq "false") {
      # configure remote nodes
      configureRemoteNodes();
      }

      if (1 == $CFG->SUCC_REBOOT) {
        set_bold();
        print_error(411);
        reset_bold();
      }
   }
   else {
      set_bold();
      print_error(326);
      reset_bold();
      exit 100;
   }
}

sub HAUpgrade
{
   precheck_siha();

   # instantiate scripts
   instantiate_scripts();

   # run directory creation, files creation/permissions modules
   setup_env();

   if (isCkptexist("ROOTHAS_ACFSINST"))
   {
     trace("Re-run 'acfsroot install' to complete acfs driver install");
     goto ACFS_INST;
   }

   if ($CFG->platform_family eq "windows") {
	   s_osd_setup();
   }

   prepare_to_upgrade_siha();

   # if old SI CSS was running when roothas.pl was invoked *or* ASM needs to be
   # upgraded,
   # a) For post-11.2 upgrades, restart the stack so that ohasd starts all autostart
   #    resources (ASM, DB). CSS gets started as a dependency here.
   # b) For pre-11.2 upgrades, start SIHA CSS
   # CSS needs to be up for ASMCA to configure/start the new ASM.
   if ($CFG->restart_css ||
      ($CFG->params('ASM_UPGRADE') =~ m/true/i))
   {
     if (! isOldVersionLT112())
     {
       trace("Upgrading from a post 11.2 version");
       # Enable ora.cssd
       my $crsctl = catfile($CFG->ORA_CRS_HOME, 'bin', 'crsctl');
       my @cmd = ($crsctl, 'modify', 'res', 'ora.cssd', '-attr',
                   "\"ENABLED=1\"", '-init');
       my $status = system_cmd(@cmd);
       if (0 != $status)
       {
         die("Failed to enable CSSD startup");
       }
      
       if ($CFG->platform_family ne "windows") {
         # Enable ora.diskmon
         @cmd = ($crsctl, 'modify', 'res', 'ora.diskmon', '-attr',
                  "\"ENABLED=1\"", '-init');
         $status = system_cmd(@cmd);
         if (0 != $status)
         {
           die("Failed to enable DISKMON startup");
         }
       }

       trace("Restarting the stack");
       # restart the ohasd so that all its managed resources start with the new profiles
       stopOracleRestart() or
         die "Failed to stop the Oracle Restart stack";
   
       start_service ("ohasd", $CFG->HAS_USER) or
         die "ohasd failed to start";
     }
     else
     {
       trace("Upgrading from a pre 11.2 version, starting CSS");
       if (!CSS_start()) {
          print_error(201);
          exit 1;
       }
     }
   }

   #invoke upgrademodel for SIHA
   if (! isOldVersionLT112())
   {
      upgsihamodel();
   }

   # add ons by default for SIHA
   if (! isONSexist()) {
      add_ons();
   }

   # start evmd in SIHA mode to replace eonsd functionality
   if (!start_resource("ora.evmd", "-init")) {
      print_error(202);
      exit 1;
   }

   # backup OLR
   my $ocrconfig = catfile($CFG->params('ORACLE_HOME'), 'bin', 'ocrconfig');
   my $rc        = system ("$ocrconfig -local -manualbackup");

   if ($rc == 0) {
      trace ("$ocrconfig -local -manualbackup ... passed");
   }
   else {
      trace ("$ocrconfig -local -manualbackup ... failed");
   }

   cleanASMFiles();  

ACFS_INST:
   perform_installUSMDriver(0);

   remove_checkpoints();

   print_error(327);
}

sub prepare_to_upgrade_clusterware
{
   # Check if CRS is already configured
   my $crsConfigured   = FALSE;
   my $gpnp_setup_type = GPNP_SETUP_BAD;

   check_CRSConfig($CFG->ORA_CRS_HOME,
                   $CFG->HOST,
                   $CFG->params('ORACLE_OWNER'),
                   $CFG->params('ORA_DBA_GROUP'),
                   $CFG->params('GPNPGCONFIGDIR'),
                   $CFG->params('GPNPCONFIGDIR'),
                   $crsConfigured,
                   $gpnp_setup_type) || die(dieformat(315));

   # Verify bdb location. Initially its in CRS HOME. oclumon changes later
   if (isCRFSupported() && !is_dev_env() && !isReposBDB()) {
      my $bdbloc = catfile($CFG->ORA_CRS_HOME, 'crf', 'db', $CFG->HOST);
      s_crf_check_bdbloc($bdbloc, $CFG->params('NODE_NAME_LIST'));
   }

   # In case of upgrade crs stack is always configured. But to upgrade to
   # 11.2 we still need to do same things as we do for fresh install.
   # TODO: going forward when we upgrade from 11.2 to higher versions
   # this step may not be required.

   # Check existing configuration, Create and populate OLR and OCR keys
   upgrade_olr_config() || die(dieformat(316));

   if (isOldVersionLT112())
   {
      # Initialize the SCR settings.
      s_init_scr ();

      # incase of upgrade voting files would be upgraded from previous version
      # create GPnP wallet/profile
      initialize_local_gpnp($CFG->HOST, $gpnp_setup_type);
   }

   create_IPDOS_config();

   # Setup OHASD service/daemon
   perform_register_ohasd() || die(dieformat(317));

   # Start OHASD service/daemon
   perform_start_ohasd() || die(dieformat(318));

   # create OHASD resources
   create_ohasd_resources();

   # install USM driver
   perform_installUSMDriver(1);

   #
   # NOTE: This is commented out for 12.1 Post 12.1
   # branching, it will be enabled.
   #
   # install KA driver. Supported only in cluster.
   #perform_installKADriver();

   # upgrade CSS config, propagate cluster-wide config if needed
   if (! perform_upgrade_config($gpnp_setup_type)) {
      my $trace_file = $CFG->crscfg_trace_file;
      print_error(197);
      trace ("Cluster configuration upgrade failed. " .
             "collect the  $trace_file and contact Oracle Support");
      exit 1;
   }

   # Start Oracle Clusterware stack
   upgrade_cluster();
}

sub upgrade_cluster
{
   set_logging();
   if ($CFG->OLD_CSSD_LOG_LEVEL)
   {
     set_CSSD_loglevel($CFG->OLD_CSSD_LOG_LEVEL);
   }

   # Start EVM
   if (! start_resource("ora.evmd", "-init")) {
      print_error(117);
      die(dieformat(250));
   }

   # Start gpnpd
   if (! start_clusterware(START_STACK_GPNPD)) {
      print_error(117);
      die(dieformat(242));
   }

   # Upgrade the CSS voting disks.Only valid for Pre112 upgrades
   # Note: cssvfupgd can only be executed once during an upgrade, hence only
   # call it if this is the first node to upgrade
   if ((isOldVersionLT112()) && (isFirstNodeToUpgrade()) &&
       (! CSS_upgrade($CFG)))
   {
      print_error(117);
      die(dieformat(243));
   }

   # Start CSS in clustered mode
   if (! CSS_start_clustered()) {
      print_error(117);
      die(dieformat(244));
   }

   # Start CTSS with reboot option to signal step sync
   # Note: Before migrating stack startup to 'crsctl start crs',
   #       'CTSS_REBOOT=TRUE' is a workaround to signal step sync.
   if (! start_resource("ora.ctssd", "-init",
                        "-env", "USR_ORA_ENV=CTSS_REBOOT=TRUE"))
   {
      print_error(117);
      die(dieformat(245));
   }

   # Start CRF
   if (! isCRFSupported() || ! start_resource("ora.crf", "-init")) {
      trace ("start_clster: Failed to start CRF");
      # We don't want IPD failure to stop rest of the stack
      # from coming up. So, no exit here!
   }

   # startup the HAIP if it has been configured
   enable_HAIP();

   # Start ASM if needed. Start if old version is 11.2 and ocr/vd is on ASM
   if ((! isOldVersionLT112()) && (isOCRonASM()) &&
      (! start_resource("ora.asm", "-init")))
   {
      print_error(117);
      die(dieformat(247));
   }

   if (! upgrade_OCR()) {
      print_error(117);
      die(dieformat(248));
   }

   # Start CRS
   if (! start_resource("ora.crsd", "-init")) {
      print_error(117);
      die(dieformat(249));
   }

   if (!wait_for_stack_start(360)) {
      die(dieformat(251));
   }

   print_error(343);
}

sub upgrade_OCR {
  my $ret       = TRUE;
  my $oraowner  = $CFG->params('ORACLE_OWNER');
  my $oraDbaGrp = $CFG->params("ORA_DBA_GROUP");

  # ocrconfig - Create OCR keys
  trace ("Creating or upgrading OCR keys");
  my $status = run_crs_cmd('ocrconfig', '-upgrade', $oraowner, $oraDbaGrp);

  if ($status == 0)
  {
    trace ("OCR keys are successfully populated");
    s_reset_srvconfig () or die(dieformat(295));
  } else {
    trace ("ocrconfig -upgrade failed with status $status");
    print_error(157);
    $ret = FALSE;
  }

  return $ret;
}

sub prepare_to_upgrade_siha
{
   my $local_config_exists = local_only_config_exists();

   if (! isOldVersionLT112()) {
      $local_config_exists = FALSE;
      performpost112upg()
   }

   # delete olr file if exists
   if (-e $CFG->OLR_LOCATION) {
      if (isOldVersionLT112())
      {
         if ($CFG->DEBUG) { trace ("unlink " . $CFG->OLR_LOCATION); }
         unlink ($CFG->OLR_LOCATION) || print_error(102, $CFG->OLR_LOCATION, $!);
      }
   }

   validate_olrconfig($CFG->OLR_LOCATION,
                      $CFG->ORA_CRS_HOME) || die(dieformat(292));

   if (! $local_config_exists) {
      createLocalOnlyOCR();
   }

   chdir(catdir($CFG->ORA_CRS_HOME, 'log'));
   upgrade_local_OCR();

   upgrade_olr();

   # Delete ohasd resources for SIHA for 11.2 upgrade
   my $del_ohasd_res;
   if (! isOldVersionLT112()) { $del_ohasd_res = TRUE; }

   start_or_upgrade_siha($local_config_exists, $del_ohasd_res);
}

sub upgrade_olr_config
{
   my $ckptName = "ROOTCRS_OLR";

   if (isOLRConfigured()) {
      return SUCCESS;
   }

   if (! $CFG->SIHA) {
      initial_cluster_validation();
   }

   upgrade_local_OCR();

   upgrade_olr();

   writeCkpt($ckptName, CKPTSUC);
   print("OLR initialization - successful\n");
   return SUCCESS;
}

=head2 perform_upgrade_config

   Upgrades configuration and pushes cluster-wide gpnp setup

=head3 Parameters

   The parameter hash

=head3 Returns

  TRUE  - A  configuration was found and upgraded
  FALSE - No configuration was found or upgraded

=cut

sub perform_upgrade_config
{
   my $gpnp_setup_type = $_[0];
   my $success         = TRUE;

   if ($CFG->DEBUG) { trace("OCR upgraded; gpnp setup type: $gpnp_setup_type"); }

   if ((isOldVersionLT112()) &&
       ($gpnp_setup_type != GPNP_SETUP_GOTCLUSTERWIDE) &&
       ($gpnp_setup_type != GPNP_SETUP_CLUSTERWIDE))
   {
      # Push local gpnp setup to be cluster-wide.
      # This will copy local gpnp file profile/wallet setup to a
      # list of cluster nodes, including current node.
      # This promotes a node-local gpnp setup to be "cluster-wide"
      trace ("Promoting local GPnP setup to cluster-wide. Nodes " .
             $CFG->params('NODE_NAME_LIST'));

      if (! push_clusterwide_gpnp_setup($CFG->params('NODE_NAME_LIST'))) {
         print_error(151);
         $success = FALSE;
      }
   }
   else {
      trace ("Skipping push GPnP configuration cluster-wide");
   }

  return $success;
}

sub upgrade_olr
{
   my $OCRCONFIGBIN = catfile ($CFG->ORA_CRS_HOME, "bin", "ocrconfig");
   my $CLSCFGBIN    = catfile ($CFG->ORA_CRS_HOME, "bin", "clscfg");
   my $ckptName     = "ROOTCRS_OLR";
   my @out          = ();
   my $status;
   my $rc = FALSE;

   # upgrade keys in OLR
   if (! isOldVersionLT112()) 
   {
      trace ("$CLSCFGBIN -localupgrade");
      if ($CFG->SIHA) {
         $status = run_as_user($CFG->params('ORACLE_OWNER'),
                               "$CLSCFGBIN -localupgrade");
      }
      else {
         @out = system_cmd_capture("$CLSCFGBIN -localupgrade");
         $status = shift @out;
      }
   }
   else {
      # Upgrading from pre 11.2. OLR does not exist. Use info from existing OCR
      # to populate OLR. Create OLR from scratch.
      if ($CFG->SIHA) {
        $status = run_as_user($CFG->params('ORACLE_OWNER'),"$CLSCFGBIN -localadd");
      }
      else {
        @out = system_cmd_capture("$CLSCFGBIN -localadd -avlookup");
        $status = shift @out;
      }
   }

   if (0 == $status) {
      trace ("Keys created in the OLR successfully");
      writeCkpt($ckptName, CKPTSUC);
      $rc = TRUE;
      $CFG->wipCkptName("ROOTCRS_STACK");
   }
   else {
      trace("Failed to create keys in the OLR, rc = $status, Message:\n ".
             "@out \n");
      writeCkpt($ckptName, CKPTFAIL);
      die(dieformat(188));
   }

   return $rc;
}

sub get_oldconfig_info 
{
  if (! $CFG->UPGRADE)
  {
    trace("Not a upgrade scenario");
    return;
  }

  trace ("Retrieving older cluster configuration data");
  my $oldCrsHome;
  my $ckptName = "ROOTCRS_OLDHOMEINFO";
  my $ckptStatus;
  my @oldCrsVer;
  my $verstring;

  setConfiguredCRSHome();

  $CFG->wipCkptName($ckptName);

  if (isCkptexist($ckptName)) {
     $ckptStatus = getCkptStatus($ckptName);
  } else {
     writeCkpt($ckptName, CKPTSTART);
  }

  # Get Old CRS Home
  if ($ckptStatus eq CKPTSUC) {
     $oldCrsHome = getCkptPropertyValue($ckptName, "OLD_CRS_HOME");
     $oldCrsHome = trim($oldCrsHome);
     trace("old crs home from ckpt property is $oldCrsHome");
  } else {
     # Get old CRS home
     $oldCrsHome = getConfiguredCRSHome();
  }

  if (! $oldCrsHome)
  {
    trace("Failed to retrieve old Grid Infrastructure home location");
    writeCkpt($ckptName, CKPTFAIL);
    $CFG->wipCkptName("ROOTCRS_STACK");
    die(dieformat(398)); 
  }

  trace("Old CRS Home = $oldCrsHome");
  $CFG->oldconfig('ORA_CRS_HOME', $oldCrsHome);
  $CFG->OLD_CRS_HOME($oldCrsHome);

  if (isOCRonASM()) {
      trace("copying gpnp files to use new home binaries");
      copy_gpnpfiles($CFG->OLD_CRS_HOME, $CFG->ORA_CRS_HOME);
  }

  # Get old CRS version
  if ($ckptStatus eq CKPTSUC) {
     $verstring = getCkptPropertyValue($ckptName, "OLD_CRS_VERSION");
     $verstring = trim($verstring);
     @oldCrsVer = split(/\./, $verstring);
     trace("old crs version from ckpt property is @oldCrsVer");
  }
  else
  {
     if ($CFG->JOIN)
     {
       # Note: the active version was already changed before joining.
       my $relver = getcursoftversion($CFG->HOST);
       @oldCrsVer = split(/\./, $relver);
     }
     else
     {
       # Get old CRS version (actually active version), 
       # use the configured CRS home 
       @oldCrsVer = get_crs_version(getConfiguredCRSHome());
     }
  }

  $CFG->oldconfig('ORA_CRS_VERSION', \@oldCrsVer);
  trace("The active version of the Oracle Clustereware is '@oldCrsVer'");

  my $verinfo = join('.', @oldCrsVer);
  trace("Version Info retreived is : $verinfo");

  # Get old CSSD log level
  my $lvl = -1;
  if ($ckptStatus eq CKPTSUC && isCkptPropertyExists($ckptName, "OLD_CSSD_LOGLEVEL")) 
  {
    $lvl = getCkptPropertyValue($ckptName, "OLD_CSSD_LOGLEVEL");
    $lvl = trim($lvl);
    trace("Old CSSD log level from ckpt property is $lvl");
  }
  else
  {
    $lvl = get_CSSD_loglevel_fromOld();
    if (-1 == $lvl)
    { 
      trace("Unable to retrieve CSSD log level from the old stack");
    } 
  }

  if (-1 != $lvl)
  { 
    trace("Old CSSD log level is $lvl");
    $CFG->OLD_CSSD_LOG_LEVEL($lvl);
  }

  my $isFirstNodeToUpgrade = FALSE;
  if (isFirstNodeToUpgrade())
  {
    $isFirstNodeToUpgrade = TRUE;
  }

  if ($isFirstNodeToUpgrade)
  {
    # The old stack should be always up
    if (CKPTSUC eq $ckptStatus)
    {
      trace("Attempt to get old configuration info from the ckpt property");
    }
    else
    {
      if (FAILED == check_OldCrsStack())
      {
        trace("Make sure the older stack is completely down");
        stopClusterware($oldCrsHome, "crs") || die(dieformat(349));
        trace("The old stack isn't up, try to bring it up");
        start_OldCrsStack();
      }
    }
  }

  # If old CRS version < 11.2, the NETWORKS/GUID/OCRID is required
  # for initializing GPnP on the local node.
  my $oldClusterID = -1;
  my $oldOCRID = -1;
  my ($networks, $nodes);

  if ($isFirstNodeToUpgrade && isOldVersionLT112())
  {
    if (CKPTSUC eq $ckptStatus)
    {
      if (isCkptPropertyExists($ckptName, "CLUSTER_GUID"))
      {
        $oldClusterID = getCkptPropertyValue($ckptName, "CLUSTER_GUID");
        $oldClusterID = trim($oldClusterID);
      }
      if (isCkptPropertyExists($ckptName, "OCRID"))
      {
        $oldOCRID = getCkptPropertyValue($ckptName, "OCRID");
        $oldOCRID = trim($oldOCRID);
      }
      $networks =  getCkptPropertyValue($ckptName, "NETWORKS");
      $networks = trim($networks);
    }
    else
    {
      # Get cluster GUID, use the configured CRS home
      $oldClusterID = get_clusterguid(getConfiguredCRSHome());
      # Get cluster OCRID, use the configured CRS home
      $oldOCRID = get_ocrid(getConfiguredCRSHome());
      # populate $NETWORK/NODE_NAME_LIST info for upgrade
      ($networks, $nodes) = get_upgrade_netinfo();
    }

    if (! $networks)
    {
      trace("Failed to retrieve old network configuration");
      writeCkpt($ckptName, CKPTFAIL);
      $CFG->wipCkptName("ROOTCRS_STACK");
      die(dieformat(399));
    } 

    $CFG->oldconfig('CLUSTER_GUID', $oldClusterID) unless (-1 == $oldClusterID);
    $CFG->oldconfig('OCRID', $oldOCRID) unless (-1 == $oldOCRID);
    $CFG->oldconfig('NETWORKS', $networks);
    $CFG->params('NETWORKS', $networks);

    trace ("  old ClusterID =$oldClusterID");
    trace ("  old OCRID     =$oldOCRID");
    trace ("  old networks  =$networks");
  }

  # If old CRS version >= 11.2, the nodeList/localNode is required 
  # for ASM rolling/non-rolling upgrade on the first node
  my ($nodeList, $localNode, $asmConf);
  $asmConf = -1;
  if (($isFirstNodeToUpgrade) && (! isOldVersionLT112()))
  {
    if (CKPTSUC eq $ckptStatus)
    {
      $localNode = trim(getCkptPropertyValue($ckptName, "LOCAL_NODE"));
      $nodeList  = trim(getCkptPropertyValue($ckptName, "NODE_LIST"));
      $asmConf   = trim(getCkptPropertyValue($ckptName, "ASM_CONFIGURED"));
    }
    else
    {
      $localNode = getLocalNodeName(getConfiguredCRSHome());
      $nodeList  = getNodeNumList(getConfiguredCRSHome());
      $asmConf   = isASMConfigured(getConfiguredCRSHome());
    }

    if ((! $localNode) || (! $nodeList) || (-1 == $asmConf))
    {
      trace("Failed to retrieve the required information for ASM upgrade");
      writeCkpt($ckptName, CKPTFAIL);
      $CFG->wipCkptName("ROOTCRS_STACK");
      exit 1;
    }
   
    $CFG->oldconfig('LOCAL_NODE', $localNode);
    $CFG->oldconfig('NODE_LIST', $nodeList);
    $CFG->oldconfig('ASM_CONFIGURED', $asmConf);

    trace("  localNode =<$localNode>");
    trace("  nodeList  =<$nodeList>");
    trace("  asmconf   =$asmConf");
  }

  trace ("  old CrsHome        =$oldCrsHome");
  trace ("  old CrsVer         =@oldCrsVer");
  trace ("  old CSSD log level = $lvl");

  if ($ckptStatus ne CKPTSUC)
  {
     trace("Write old configuation data into the checkpoint file");
     writeCkptProperty($CFG->wipCkptName, "OLD_CRS_HOME", $oldCrsHome);
     writeCkptProperty($CFG->wipCkptName, "OLD_CRS_VERSION", $verinfo);
     if (-1 != $lvl)
     {
       writeCkptProperty($CFG->wipCkptName, "OLD_CSSD_LOGLEVEL", $lvl);
     }

     if (isOldVersionLT112())
     {
       if (-1 != $oldClusterID)
       {
         writeCkptProperty($CFG->wipCkptName, "CLUSTER_GUID", $oldClusterID);
       }
       if (-1 != $oldOCRID)
       {
         writeCkptProperty($CFG->wipCkptName, "OCRID", $oldOCRID);
       }
       if ($networks)
       {
         writeCkptProperty($CFG->wipCkptName, "NETWORKS", $networks);
       }
     }
     else
     {
       if ($localNode)
       {  
         writeCkptProperty($CFG->wipCkptName, "LOCAL_NODE", $localNode);
       }
       if ($nodeList)
       {
         writeCkptProperty($CFG->wipCkptName, "NODE_LIST", $nodeList);
       }
       if (-1 != $asmConf)
       {
         writeCkptProperty($CFG->wipCkptName, "ASM_CONFIGURED", $asmConf);
       }
     }
  }

  # get cvu resource values when old stack is running or else get from check point
  my $cvuCheckInterval = "0";
  if ($isFirstNodeToUpgrade) {
     if (isCVUConfigured($oldCrsHome) || CKPTSUC eq $ckptStatus) {
        if (isCkptPropertyExists ($ckptName, "CVU_CHECK_INTERVAL")) {
           $cvuCheckInterval = getCkptPropertyValue ($ckptName, "CVU_CHECK_INTERVAL");
           if (!$cvuCheckInterval) {
              writeCkpt($ckptName, CKPTFAIL);
              die(dieformat(449));
           }
        }
        else {
           $cvuCheckInterval = get_CVU_checkInterval ($oldCrsHome);
           writeCkptProperty ($ckptName, "CVU_CHECK_INTERVAL", $cvuCheckInterval);
           writeCkptProperty ($ckptName, "IS_CVU_CONFIGURED", TRUE);
        }

        my $isCvuConfigured = TRUE;
        if (isCkptPropertyExists ($ckptName, "IS_CVU_CONFIGURED")) {
           $isCvuConfigured = getCkptPropertyValue ($ckptName, "IS_CVU_CONFIGURED");
           if (!$isCvuConfigured) {
              writeCkpt($ckptName, CKPTFAIL);
              die(dieformat(450));
           }
        }
        trace ("cvu check interval = $cvuCheckInterval cvu config is $isCvuConfigured");
        $CFG->oldconfig ('IS_CVU_CONFIGURED', $isCvuConfigured);
        $CFG->oldconfig ('CVU_CHECK_INTERVAL', $cvuCheckInterval);
     }
     else {
        $CFG->oldconfig ('IS_CVU_CONFIGURED', FALSE);
        $CFG->oldconfig ('CVU_CHECK_INTERVAL', "0");
     }
     trace ("cvu check interval = $cvuCheckInterval");
  }

  writeCkpt($ckptName, CKPTSUC);
  $CFG->wipCkptName("ROOTCRS_STACK");
  return;
}

####---------------------------------------------------------
#### Function for returning CLUSTER_GUID. 
# 1) First , checks for clusterware active version 
#    NOTE: CFG->oldconfig('ORA_CRS_VERSION') must be set.
# 2) If, version < 11.1.0.7 , returns -1.
# 3) Else, get the clusterguid from "crsctl get css clusterguid".
# ARGS: 1
# ARG1: ORA_CRS_HOME
# @returns ID fetched from  crsctl or -1 in case of error

sub get_clusterguid
{
   my $home = $_[0];
   my $id = -1;

   ## If here , must be 11.1.0.7 and higher
   my @OLD_CRS_VERSION = @{$CFG->oldconfig('ORA_CRS_VERSION')};
   my $old_crs_ver = join('.', @OLD_CRS_VERSION);
   if (-1 == versionComparison($old_crs_ver, "11.1.0.7.0"))
   {
      trace("Skipping clusterguid fetch for ".join('.',@OLD_CRS_VERSION));
      return -1;
   }
   trace("Fetching clusterguid from ".join('.',@OLD_CRS_VERSION));

   my $cmd;
   if (! defined $home) {
     $cmd = crs_exec_path('crsctl');
   } else {
     $cmd = catfile( $home, 'bin', 'crsctl' );
   }

   # run "crsctl get css clusterguid"
   my @out = system_cmd_capture(($cmd, "get", "css", "clusterguid"));
   my $rc  = shift @out;

   # if succeeded, get the guid, output must be a single line
   if ($rc == 0) {
      my $outid = $out[0];
      $id = trim($outid);
      trace( "Got CSS GUID: $id (".join(' ',@out).")" );
   }
   else
   {
      trace ("Retrieval of CSS GUID failed (rc=$rc), ".
             "with the message:\n".join("\n", @out)."\n");
   }

   if ($id == -1) {
      trace("get_clusterguid : Failed to get clusterguid");
   }
   return $id;
}

####---------------------------------------------------------
#### Function for returning OCRID 
# ARGS: 1
# ARG1: ORA_CRS_HOME
# @returns ID fetched from  ocrcheck or -1 in case of error
sub get_ocrid
{
   my $crshome = $_[0];
   my $id      = -1;
   my $ocrcheck;

   if ($CFG->platform_family eq "windows") {
      $ocrcheck = catfile($crshome, 'bin', 'ocrcheck.exe');
   }
   else {
      $ocrcheck = catfile($crshome, 'bin', 'ocrcheck');
   }

   trace("Executing ocrcheck to get ocrid");
   open(OCRCHECK, "$ocrcheck |" );
   my @output = <OCRCHECK>;
   close(OCRCHECK);

   my @txt = grep (/ ID /, @output);
   foreach my $line (@txt) {
      my ($idstring, $oid)  = split(/:/, $line);
      $id = trim($oid);
   }

   if ($id == -1) {
      trace("get_ocrid: Failed to get ocrid ");
   }
   else {
     $CFG->OCR_ID($id);
   }

   return $id;
}

=head2 get_upgrade_netinfo

  This is a top-level call to get a string with network information for 
  upgrade config.
  (This is an analog of NETOWORKS installer interview parameter.)
  Note: OLD_CRS_HOME must be set to use this function.

=head3 Parameters

  None.

=head3 Returns

  returns a string - a comma-separated list of net intf defs, installer-style

=cut

sub get_upgrade_netinfo
{
   my @iflist_out;    # oifcfg iflist results
   my @getif_out;     # oifcfg getif results
   my @olsnodes_out;  # olsnodes -p results
   my @iflist_info;   # parsed oifcfg iflist results
   my @getif_info;    # parsed oifcfg getif results
   my @ols_info;      # parsed olsnodes results
   my @olsif_info;    # parsed olsnodes results matched against oifcfg iflist
   my @net_info;      # consolidated parsed interfaces to use in prf net cfg
   my $rc;
   my $generrmsg = "Cannot get node network interfaces";

   my $OLD_CRS_HOME = $CFG->OLD_CRS_HOME;
   check_dir( $OLD_CRS_HOME ) or die(dieformat(252));

   my @OLD_CRS_VERSION = @{$CFG->oldconfig('ORA_CRS_VERSION')};

   if (! @OLD_CRS_VERSION) {
     @OLD_CRS_VERSION = get_crs_version($CFG->ORA_CRS_HOME);
   }

   # oifcfg iflist
   ($rc,@iflist_out) = get_oifcfg_iflist($CFG->ORA_CRS_HOME);
   die(dieformat(309, join(' ',@iflist_out))) if ($rc!=0);

   # check if old crs is up, if so, do oifcfg getif 
   my $crs_is_up = check_OldCrsStack();
   if ($crs_is_up) {
      # getif must be invoked from old crshome
      ($rc,@getif_out) = get_oifcfg_getif($OLD_CRS_HOME);
      die(dieformat(309, join(' ', @getif_out))) if ($rc!=0);
   } else {
     # get getif data from OCR if the stack is down
      ($rc,@getif_out) = get_ocr_getif_info ($CFG->ORA_CRS_HOME);
      die(dieformat(309, join(' ', @getif_out))) if ($rc!=0);
   }

   # olsnodes private addrs
   if ($crs_is_up &&
      (! ($OLD_CRS_VERSION[0] == 10 && $OLD_CRS_VERSION[1] == 1)))
   {
      ($rc,@olsnodes_out) = get_olsnodes_info($OLD_CRS_HOME, '-p');
      die(dieformat(309, join(' ', @olsnodes_out))) if ($rc!=0);
   
   } else {
      # CRS <10.2 does not support olsnodes -p

      if (! $crs_is_up ) {
         trace("old version clusterware not running -- will try OCR interconnects instead ");
      } else {
         trace( "\"olsnodes -p\" unavailable in ".
             join('.', @OLD_CRS_VERSION). 
             " -- will try OCR interconnects instead ");
      }

      ($rc,@olsnodes_out) = get_ocr_privatenames_info ($CFG->ORA_CRS_HOME);
      die(dieformat(309, join(' ', @olsnodes_out))) if ($rc!=0);
   } 

   @iflist_info = parse_netinfo( \@iflist_out ); 
   $rc = shift @iflist_info; 
   die(dieformat(309, "failed to parse oifcfg iflist output")) if ($rc!=0);

   @getif_info = parse_netinfo( \@getif_out ); 
   $rc = shift @getif_info; 
   die(dieformat(309, "failed to parse oifcfg getif output")) if ($rc!=0);

   @ols_info = parse_olsnodesp_netinfo( \@olsnodes_out ); 
   $rc = shift @ols_info; 
   die(dieformat(309, "failed to parse olsnodes -p output")) if ($rc!=0);

   # validate olsnodes info against getif/iflist data
   # first, match to node interfaces 
   @olsif_info = match_node_netintfs( \@ols_info, \@iflist_info );
   $rc = shift @olsif_info; 
   die(dieformat(309, "failed to get olsnodes networks")) if ($rc!=0);

   # then, consolidate oifcfg-getif and olsnodes-p results
   @net_info = match_getif_netintfs( \@getif_info, \@olsif_info );
   $rc = shift @net_info; 
   die(dieformat(309, "failed to consolidate network info")) if ($rc!=0);

   my $s_instiflist = '';
   foreach (0..$#net_info) {
      my $intfref = $net_info[$_];
      $s_instiflist = oifcfg_intf_to_instlststr( $intfref, $s_instiflist );
   }
   trace ("upgrade netlst: \"".$s_instiflist."\""); 

   # convert netinfo into cmdline pars 
   if ($CFG->DEBUG) {
     my @netprogram = instlststr_to_gpnptoolargs( $s_instiflist );
     trace ("upgrade netcmd: \"".join(' ',@netprogram)."\""); 
   }

   # get a comma-separated list of cluster nodes to push cluster-wide
   # gpnp setup to
   my $s_cluster_nodes = get_upgrade_node_list( \@ols_info );
   trace ("upgrade node list: \"".$s_cluster_nodes."\""); 

   return ($s_instiflist, $s_cluster_nodes);
}

=head2 get_oifcfg_iflist

  Gets "oifcfg iflist" networks interface info, e.g.
  ("eth0  10.0.0.0  PRIVATE 255.255.252.0", 
   "eth1  140.87.4.0  UNKNOWN 255.255.252.0") 
  Note that adapter name (e.g. eth1) can be quoted and contain spaces
  on some platforms, and ip net addr can be ipv6. 

=head3 Parameters

  string with oifcfg home location. If undef, then current home is used.

=head3 Returns

  =head4 returns a list of strings-net intf defs
  =head4 result code (0 for success) as a first member of array.

=cut

sub get_oifcfg_iflist
{
   return get_oifcfg_info(($_[0], 'iflist','-p','-n'));
}

=head2 get_oifcfg_getif

  Gets "oifcfg getif" networks interface info, e.g.
  ("eth0  10.0.0.0  global  public", 
   "eth1  140.87.4.0  global  cluster_interconnect") 
  Note that adapter name (e.g. eth1) can be quoted and contain spaces
  on some platforms, and ip net addr can be ipv6. 

=head3 Parameters

  string with oifcfg home location. If undef, then current home is used.

=head3 Returns

  =head4 returns a list of strings-net intf defs
  =head4 result code (0 for success) as a first member of array.

=cut

sub get_oifcfg_getif
{
   return get_oifcfg_info(($_[0], 'getif'));
}

=head2 get_oifcfg_info

  Gets oifcfg networks interface info for specified command line params.

=head3 Parameters

  =head4 string with oifcfg home location. If undef, then current home is used.
  =head4 Rest of arguments passed to oifcfg cmdline. 

=head3 Returns

  =head4 returns a list of strings-net intf defs; Warning messages, if any, 
         are filtered out.
  =head4 result code (0 for success) as a first member of array.

=cut

sub get_oifcfg_info
{
   my @intfs = ();

   my ($home, @args) = @_;
   my $cmd;
   if (! defined $home) {
     $cmd = crs_exec_path('oifcfg');
   } else {
     $cmd = catfile( $home, 'bin', 'oifcfg' );
   }
   # run oifcfg asking for intf, net, type and mask
   my @out = system_cmd_capture(($cmd, @args));
   my $rc  = shift @out;

   # read-in interface list 
   if (0 == $rc) {
      trace "---Got oifcfg out ($cmd ".join(' ',@args)."):";
      foreach (0..$#out) {
         my $intf = $out[$_];
         trace $intf ;
         # total failure should return rc, else filter out non-fatal
         # error messages, if any, e.g. "PRIF-nn: error....."
         if ($intf !~ /^PR[A-Z]+-[0-9]+: /) {
           push @intfs, $intf ;
         } else {
           trace( $intf );
           print_error(173, $intf);
         }
      }
   } else {
      push @intfs, "$cmd ".join(' ',@args)." failed."; 
   }
   return ($rc, @intfs);
}

=head2 get_ocr_privatenames_info

  Gets OCR information about configured private nodenames in form 
  indentical to "olsnodes -p" output (nodename private_names...).

  This is used to replace olsnodes-p where not available, e.g. on 10.1.
  OCR access performed with ORA_CRS_HOME ocrdump utility by dumping
  SYSTEM.css keys.
  If all fails, will try "olsnodes" (node names only) in old home as 
  a last resort.

=head3 Parameters

  string with olsnodes home location. If undef, then current home is used.

=head3 Returns

  =head4 returns a list of strings with node names.
  =head4 result code (0 for success) as a first member of array.

=cut

sub get_ocr_privatenames_info
{
   my $home  = $_[0];
   my @nodes = ();
   my $cmd;
   if (! defined $home) {
     $cmd = crs_exec_path('ocrdump');
   } else {
     $cmd = catfile( $home, 'bin', 'ocrdump' );
   }

   # run "ocrdump -stdout -keyname SYSTEM.css"
   my @args = ($cmd, '-stdout', '-keyname', 'SYSTEM.css');
   my @out = system_cmd_capture(@args);
   my $rc  = shift @out;

   if ($CFG->DEBUG) {
     trace "---SYSTEM.css OCR dump:\n".
           join(' ',@args)."\nout: \n".join("\n",@out)."\n";
   }
   # read-in dumped css keys 
   if (0 == $rc) {
      my @nodnames = grep(/^\[SYSTEM\.css\.node_names\.[^\]\.]+\]$/,   @out);
      my @pvtnames = grep(/^\[SYSTEM\.css\.privatenames\.[^\]\.]+\]$/, @out);

      if (!(@nodnames) || !(@pvtnames)) {
         trace "Warning: OCR has no css public or private node names. ";
      }
      if ($CFG->DEBUG) {
         trace "---OCR node_names: ".  join(' ',@nodnames).
              "\n---OCR pvt_names: ".  join(' ',@pvtnames);
      }
      # to keep it simple, we do not do any matching by nodenum between 
      # node_names and privatenames, since they are not used together anyways;
      # assuming same order, same number.
 
      foreach (0..$#nodnames) {
         my $curidx = $_;
         my $nodname = $nodnames[$_];
         if (defined $nodname) {
           $nodname =~ m/^.+\.([^\]\.]+)\]$/; # take last key - nodename
           $nodname = $1;
         }
         # normally both arrays will be paired
         my $pvtname = $pvtnames[$_];
         if (defined $pvtname) {
           $pvtname =~ m/^.+\.([^\]\.]+)\]$/; # take last key - nodename
           $pvtname = $1;
         }
         if (defined $nodname) {
            push @nodes, "$nodname $pvtname";
         }
         trace ("ocr node parsed: -$nodname-$pvtname-="); 
      }
      if (scalar(@nodes) == 0) {
         trace("Failed to get a list of CSS nodes from OCR. ".
               "Setup way not work properly."); 
         print_error(171);

         # return at least a list of nodes (no -p), so setup can propagate
         # properly - run in old home.
         return get_olsnodes_info($CFG->OLD_CRS_HOME);
      }
   } else { 
      push @nodes, "".join(' ', @args)." failed.";
   }
   return ($rc, @nodes);
}

=head2 parse_netinfo

  Parse oifcfg-style netinfo (iflist/getif) into array of refs to array
  for each interface, containing interface definition elements.
  See oifcfg_intf_parse.

=head3 Parameters

  An array reference of strings with oifcfg output.

=head3 Returns

  =head4 returns a resulting array of parsed interfaces, each represented as
         an array containing string components of interface definition 
         (interface name (unquoted), masked addr, 
         scope (global/node), type (public,cluter_interconnect), mask).
  =head4 result code (0 for success) as a first member of array.

=cut

sub parse_netinfo
{
   my $netoutref = $_[0]; #ref
   my @intfs = @{$netoutref};
   my @netinfo = ();
   my $rc = 0;

   foreach (0..$#intfs) {
      my $idef = $intfs[$_];
      my $ada; 
      my $net;
      my $nod;
      my $typ;
      my $msk;
      my @net = oifcfg_intf_parse( $idef );
      ($ada, $net, $nod, $typ, $msk ) = @net;
      if ((defined $typ) && (defined $net) && (defined $ada)) {
        push @netinfo, \@net;
      }
   }
   return ($rc, @netinfo);
}

=head2 parse_olsnodesp_netinfo

  Parse olsnodes-style netinfo into array of refs to array
  for each node info. Each array contains public and private node name.

=head3 Parameters

  An array reference of strings with olsnodes output.

=head3 Returns

  =head4 returns an array of parsed olsnodes node info arrays (public/private
         node name).
  =head4 result code (0 for success) as a first member of array.

=cut

sub parse_olsnodesp_netinfo
{
   my $netoutref = $_[0]; #ref
   my @intfs = @{$netoutref};
   my @netinfo = ();
   my $rc = 0;

   foreach (0..$#intfs) {
      my $idef = $intfs[$_];
      my $n;
      my $host;
      my $pvthost;

      if ($CFG->DEBUG) { trace ("intf: $idef"); }

      # olsnodes -p will give output in form "<hostname> <privatehonm>" lines.
      $idef =~ s/^\s+|\s+$//g;
      $idef =~ s/\s+/ /g;
      $n    = rindex( $idef, ' ' );
      $host = substr( $idef, 0, $n );

      $pvthost = substr( $idef, $n+1 );

      my @net = ($host, $pvthost);
      if (defined $host) {
        push @netinfo, \@net;
      }
      if ($CFG->DEBUG) { trace ("node parsed: -$host-$pvthost-="); }
   }
   return ($rc, @netinfo);
}

=head2 match_node_netintfs

  Create a table of oifcfg-style cluster-interconnect interfaces based on 
  oldnodes -p private node names, resolved and matched against available 
  node interfaces. 

=head3 Parameters

  =head4 an array reference of parsed olsnodes-p info (refs to parsed lines).
  =head4 an array reference of parsed oifcfg-iflist info (refs to parsed lines)

=head3 Returns

  =head4 returns an array of parsed oifcfg-iflist style info (array of refs) 
         for cluster-interconnect interfaces inferred from olsnodes-p output.
  =head4 result code (0 for success) as a first member of array.

=cut

sub match_node_netintfs
{
   my $olsref = $_[0];    #ref
   my $oiflstref = $_[1]; #ref
   my @olshs = @{$olsref};     # array of parsed olsnodes host/pvthost arrays
   my @intfs = @{$oiflstref};  # array of parsed oifcfg iflist arrays
   my @netinfo = ();
   my $rc = 0;

   trace "Processing ".scalar(@olshs)." olsnodes:";
   foreach (0..$#olshs) {
      my $olsintfref = $olshs[$_];
      my $node;
      my $pvtnode;
      my $iaddr;
      my $saddr;
      ($node, $pvtnode) = @{$olsintfref};
      trace "   node $_ pub:$node pvt:$pvtnode=";

      # [ resolve private node into addr 
      if (defined $pvtnode) {
         my $name;
         my $aliases;
         my $addrtype;
         my $length;
         my @addrs;
         ($name, $aliases, $addrtype, $length, @addrs) = 
              gethostbyname $pvtnode
              or die(dieformat(253, $pvtnode, $!));

         (($addrtype == AF_INET) && (scalar(@addrs) > 0)) or next;
         ($length == 4) or print_error(34); # 16

         trace "     $pvtnode addrs: ";
         for my $iiaddr (@addrs) {
            $saddr = undef;
            $iaddr = undef;
            if (defined $iiaddr) {
               $saddr = inet_ntoa( $iiaddr );
               trace "       $saddr ";

               # toberevised: +ipv6 - inet_pton
               $iaddr = ipv4_atol($saddr); 
            }
            if (defined $saddr) {
               # Now loop through the iflist interfaces and see if 
               # node private addr matched for known adapter
               foreach (0..$#intfs) {
                  my $intfref = $intfs[$_];
                  my $ada; 
                  my $net;
                  my $nod;
                  my $typ;
                  my $msk;
                  ($ada, $net, $nod, $typ, $msk ) = @{$intfref};
                  if ((defined $msk) && (defined $net) && (defined $ada)) {

                     # toberevised: +ipv6 - inet_pton
                     my $imask = ipv4_atol($msk); 
                     my $inet  = ipv4_atol($net); 

                     my $match = FALSE;
                     $match = TRUE if (($imask & $iaddr) == $inet);
                     my $unique = TRUE;
                     if ($match) {
                        foreach (0..$#netinfo) {
                           my  $intfref1 = $netinfo[$_];
                           my  $ada1 = @{$intfref1}[0];
                           if ($ada1 eq $ada) {
                              $unique = FALSE; 
                              last;
                           }
                        }
                        if ($unique) { 
                           # make a new cluster_interconnect intf
                           my @intf = ($ada, $net, 'global', 
                                       'cluster_interconnect', $msk );
                           push @netinfo, \@intf; 
                        }
                     }
                     trace("        matching olsnodes/iflist ".
                           "(net $net == $saddr & $msk) match=$match ".
                           "unique=$unique");
                  }
               }
            }
         }
      }
      # ] 
   }
   return ($rc, @netinfo);
}

=head2 match_getif_netintfs

  Create a consolidated table of all used oifcfg-style interfaces based on 
  oifcfg getif info, ammended with resolved oldnodes -p info, if any.
  Resulting netinfo will be used as a NETWORKS info in the gpnp profile.

=head3 Parameters

  =head4 an array reference of parsed oifcfg-getif info (refs to parsed lines).
  =head4 an array reference of parsed oifcfg-olsnodes info from 
         match_node_netintfs (refs to parsed lines)

=head3 Returns

  =head4 returns a merged array of parsed oifcfg-getif style info (array of 
         refs) for available network interfaces.
  =head4 result code (0 for success) as a first member of array.

=cut

sub match_getif_netintfs

{
   my $getifref = $_[0]; #ref
   my $olsifref = $_[1]; #ref
   my @getifs = @{$getifref};  # array of parsed oifcfg getifs arrays
   my @olsifs = @{$olsifref};  # array of parsed olsnodes matched to iflist 
   my @netinfo = ();
   my $rc = 0;

   my $ada; 
   my $net;
   my $nod;
   my $typ;
   my $msk;
   my $getif_pvt_ifs = 0;
   my $getif_pub_ifs = 0;

   # copy getif array into resulting netinfo as base, to be ammended with 
   # olsnodes info
   foreach (0..$#getifs) {
      my $gifref = $getifs[$_];
      ($ada, $net, $nod, $typ, $msk ) = @{$gifref};
      if (defined $ada) {
         my @netif = ($ada, $net, $nod, $typ, $msk );
         push @netinfo, \@netif;  # add unconditionally
 
         # count if types
         if ($typ =~ m/cluster_interconnect/i) {
           $getif_pvt_ifs++;
         }
         if ($typ =~ m/public/i) {
           $getif_pub_ifs++;
         }
      }
   }
   # If getif has no interconnects, try to derive them from 
   # olsnodes/ocr-private-name info:
   # loop through olsnodes interfaces and see if there is something not
   # yet present in results
   #
   # Note: ALL olsifs are cluster_interconnects
   #
   if ( $getif_pvt_ifs == 0 ) {
     foreach (0..$#olsifs) {
        my $olsifref = $olsifs[$_];
        ($ada, $net, $nod, $typ, $msk ) = @{$olsifref};

        my $match = FALSE;
        my $typealt = FALSE;
        if (defined $ada) {

          foreach (0..$#netinfo) {
            my $netifref = $netinfo[$_];
            my $ada_if; 
            my $net_if;
            my $nod_if;
            my $typ_if;
            my $msk_if;
            ($ada_if, $net_if, $nod_if, $typ_if, $msk_if ) = @{$netifref};
            #   0        1         2        3       4
            
            # if a different interface with the same subnet found in results
            # do not add it, just update the type. 
            # (Loose match on subnet, not adapter name if ($ada eq $ada_if))
            #
            if ($net eq $net_if) {
               $match = TRUE;

               # make sure olsnodes type is included in the results type
               # - If there are multiple public types defined, override 
               #   matching interfaces to be cluster_interconnects
               # - If there is only single public, make it dual-purpose
               #
               if ($typ_if !~ m/$typ/i) { 
                 $typealt = TRUE;
                 if ($getif_pub_ifs > 1) {
                   $getif_pub_ifs--;
                   ${$netifref}[3] = $typ; # replace
                 } else {
                   ${$netifref}[3] = $typ_if .= ",$typ"; # append
                 }
               }
               last;
            }
         }
         # if olsnodes interface is not int the results yet, add a copy 
         if (! $match) {
            my @olsif = @{$olsifref};
            push @netinfo, \@olsif;
         }
         trace(" matching olsnodes/getif $ada-$net match=$match ".
               "typealt=$typealt ");
       }
     } # for olsifs
   }
   trace "---resulting upgrade iflist:";
   foreach (0..$#netinfo) {
      my $netifref = $netinfo[$_];
      ($ada, $net, $nod, $typ, $msk ) = @{$netifref};
      trace ("intf $_: -$ada-$net-$nod-$typ-$msk-");  
   }
   trace "---";
   return ($rc, @netinfo);
}

=head2 get_upgrade_node_list

  Create a comma-seaparated list of cluster nodes based on parsed oldnodes 
  info. 

=head3 Parameters

  =head4 an array reference of parsed olsnodes-p info (refs to parsed lines).

=head3 Returns

  =head4 returns a comma-separated list of cluster nodes.

=cut

sub get_upgrade_node_list
{
   my $olsref = $_[0];    #ref
   my @olshs = @{$olsref};     # array of parsed olsnodes host/pvthost arrays
   my $s_nodes_list = "";

   foreach (0..$#olshs) {
      my $olsintfref = $olshs[$_];
      my ($node, $pvtnode) = @{$olsintfref};

      if (defined $node) {
         if (! ($s_nodes_list eq "")) {
            $s_nodes_list .= ",";
         }
         $s_nodes_list .= $node;
      }
   }
   if ($CFG->DEBUG) {
      trace "Cluster node list, per olsnodes: \"$s_nodes_list\"";
   }
   return $s_nodes_list;
}

=head2 get_ocr_getif_info

  Gets OCR information about configured interfaces in form 
  indentical to "oifcfg getif" output, e.g. line 
   "Local Area Connection 4  140.87.136.0  global  cluster_interconnect,public"
  see get_oifcfg_getif()

  This is used to replace "oifcfg getif" when stack is not running during
  rolling upgrade from older pcw versions.
  OCR access performed with ORA_CRS_HOME ocrdump utility by dumping
  SYSTEM.css.interfaces.global keys.

=head3 Parameters

  string with ocrdump home location. If undef, then current home is used.

=head3 Returns

  =head4 returns a list of strings with "oifcfg getif"-like output.
  =head4 result code (0 for success) as a first member of array.

=cut

sub get_ocr_getif_info
{
   my @intfs = ();

   my $home = $_[0];
   my $cmd;
   if (! defined $home) {
     $cmd = crs_exec_path('ocrdump');
   } else {
     $cmd = catfile( $home, 'bin', 'ocrdump' );
   }
   # run "ocrdump -stdout -keyname SYSTEM.css.interfaces.global"
   my @args = ($cmd, '-stdout', '-keyname', 'SYSTEM.css.interfaces.global');
   my @out = system_cmd_capture(@args);
   my $rc  = shift @out;

   if ($CFG->DEBUG) {
     trace "---SYSTEM.css.interfaces.global OCR dump:\n".
           join(' ',@args)."\nout: \n".join("\n",@out)."\n";
   }
   # read-in dumped css keys 
   if (0 == $rc) {
      foreach (0..$#out) {
         my $curidx = $_;
         my $key;
         my $itf;
         my $ada;
         my $net;
         my $typ;

         # e.g. 
         # [SYSTEM.css.interfaces.global.eth0.192|d168|d1|d0.1]
         # ORATEXT : cluster_interconnect,public
         $key = $out[$_];
         $ada = $net = $typ = $itf = undef;
         if ($key =~ m/^\[SYSTEM\.css\.interfaces\.global\.([^\]^\.]+)\.([^\]^\.]+)\.1\]$/) { # take last key - nodename
           $ada = $1;
           $net = $2;
           if (defined $ada && defined $net) {
             $net =~ s/\|d/\./g;  # for ipv4, make ip notation
             $key = $out[$_+1];
             if ( $key =~ m/^ORATEXT : (.+)$/) { # take type classification for current network
               $typ = $1;
               $itf = "$ada  $net  global  $typ";
             }
             if ($CFG->DEBUG) {
               trace ("ocr intf parsed: [$curidx] -$ada-$net-$typ-=");
             }
           }
         }
         if (defined $itf) {
           push @intfs, $itf;
         }

      }
        trace ("---ocr iflist:");
        trace_lines(@intfs);
        trace ("---");
      if (scalar(@intfs) == 0) {
         $rc = 2;
         trace("Failed to get a list of interfaces from OCR. ".
               "Setup may not work properly."); 
         print_error(172);
      }
   } else { 
      push @intfs, "".join(' ', @args)." failed.";
   }
   return ($rc, @intfs);
}

sub queryClusterConfig
{
   trace("Query cluster configuration");

   if (! isOldVersionLT112())
   {
      #Invoke the 11.2patchupg module. 
      #TODO - need to see if this will also be applicable for upgrade to 12g 
      performpost112upg();
      return;
   }

   my $new_crs_running = FALSE;
   my $old_crs_running = FALSE;

   #Check if Old Clusterware is running
   if ($CFG->isRerun) {
      trace("Trying to check if new CRS stack is partially up");
      $new_crs_running = check_NewCrsStack();
   }

   if (!$new_crs_running) {
      $old_crs_running = check_OldCrsStack();
   }
   else {
      $old_crs_running = FALSE;
   }

   trace("old crs running is $old_crs_running");
   trace("new crs running is $new_crs_running");
   $ENV{'ORACLE_HOME'} = $CFG->params('ORACLE_HOME');
  
   if ($old_crs_running) {
      # stop the stack if it is already running.
      stop_OldCrsStack();

      if ($CFG->platform_family eq "windows") {
         if (! s_deltOldServ()) {
            die(dieformat(302));
         }
      }
   }

   # update ons.config file
   update_ons_config();

   my $vfds = CSS_get_old_VF_string();
   $CFG->VF_DISCOVERY_STRING($vfds);
   if (! $CFG->VF_DISCOVERY_STRING) {
      die(dieformat(303));
   }

   trace ("Voting file discovery string:", $CFG->VF_DISCOVERY_STRING);
  
   # Prepare the voting files for upgrade; if there is an error here
   # it should not be considered fatal, since this is only an attempt
   # to correct a case where a voting file's skgfr block 0 was
   # overwritten, which is not likely to have happened
   # $CFG->CSS_prep_old_VFs(); - call to clsfmt has been defunct
}

sub isNodeappsExists
#-------------------------------------------------------------------------------
# Function: Check if nodeapps exists 
# Args    : none
# Returns : TRUE  if     exists
#           FALSE if not exists
#-------------------------------------------------------------------------------
{
   my @crs_nodevip_list_old = get_OldVipInfoFromOCRDump();
   if (scalar(@crs_nodevip_list_old) == 0) {
      return FALSE;
   }

   return TRUE;
}

# Stops the old running crs stack.
sub stop_OldCrsStack
{
  my $OLD_CRS_HOME = $CFG->OLD_CRS_HOME;
  my $status = s_stop_OldCrsStack($OLD_CRS_HOME);
  if (0 == $status) {
      trace ("Old CRS stack stopped successfully");
  } else {
      trace ("Unable to stop Old CRS stack");
      die;
  }
  sleep(60);
}

# Start the old CRS stack.
sub start_OldCrsStack
{
  my $old_crshome = $CFG->OLD_CRS_HOME;
  my $crsctl      = catfile ($old_crshome, 'bin', 'crsctl');
  my $rc;
  my @output;
  my $retries = 25;
  my $is_up = FALSE;

  @output = system_cmd_capture($crsctl, 'start', 'crs');
  $rc = shift (@output);

  if (0 != $rc) {
      die(dieformat(117));
  }

  while ($retries) {
     if (check_OldCrsStack()) {
        $is_up = TRUE;
        last;
     }
     trace ("Waiting for old Grid Infrastructure stack to start");
     sleep (5);
     $retries--;
  }

  if ($is_up) {
     trace ("old Grid Infrastructure stack started successfully");
  } else {
     die(dieformat(120));
  }
}

sub check_OldCrsStack
#-------------------------------------------------------------------------------
# Function: Check if the stack is up from old CrsHome
# Args    : none
#-------------------------------------------------------------------------------
{
   trace("check old crs stack");
   my $old_crshome = $CFG->OLD_CRS_HOME;
   my @old_version = @{$CFG->oldconfig('ORA_CRS_VERSION')};
   my $crsctl      = catfile ($old_crshome, 'bin', 'crsctl');
   my $crs_stat    = catfile ($old_crshome, 'bin', 'crs_stat');
   my $status      = FAILED;

   if (! isOldVersionLT112())
   {
     trace("Post-11.2 check routine");
     my @output = system_cmd_capture($crsctl, 'check', 'crs');
     my $rc = shift @output;
     if ((0 == $rc) &&
      (scalar(grep(/4537/, @output)) > 0) &&
      (scalar(grep(/4529/, @output)) > 0) &&
      (scalar(grep(/4533/, @output)) > 0))
     {
       $status = SUCCESS;
     }
   }
   elsif ($old_version[0] eq "10" &&
       $old_version[1] eq "1")
   {
      my @output = system_cmd_capture($crs_stat);
      my $rc     = shift @output;
      my @cmdout = grep(/CRS-0184/, @output);

      trace("rc=$rc");
      if ($rc == 0 && scalar(@cmdout) == 0) {
         $status = SUCCESS;
      }
   }
   else {
      my @output;
      @output         = system_cmd_capture($crsctl, 'check', 'cssd');
      my $status_cssd = shift @output;
      
      @output         = system_cmd_capture($crsctl, 'check', 'crsd');
      my $status_crsd = shift @output;

      my $status_evmd = FALSE;
      if (($CFG->platform_family eq "windows") &&
          ($old_version[0] eq "10" && $old_version[1] eq "2"))
      {
         trace("Ignore evmd status");
         $status_evmd =FALSE;
      }
      else {
         @output = system_cmd_capture($crsctl, 'check', 'evmd');
         $status_evmd = shift @output;
      }

      #All the three daemons need to be up.
      if ((! $status_crsd) &&
          (! $status_evmd) &&
          (! $status_cssd))
      {
         $status = SUCCESS;
      }
   }

   if ($status == SUCCESS) {
      trace ("Earlier version Oracle Clusterware is running");
   }
   else {
      trace ("Earlier version Oracle Clusterware is not running");
   }

   return $status;
}

#update ons.config - Bug 8424681
 
sub update_ons_config
{
   my @nodelist = getONSnodelist();
   my $node;
   my $host = $CFG->HOST;
   my $str = "nodes="; 
   my $ONSCONFFILE = catfile($CFG->params('ORACLE_HOME'), 'opmn' ,
                             'conf', 'ons.config');

   foreach $node (@nodelist)
   {
    my($onslocport, $onsremport) = get_ons_port($node);
    trace ("ons remoteport for $node is $onsremport");
    if ($node ne $nodelist[-1])
    {
       $str = $str . "$node:$onsremport" . ",";
    }
    else
    {
       $str = $str . "$node:$onsremport";
    }
   }

   trace("ons nodes string is $str");

   my($onslocport, $onsremport) = get_ons_port($host);
   trace("ons conf file is $ONSCONFFILE");
   open(FONS, ">>$ONSCONFFILE") or
        print_error(185, $ONSCONFFILE, $!);
   print FONS "localport=$onslocport\n";
   print FONS "remoteport=$onsremport\n";
   print FONS "$str\n";
   close FONS;
}

sub getONSnodelist
#---------------------------------------------------------------------
# Function: Get the list of nodes where ONS is configured. The nodes could
# be outside of the cluster nodes as well(Bug 8563905).
#---------------------------------------------------------------------
{

   my $cmd;
   my @nodelist;
   my $ORA_CRS_HOME = $CFG->ORA_CRS_HOME;
   my ($ONSCONFFILE, $line, $e, $e1, $e2, $e3, $e4);
   my ($home, $onsnodes, $node, $port, $Name, $list);
   my @tmparr;
   my @nodelistocr;
   my @nodelistons;
   my %union = ();

   #Get the onsnodelist from ons.config file
   # if version is 10.1, use dbhome where ons is configured. Otherwise, use OLD_CRS_HOME.
   my @old_version = @{$CFG->oldconfig('ORA_CRS_VERSION')};
   if ($old_version[0] eq "10" &&
      $old_version[1] eq "1") {
      $home = get101viphome();
   }
   else
   {
      $home = $CFG->OLD_CRS_HOME;
   }

   $ONSCONFFILE = catfile( $home, 'opmn' , 'conf', 'ons.config');
   trace("ons conf file is $ONSCONFFILE");
   open(FONS, $ONSCONFFILE) or
        trace("Could not  open \"$ONSCONFFILE\": $!");

   while(<FONS>) {
      if(/^nodes\b/i) { $onsnodes=$_;  }
   }
   close (FONS);

   ($Name, $list) = split(/=/, $onsnodes);
   @tmparr = split(/,/, $list);
   foreach my $elem (@tmparr)
   {
      ($node, $port) = split(/:/, $elem);
      push @nodelistons, $node;
   }

   trace("ons node list from ons.config is @nodelistons\n");

   #Get the ons nodelist from OCR
   $cmd = catfile( $ORA_CRS_HOME, 'bin', 'ocrdump' );
   my $ocrkey = "DATABASE.ONS_HOSTS";
   my @args = ($cmd, '-stdout', '-keyname', $ocrkey);
   my @out = system_cmd_capture(@args);
   my $rc  = shift @out;

   foreach $line (@out)
   {
      if($line =~ m/PORT\]/i)
      {
          ($e1, $e2, $e3, $e4) = split(/\./, $line);
          trace("ONS is configured on node $e3. Adding to onsnodelist");
          $e3 =~ s/!/./g;
          push @nodelistocr, $e3;
      }
   }

   trace("ons nodelist from OCR is @nodelistocr\n");

   foreach $e (@nodelistons) {$union{$e} = 1 }
   foreach $e (@nodelistocr) {$union{$e} = 1 }
   @nodelist = keys %union;

   trace("union of on nodelist from ons.config and OCR is @nodelist\n");

   return @nodelist;
}

# gets the VIp information from the OCR. CRS stack needs to be up
# before calling this sub routine.
sub get_OldVipInfo
{
  my @CRS_NODEVIP_LIST;
  my $vip_index     = 0;
  my $OLD_CRS_HOME  = $CFG->OLD_CRS_HOME;
  my @SNODES        = split (/,/, $CFG->params('NODE_NAME_LIST'));
  my ($srvctlbin, $vip_name, $ip, $old_netmask, $intif);
  my ($new_netmask, $vip);

  # if version is 10.1, use dbhome. Otherwise, use OLD_CRS_HOME.
  my @old_version = @{$CFG->oldconfig('ORA_CRS_VERSION')};

  if ($old_version[0] eq "10" &&
      $old_version[1] eq "1") {

     my $ons_home = get101viphome();
     $srvctlbin = catfile ($ons_home, 'bin', 'srvctl');
     $ENV{'ORACLE_HOME'}  = $ons_home;
  } else {
     $srvctlbin = catfile ($OLD_CRS_HOME, 'bin', 'srvctl');
  }

  foreach my $nodename (@SNODES) {
     if (! isOldVersionLT112()) {
        open(SRVCMDF, "$srvctlbin config nodeapps -n $nodename -S 1|")
             || die(dieformat(289));
        my @buffer = grep(/ip=/, <SRVCMDF>);
        close SRVCMDF;

        trace("output(srvctl config nodeapps -S): @buffer");

        if (scalar(@buffer) > 0) {
           trace ("buffer=$buffer[0]");
           my @list = split(/ +/, $buffer[0]);
           foreach (@list) {
              if ($_ =~ /name=/) {
                 $vip_name = parseText ($_);
              }
              elsif ($_ =~ /ip=/) {
                 $ip = parseText ($_);
              }
              elsif ($_ =~ /netmask=/) {
                 $old_netmask = parseText ($_);
              }
              elsif ($_ =~ /interfaces=/) {
                 $intif = parseText ($_);
              }
           }

       }  else {
    trace("Could not find IP details from running the command 'srvctl config nodeapps -n $nodename -S 1");
      }
     }
     else {
        open(SRVCMDF, "$srvctlbin config nodeapps -n $nodename -a|")
             || die(dieformat(289));
        my @buffer = <SRVCMDF>;
        close SRVCMDF;

        trace("output(srvctl config nodeapps): @buffer");

        my $VipValue = $buffer[0];
        chomp($VipValue);
        trace("VIpValue =  $VipValue");
        my ($Name, $Value) = split(/:/, $VipValue, 2);
        my $val1;
        ($val1, $vip_name, $ip, $old_netmask, $intif)
                           = split(/\//, $Value);
     }

     chomp $vip_name;
     chomp $ip;
     chomp $old_netmask;
     chomp $intif;
     $intif =~ s/\:/\|/g;

     trace("Older VIP IP address (vip_name) = $vip_name");
     trace("Older VIP IP name(ip) = $ip");
     trace("Older VIP IP netmask (old_netmask) = $old_netmask");
     trace("Older VIP IP interface (intif) =$intif");

     # use vip_name if it exists, otherwise use ip
     if (! $vip_name) {
        $vip = $ip;
     }
     else {
        $vip = $vip_name;
     }

     if (validateNetmask($old_netmask, $intif, \$new_netmask)) {
        if ($intif) {
           $CRS_NODEVIP_LIST[$vip_index] = "$vip/$new_netmask/$intif";
        } 
        else {
           $CRS_NODEVIP_LIST[$vip_index] = "$vip/$new_netmask";
        } 
     } 
     else {
        if ($intif) {
           $CRS_NODEVIP_LIST[$vip_index] = "$vip/$new_netmask/$intif";
        } 
        else {
           $CRS_NODEVIP_LIST[$vip_index] = "$vip/$old_netmask";
        }
     }

     trace ("vip on $nodename = $CRS_NODEVIP_LIST[$vip_index]");
     $vip_index++;
  }

   sub parseText {
      # extract netmask and interface
      my $line = $_[0];
      $line =~ s/{//g;
      $line =~ s/}//g;
      my ($dummy, $text) = split (/=/, $line);
      chomp $text;

      return $text;
   }

  $ENV{'ORACLE_HOME'} = $CFG->params('ORACLE_HOME');
  return @CRS_NODEVIP_LIST;
}

sub get_OldVipInfoFromOCRDump
#-------------------------------------------------------------------------------
# Function:  Get old VIP info from ocrdump
# Args    :  none
# Returns :  @vip_list
#-------------------------------------------------------------------------------
{
   trace("Getting old VIP details from ocrdump");
   my $ocrdump  = catfile ($CFG->params('ORACLE_HOME'), 'bin', 'ocrdump');
   my @nodes    = split (/,/, $CFG->params('NODE_NAME_LIST'));
   my $ix       = 0;
   my @vip_list;

   foreach my $nodename (@nodes) {
      $nodename = lc($nodename);
      # get IP from DATABASE.NODEAPPS.$nodename.VIP.IP
      if ($CFG->platform_family eq "windows") {
         open (OCRDUMP, "$ocrdump -stdout -keyname " .
               "DATABASE.NODEAPPS.$nodename.VIP.IP |");
      }
      else {
         open (OCRDUMP, "$ocrdump -stdout -keyname " .
               "'DATABASE.NODEAPPS.$nodename.VIP.IP' |");
      }

      my @output = <OCRDUMP>;
      close (OCRDUMP);

      if ($CFG->DEBUG) { trace("Dump from OCR: @output"); }

      my @txt = grep (/ORATEXT/, @output);
      my ($key, $ip) = split (/: /, $txt[0]);
      chomp($ip);
      trace("IP address (found from dump) = $ip");

      # get NETMASK from DATABASE.NODEAPPS.$nodename.VIP.NETMASK
      if ($CFG->platform_family eq "windows") {
         open (OCRDUMP, "$ocrdump -stdout -keyname " .
               "DATABASE.NODEAPPS.$nodename.VIP.NETMASK |");
      }
      else {
         open (OCRDUMP, "$ocrdump -stdout -keyname " .
               "'DATABASE.NODEAPPS.$nodename.VIP.NETMASK' |");
      }

      @output = <OCRDUMP>;
      close (OCRDUMP);

      if ($CFG->DEBUG) { trace("Dump from OCR: @output"); }

      @txt = grep (/ORATEXT/, @output);
      my $old_netmask;
      ($key, $old_netmask) = split (/: /, $txt[0]);
      chomp($old_netmask);

      trace("IP netmask (found from dump) = $old_netmask");

      # get network interface name
      if ($CFG->platform_family eq "windows") {
         open (OCRDUMP, "$ocrdump -stdout -keyname " .
               "CRS.CUR.ora!$nodename!vip.USR_ORA_IF |");
      }
      else {
         open (OCRDUMP, "$ocrdump -stdout -keyname " .
               "'CRS.CUR.ora!$nodename!vip.USR_ORA_IF' |");
      }

      @output = <OCRDUMP>;
      close (OCRDUMP);

      if ($CFG->DEBUG) { trace("Dump from OCR: @output"); }

      @txt = grep (/ORATEXT/, @output);
      my $intif;
      ($key, $intif) = split (/: /, $txt[0]);
      my $new_netmask;
      chomp($intif);
      trace("Network interface (found from dump)=$intif");

      if ($ip ne "" && $old_netmask ne "") {
         if (validateNetmask($old_netmask, $intif, \$new_netmask)) {
            if ($intif) {
               $vip_list[$ix] = "$ip/$new_netmask/$intif";
            }
            else {
               $vip_list[$ix] = "$ip/$new_netmask";
            }
         }
         else {
            if ($intif) {
               $vip_list[$ix] = "$ip/$new_netmask/$intif";
            }
            else {
               $vip_list[$ix] = "$ip/$old_netmask";
            }
         }

         $ix++;
      }
   }

   if ($CFG->DEBUG) { trace("vip_list = @vip_list"); }

   return @vip_list;
}

sub validateNetmask
#-------------------------------------------------------------------------------
# Function: Validate netmask 
# Args    : [0] - old netmask
#           [1] - network interface 
#           [1] - new netmask
# Returns : TRUE  if success
#           FALSE if failed
#           new netmask
#-------------------------------------------------------------------------------
{
   my $old_netmask      = $_[0];
   my $netif            = $_[1];
   my $new_netmask_ref  = $_[2];
   my $oifcfg           = catfile($CFG->OLD_CRS_HOME, 'bin', 'oifcfg');
   my $success          = TRUE;
   $$new_netmask_ref    = $old_netmask;

   # if version is 10.1, use new GRID home for oifcfg
   # 10.1 oifcfg does not support -n option.
   my @old_version = @{$CFG->oldconfig('ORA_CRS_VERSION')};

   if ($old_version[0] eq "10" &&
      $old_version[1] eq "1") 
   {
      $oifcfg = catfile($CFG->ORA_CRS_HOME, 'bin', 'oifcfg');
   }

   open OPUT, "$oifcfg iflist -p -n |";

   if ($netif) { #not null
      my @output = grep { /\b$netif\b/i } (<OPUT>);

      if ($CFG->DEBUG) { trace("OIFCFG_IFLIST output : @output"); }

      if (scalar(@output) == 0) { #not found
         trace("Unable to find netmask for network interface=$netif");
         $success = FALSE;
      }
      else {
         if ($CFG->DEBUG) { trace("Processing OIFCFG_IFLIST output: $output[0]"); }

         my @netmask_list  = split (/ +/, $output[0]);
         if (scalar(@netmask_list) > 2) {
             $$new_netmask_ref = $netmask_list[-1]; #last entry
             chomp $$new_netmask_ref;
             if ($old_netmask ne $$new_netmask_ref) {
                trace("old_netmask=$old_netmask does NOT match " .
                      "new_netmask=$$new_netmask_ref");
                $success = FALSE;
             }
         } else {
            $success = FALSE;
         }
      }
   } else {
      my @output = grep { /\b$old_netmask\b/i } (<OPUT>);

      if ($CFG->DEBUG) { trace("Processing OIFCFG IFLIST output: @output"); }

      if (scalar(@output) == 0) { #not found
         trace("Unable to find netmask=$old_netmask");
         $success = FALSE;
      }
   }

   close OPUT;
   return $success;
}

sub performpost112upg
{
   my $oldcrshome;
   my $oldolrlocation;
   my $crshome     = $CFG->ORA_CRS_HOME;
   my $olrlocation = $CFG->OLR_LOCATION;
   my $host        = $CFG->HOST;
   my $success;
   my @old_ver     = @{$CFG->oldconfig('ORA_CRS_VERSION')};
   my $oldcrsver   = join('.',@old_ver);
   my $old_crs_running;

   #Find old home
   $oldcrshome = s_get_olr_file ("crs_home");

   if ($oldcrshome eq $crshome) {
      trace("Upgrade actions are already performed");
      return;
   }

   #Check if Old Clusterware is running
   if (! $CFG->SIHA) {
      $old_crs_running = check_OldCrsStack();
   }
 
   $oldolrlocation = s_get_olr_file ("olrconfig_loc");

   #check validity of olr file before upgrade. Part fix for bug 9466585
   if (checkOLR($oldcrshome) != SUCCESS) {
      die(dieformat(33, $oldcrsver));
   }
  
   trace("The old crs home = $oldcrshome");
   trace("The old OLR location = $oldolrlocation");

   if ((! $CFG->SIHA) && (! $old_crs_running))
   {
     trace("Make sure the older stack is completely down");
     stopClusterware($oldcrshome, "crs") || die(dieformat(349));
   }

   if ($CFG->SIHA) {
      if (!perform_sihaASM_upg())
      {
        die(dieformat(304));
      }
   }

   my $asmConf = $CFG->oldconfig('ASM_CONFIGURED');
   trace("ASM_CONFIGURED: $asmConf");
   
   # Use ASMCA only on first node when old CRS version >= 11.2 to
   # set ASM in rolling migration mode if upgrade is rolling
   if (($asmConf) && (isRolling()) && (! $CFG->SIHA) &&
        (isFirstNodeToUpgrade()))
   {
     if (! $old_crs_running)
     {
       trace("Old stack is down. Attempting to start for rolling migration");
       start_OldCrsStack();
       $old_crs_running = SUCCESS;
     }
     
     if (! upgradeASM(FIRST_NODE, TRUE,
                       $CFG->oldconfig('ORA_CRS_HOME'),
                       $CFG->oldconfig('NODE_LIST'),
                       $CFG->oldconfig('LOCAL_NODE'))) 
     {
       die(dieformat(305));
     }
   }

   if ((! $CFG->SIHA) && $old_crs_running) {
     # stop the stack if it is already running.
     if (! stopClusterware($oldcrshome, "crs")) {
        die(dieformat(349));
     }
   }

   if (($asmConf) && (! isRolling()) && (! $CFG->SIHA) &&
        (isFirstNodeToUpgrade()))
   {
     if (! upgradeASM(FIRST_NODE, FALSE,
                       $CFG->oldconfig('ORA_CRS_HOME'),
                       $CFG->oldconfig('NODE_LIST'),
                       $CFG->oldconfig('LOCAL_NODE')))
     {
       die(dieformat(306));
     }
   }

   if ($CFG->SIHA) {
      # Disable ora.cssd
      my $crsctl = catfile($CFG->ORA_CRS_HOME, 'bin', 'crsctl');
      my @cmd = ($crsctl, 'modify', 'res', 'ora.cssd', '-attr',
                  "\"ENABLED=0\"", '-init');
      my $status = system_cmd(@cmd);
      if (0 != $status)
      {
        die("Failed to disable CSSD startup");
      }

      if ($CFG->platform_family ne "windows")
      {
        # Disable ora.diskmon
        @cmd = ($crsctl, 'modify', 'res', 'ora.diskmon', '-attr',
                 "\"ENABLED=0\"", '-init');
        $status = system_cmd(@cmd);
        if (0 != $status)
        {
          die("Failed to disable DISKMON startup");
        }
      }

      if (! stopOracleRestart()) {
         die(dieformat(348));
     }
   }

   if ($CFG->platform_family eq "windows") {
      s_unregister_service("ohasd");
      if (! $CFG->SIHA) {
         s_upgrade_services();
      }
      if ((! is_dev_env()) && (isRolling())) {
         if (! upgrade_asm_service()) {
            die(dieformat(307));
         }
      }
   }

   #take old olr.loc backup
   if ($CFG->platform_family eq "windows") {
      my $olr_loc     = $CFG->params('OLRLOC');
      my $olr_loc_bkp = $CFG->params('OLRLOC') . '.bkp';
      trace("backing up olr registry");
      s_copyRegKey($olr_loc, $olr_loc_bkp);
   }
   else {
      my $oldolr = $CFG->params('OLRCONFIG');
      my $oldolrbkp = $oldolr . ".bkp";
      trace("backing up old olr.loc file $oldolr");
      if (! copy_file($oldolr, $oldolrbkp))
      {
        trace("Failed to backup old olr.loc file $oldolr");
      }
   }

   if (! $CFG->SIHA)
   {
      if (isFirstNodeToUpgrade ())
      {
         #copy necessary files to new home
         copyfiles($oldcrshome, $crshome);

         #copy olr file to new home and set permissions
         copy_file($oldolrlocation, $olrlocation);
      }
      elsif (!isHomeShared())
      {
         #copy gpnp global files to new home
         copyfiles($oldcrshome, $crshome);

         #copy olr file to new home and set permissions
         copy_file($oldolrlocation, $olrlocation);
      }

      copy_gpnpnodefiles($oldcrshome, $crshome);
   } else {
          #copy necessary files to new home
          copyfiles($oldcrshome, $crshome);
          copy_gpnpnodefiles($oldcrshome, $crshome);

          #copy olr file to new home and set permissions
          copy_file($oldolrlocation, $olrlocation);
   }

   if ($CFG->SIHA) {
     s_set_ownergroup ($CFG->params('ORACLE_OWNER'), $CFG->params('ORA_DBA_GROUP'),
                       $olrlocation);
   } else {
     s_set_ownergroup ($CFG->SUPERUSER, $CFG->params('ORA_DBA_GROUP'), $olrlocation);
   }

   s_set_perms ("0600", $olrlocation);

  #For handling CHM Upgrade. Bug 11852891. 
  if (!($CFG->SIHA) && !(isVersionLT11202())) {
    if (!perform_CHM_upgrade()) {
       trace("Failed to perform Cluster Health Monitor upgrade");
       die(dieformat(404));
    }
  }

}

sub checkOLR
#-------------------------------------------------------------------------------
# Function: check if  OLR file exists and is configured.
# Args    : none
# Returns : SUCCESS or FAILED
#-------------------------------------------------------------------------------
{
   my $homeDir = $_[0];

   my $crshome = ($homeDir) ? $homeDir : ($CFG->params('ORACLE_HOME')); 
   my $cmd = catfile($crshome, 'bin', 'ocrcheck');
   my @out = system_cmd_capture($cmd, "-local", "-debug");
   my $rc  = shift @out;
   my $check_passed = FAILED;

   trace("checkOLR rc=$rc");
   my @cmdout1 = grep(/(^PROTL-708)/, @out);
   my @cmdout2 = grep(/(^PROTL-715)/, @out);
   my @cmdout3 = grep(/(^PROTL-721)/, @out);
   
   if ($rc == 0 && scalar(@cmdout1) == 0 &&
       scalar(@cmdout2) == 0 && scalar(@cmdout3) == 0)
   {
     $check_passed = SUCCESS;
     trace ("OLR check: passed");
   } 
   else {
     trace ("OLR check: failed");
   }

   return $check_passed;
}

#-------------------------------------------------------------------------------
# Function: checks if the user running the upgrade matches with the owner
# of the old crs home.
# Args    : none
# Returns : TRUE or FALSE
#-------------------------------------------------------------------------------
sub checkOldCrsOwner
{
  my $ch = $CFG->OLD_CRS_HOME;
  my $cssd_bin = "$ch/bin/ocssd.bin";
  my $old_ch_owner = getBinaryOwner($cssd_bin);

  my $ch_owner = $CFG->params('ORACLE_OWNER');

  trace("new CH owner = $ch_owner");
  trace("old CH owner = $old_ch_owner");
  if ($old_ch_owner != $ch_owner)
  {
    print_error(360, $old_ch_owner, $ch_owner);
    return FALSE;
  }

  return TRUE;
}

#------------------------------------------------------------------------------
# Function: Gets the OS owner of the given binary.
# Args    : the name of the binary
# Returns : owner name.
#-------------------------------------------------------------------------------
sub getBinaryOwner
{

  my $binary = $_[0];
  my $binary_o;

  if ( -f $binary )
  {
     my ($dev, $ino, $mode, $nlink, $uid, $gid, $rdev, $size, $atime, $mtime,
         $ctime, $blksize, $blocks) = stat( $binary );

     my ($name, $passwd, $uid, $ugid, $quota, $comment, $gcos, $dir, $shell)
       = getpwuid( $uid );
     $binary_o = $name;

     trace( "Owner of executable $binary:: $binary_o" );
  }
  else
  {
     print_error(359, $binary);
  }
  return $binary_o;
}

#------------------------------------------------------------------------------
# Function: Does checks before proceeding with the upgrade.
# Args    : none.
# Returns : TRUE or FALSE
#-------------------------------------------------------------------------------
sub preUpgradeChecks
{
  # check if the crs owner of the new home is same as the crs owner of the old home
  if (!checkOldCrsOwner())
  {
    return FALSE;
  }

  # none of the pre upgrade checks failed.
  trace ("all pre-upgrade checks passed");
  return TRUE;
}

sub isHomeShared
{
   #use a path that's always writable in the grid home to do sharedness check.
   my $pathtochk  = catdir($CFG->params('ORACLE_HOME'), 'crs', 'install');
   my @capout = ();
   my $rc;
   my $user       = $CFG->params('ORACLE_OWNER');
   my $nodelist   = $CFG->params('NODE_NAME_LIST');
   my $CKPTTOOL   = catfile ($CFG->params('ORACLE_HOME'), 'bin', 'cluutil');
   my @program    = ($CKPTTOOL, '-chkshare', -oh , $pathtochk,
                     '-localnode', $CFG->HOST, '-nodelist', $nodelist);
   my $status;

   trace("checking if path $pathtochk is shared");
   # run as specific user, if requested
   $rc = run_as_user2($user, \@capout, @program);
   # cluutil return 0 err code and errors, if any, on stdout
   if (scalar(grep(/TRUE/, @capout)) > 0)
   {
     trace("The oracle home " . $CFG->params('ORACLE_HOME') . " is shared ");
     $status = TRUE;
   }
   else
   {
     trace("The oracle home " . $CFG->params('ORACLE_HOME') . " is not shared ");
     $status = FALSE;
   }
   
   return $status;
}

sub copyfiles
{
   my $sourcedir = $_[0];
   my $destdir   = $_[1];
   my $host = tolower_host();

   #copy gpnp, network/admin/, cdata and opmn/conf files
   if (! $CFG->SIHA) {
      copy_gpnpglobalfiles($sourcedir, $destdir); 
      copydir(catdir($sourcedir,'network','admin'), catdir($destdir,'network','admin'));
      copydir(catdir($sourcedir,'cdata'), catdir($destdir,'cdata'));

      # for opmn/conf, we only need to copy ons.config file
      my $sourcefile = catfile ($sourcedir, 'opmn', 'conf', 'ons.config');
      my $destfile   = catfile ($destdir, 'opmn', 'conf', 'ons.config');
      copy_file ($sourcefile, $destfile,
                 $CFG->params('ORACLE_OWNER'), $CFG->params('ORA_DBA_GROUP'));
   } 
   else {
      copydir (catdir($sourcedir,'network','admin'), catdir($destdir,'network','admin'));
      
      #Remove line starting with ADR_BASE from listener.ora 
      my $listenerfile = catfile($destdir,'network','admin','listener.ora');
      my $listenerfilebkup = catfile($destdir,'network','admin','listener.ora.bak');

      open (FP, "<","$listenerfile") || die(dieformat(206, $listenerfile, $!));

      my @file_lines = <FP>;
      close( FP );

      #backup the listener.ora file.
      if (!copy_file($listenerfile, $listenerfilebkup))
      {
         trace("Info: Failed to copy from $listenerfile to $listenerfilebkup");
      }

      #Write into listener.ora only those lines that don't begin with ADR_BASE

      open (FP, ">","$listenerfile") || die(dieformat(207, $listenerfile, $!));

      foreach my $line ( @file_lines ) {
         print FP $line unless ( $line =~ /^ADR_BASE/ );
      }
      close ( FP );
   }
}

sub perform_sihaASM_upg {
  my $success = TRUE;
  my $status;
  my @runasmca;

  trace("Start ASM  upgrade for Oracle Restart");
  @runasmca = (catfile ($CFG->ORA_CRS_HOME, "bin", "asmca"),
                  '-silent', '-upgradeLocalASM');

  trace ("Executing as " . $CFG->params('ORACLE_OWNER') . ": @runasmca");
  $status = run_as_user($CFG->params('ORACLE_OWNER'), @runasmca);

  if ($status != 0) {
    $success = FALSE;
    print_error(164);
  }

  return $success;
}

sub isFirstNodeToUpgrade
{
   trace ("isFirstNodeToUpgrade...");

   if ($CFG->JOIN) 
   { 
     trace("Join an upgraded cluster");
     return FALSE; 
   }

   # This function is frequently invoked on each node during upgrade, and
   # it could take a long time especially when stack is down.
   # So we should not do the same query every time, instead we should
   # cache the value of $firstnode_to_upgrade the first time this function
   # is executed.
   if (defined $CFG->isFirstNodeToUpgrade)
   {
     my $cachedValue = $CFG->isFirstNodeToUpgrade;
     trace ("isFirstNodeToUpgrade: $cachedValue");
     return $CFG->isFirstNodeToUpgrade;
   }

   my $firstnode_to_upgrade = TRUE;
   my $crsctl = crs_exec_path("crsctl");

   # get current releaseversion
   open (QUERYCRS, "$crsctl query crs releaseversion |");
   my $output = <QUERYCRS>;
   close (QUERYCRS);

   my $release_version = getVerInfo($output);

   # get current softwareversion
   my @nodes = split (/,/, $CFG->params('HOST_NAME_LIST'));
   my $software_version;
   my $host = tolower_host(); 

   foreach my $nodename (@nodes)
   {
     if (lc($nodename) =~ /\b$host\b/i) { next; }
     $software_version = getcursoftversion($nodename);
     trace ("software_version on $nodename=$software_version");

     # compare version
     if ($software_version eq $release_version)
     {
       $firstnode_to_upgrade = FALSE;
       last;
     }
   }

   trace("First node to upgrade is $firstnode_to_upgrade");
   $CFG->isFirstNodeToUpgrade($firstnode_to_upgrade);
   return $firstnode_to_upgrade;
}

sub perform_CHM_upgrade {
  my $crfhome = $_[0];
  my $patching = 1;
  my $oldcrfhome;
  my $oldorafile;
  my $oldcfgfile;
  my $bdblocbkp;
  my $oldorafilebkp;
  my $oldcfgfilebkp;
  my $neworafile;
  my $newcfgfile;
  my $oldbdbloc;
  my $oldbdbsize;
  my $HOST = tolower_host();

  if ($crfhome eq "")
  {
    $crfhome = $CFG->ORA_CRS_HOME;
    $patching = 0; #It says it is rolling upgrade
  }

  #Find old home
  $oldcrfhome = s_get_olr_file ("crs_home");

  if ($oldcrfhome eq $crfhome) {
     trace("11.2 Cluster Health Monitor upgrade actions are already performed");
     return TRUE;
  }
  trace("The old crf home = $oldcrfhome");

  #Get the bdb location from the old environment
  $oldbdbloc = getCHMAttrib("BDBLOC");
  if ($oldbdbloc eq "default")
  {
    $oldbdbloc = catfile ($oldcrfhome, "crf", "db", "${HOST}");
  }
  $oldbdbsize = getCHMAttrib("BDBSIZE");

  #  Get the old ora file and cfg file full path
  $oldorafile = catfile ($oldcrfhome, "crf", "admin", "crf${HOST}.ora");
  if (!-e $oldorafile)
  {
    trace("Info: No old Cluster Health Monitor configuration file present at ", $oldorafile); 
    return TRUE;
  }

  $oldcfgfile = catfile ($oldcrfhome, "crf", "admin", "crf${HOST}.cfg");
  if (!-e $oldcfgfile)
  {
    trace("Info: No old Cluster Health Monitor configuration (cfg) file present at ", $oldcfgfile); 
    return TRUE;
  }

  # Check if the new admin directory path exists.
  my $newadmindir = catdir ($crfhome, "crf", "admin");
  if (!-d $newadmindir)
  {
    trace("Info: New admin directory doesn't exist at", $newadmindir); 
    die(dieformat(406, $newadmindir));
  }

  #  Get the new ora file and cfg file full path
  $neworafile = catfile ($crfhome, "crf", "admin", "crf${HOST}.ora");
  $oldorafilebkp = "$neworafile"."bkp";
  $newcfgfile = catfile ($crfhome, "crf", "admin", "crf${HOST}.cfg");
  $oldcfgfilebkp = "$newcfgfile"."bkp";

  # Check if it is a rolling upgrade or patching
  if (((isRolling()) && (!$CFG->SIHA)) || $patching == 1)
  {
    # Create backup of the old configuration file to hande in case of failure.
    if (!copy_file($oldcfgfile, $oldcfgfilebkp))
    {
      trace("Info: Failed to copy from $oldcfgfile to $oldcfgfilebkp");
    }

    #copy the old configuration file.
    if (!copy_file($oldcfgfile, $newcfgfile))
    {
      trace("Info: Failed to copy from $oldcfgfile to $newcfgfile");
    }

    # Create a bakup of ora file to revert back in case of failure.
    if (!copy_file($oldorafile, $oldorafilebkp))
    {
      trace("Info: Failed to copy from $oldorafile to $oldorafilebkp");
    }

    # Just retain BDB Location and BDB Size from the old ora file.
    open (FP, ">$neworafile") || die(dieformat(408, $neworafile));

    if (index($oldbdbloc, $oldcrfhome) == -1)
    {
      print FP "BDBLOC=$oldbdbloc\n";
    }
    elsif ($patching == 1)
    {
      #Create a backup of oldbdb directory to handle in case of failure.
      $bdblocbkp = catdir("$crfhome", "crf", "db", "$HOST");
      movedir($oldbdbloc, $bdblocbkp);
    }
      
    # Don't move the bdb files in case of patching.
    if ($patching == 0)
    {
      my $hostbk = "$HOST"."bkp";
      #Create a backup of oldbdb directory to handle in case of failure.
      $bdblocbkp = catdir("$crfhome", "crf", "db", "$hostbk");
      movedir($oldbdbloc, $bdblocbkp);
    }
    print FP "BDBSIZE=$oldbdbsize\n";
    close (FP);
  }
  
  return TRUE;
}

sub upgrade_asm_service
{
  my $success = TRUE;

  # Do not change the order of these parameters as asmca requires the
  # parameters to be in a specific order or it will fail

  my @asmca = (catfile ($CFG->ORA_CRS_HOME, "bin", "asmca"),
                  '-silent', '-upgradeLocalASM', '-winSvc');

  trace ("Executing as " . $CFG->params('ORACLE_OWNER') . ": @asmca");
  my $status = run_as_user($CFG->params('ORACLE_OWNER'), @asmca);

  if ($status != 0) {
    $success = FALSE;
    print_error(162);
  }

  return $success;
}

sub upgrade_config {
  my $crsctlbin = crs_exec_path('crsctl');
  my $success = FAILED;
  my $status;
  my $cmd;
  my @relVer = split(/\./, getcrsrelver());
  my $newMajorVersion = $relVer[0]; 
  my @oldVer = @{$CFG->oldconfig('ORA_CRS_VERSION')};

  if ($newMajorVersion == $oldVer[0])
  {
    trace("Not a major version upgrade, skip startupgrade");
    return SUCCESS;
  }

  # Tell crs subsystem to copy the old resource profiles to new
  # engine. Which does not copy the nodeapps.
  $cmd = "$crsctlbin startupgrade";
  trace ("Invoking \"$cmd\"");
  $status = system_cmd ("$cmd");

  if (0 == $status)
  {
    trace("Successfully completed 'crsctl startupgrade'");
    $success = SUCCESS;
  }
  else
  {
    if ($CFG->isRerun)
    {
      trace("Sleep unitl CRSD comes up");
      for (my $i = 0; $i < 36; $i++)
      {
        sleep(10);
        if (GI_STACK_UP ==  checkGIStack())
        {
          $status = system_cmd("$cmd");
          if (0 == $status)
          {
            trace("Succeeded in executing 'crsctl startupgrade' during rerun");
            $success = SUCCESS;
            last;
          }
        }
      }

      if (GI_STACK_UP != checkGIStack()) { die(dieformat(251)); }
    }
    else
    {
      trace("Failed to execute 'crsctl startupgrade'");
      $success = FAILED;
    }
  }

  return $success;
}

# Get CRS active version major number (i.e. 10, 11, etc.) Stack must be up.
sub getCRSMajorVersion {
  my $crsctlbin = crs_exec_path('crsctl');
  my $ver       = 0;
  my @cmd       = ($crsctlbin, 'query', 'crs', 'activeversion');
  my @out       = system_cmd_capture(@cmd);
  my $rc        = shift @out;

  if ($rc == 0) {
     my $verinfo    = getVerInfo($out[0]);
     my @versionarr = split(/\./, $verinfo);
     $ver           = $versionarr[0];
     trace("crs major version=$ver");
  }
  else {
     my $mycmd = join(' ', @cmd);
     print_error(180, $mycmd, $rc);
     trace ("@cmd ... failed rc=$rc with message:\n @out \n");
  }

  return $ver;
}

sub upgrade_node
{
   # start checkpoint
   my $ckptStatus;
   my $ckptName = "ROOTCRS_NODECONFIG";
   if (isCkptexist($ckptName)) {
      $ckptStatus = getCkptStatus($ckptName);
      trace("'$ckptName' state is $ckptStatus");

      if (($ckptStatus eq CKPTSUC) && (! $CFG->FORCE))
      {
         trace("CRS node configuration resources are already configured");
         $CFG->wipCkptName("ROOTCRS_STACK");
         return SUCCESS;
      }
   }

   writeCkpt($ckptName, CKPTSTART);
   $CFG->wipCkptName($ckptName);

   # upgrade node
   my $ckptName             = "ROOTCRS_NODECONFIG";
   my $success              = TRUE;
   trace("Performing upgraded node configurations");

   if (isOldVersionLT112()) {
      if (isLastNodeToUpgrade())
      {
         $success = configLastNode();
      }

      # clean-up from upgrade
      if ($CFG->platform_family eq 'unix') {
         s_houseCleaning();
      }

      if ($success) {
         writeCkpt($ckptName, CKPTSUC);
      }
      else {
         writeCkpt($ckptName, CKPTFAIL);
         die(dieformat(278));
      }

      return $success;
   }
   else {
      if (isFirstNodeToUpgrade())
      {
         my $cvuCheckInterval = $CFG->oldconfig('CVU_CHECK_INTERVAL');
         my $isCVUConfigured = $CFG->oldconfig ('IS_CVU_CONFIGURED');
         trace ("cvu check interval is $cvuCheckInterval and config is $isCVUConfigured");
         if ($isCVUConfigured) 
         {
            trace ("cvu is configured");
            if (!isCkptPropertyExists ($ckptName, "IS_OLD_CVU_REMOVED")) {
               trace ("removing old cvu");
               $success = remove_CVU();
               if (!$success) {
                 writeCkpt($ckptName, CKPTFAIL);
                 die(dieformat(280));
               }
               writeCkptProperty ($ckptName, "IS_OLD_CVU_REMOVED", TRUE);
            }
         }

         if ($success)
         {
           $success = upgrademodel(FIRST_NODE);
           if (!$success) {
             writeCkpt($ckptName, CKPTFAIL);
             die(dieformat(280));
           }
 
           if (!isCkptPropertyExists ($ckptName, "IS_NEW_CVU_ADDED")) {
              if (add_CVU($cvuCheckInterval)) {
                 trace ("CVU resource successfully added");
                 writeCkptProperty ($ckptName, "IS_NEW_CVU_ADDED", TRUE);
              }
              else {
                 trace("Failed to add CVU resource");
                 $success = FALSE;
              }
           }
           if (!isCkptPropertyExists ($ckptName, "IS_NEW_CVU_DISABLED")) {
              if (disable_CVU()) {
                 trace("CVU resource successfully disabled");
                 writeCkptProperty ($ckptName, "IS_NEW_CVU_DISABLED", TRUE);
              }
              else {
                 trace("Failed to disable CVU resource");
                 $success = FALSE;
              }
           }
         }

         # Attempt to stop and disable the J2EE Container
         if ($success &&
             (stop_J2EEContainer() == SUCCESS) &&
             (disable_J2EEContainer() == SUCCESS)) {
             $success = TRUE;
         } else {
             $success = FALSE;
         }

      }
      
      if (isLastNodeToUpgrade())
      {
        if (addWallet_J2EEContainer() == FAILED) {
            writeCkpt($ckptName, CKPTFAIL);
            die(dieformat(1006));
         }
         $success = upgrade_config();
         if (!$success) {
            writeCkpt($ckptName, CKPTFAIL);
            die(dieformat(454));
         }
         
         # Setting active version can handle ASM stop rolling migration
         $success = setActiveversion();
         if (!$success) {
            writeCkpt($ckptName, CKPTFAIL);
            die(dieformat(281));
         }

         if ($success) { backupOCR(); }

         if (!wait_for_stack_start(360)) {
           writeCkpt($ckptName, CKPTFAIL);
           die(dieformat(251));
         }
         
         $success = upgrademodel(LAST_NODE);
         if (!$success) {
            writeCkpt($ckptName, CKPTFAIL);
            die(dieformat(284));
         }

         if (($CFG->params('MGMT_DB') =~ m/true/i) && !add_mgmt_db_listener()) {
            writeCkpt($ckptName, CKPTFAIL);
            die(dieformat(437));
         }

         $success = upgrade_acfs_registry();
         if (!$success) {
            writeCkpt($ckptName, CKPTFAIL);
            die(dieformat(422));
         }

         # start OC4J resource
         if (start_J2EEContainer() == FAILED) {
            writeCkpt($ckptName, CKPTFAIL);
            die(dieformat(1007));
         }
         
         if (!isCkptPropertyExists ($ckptName, "IS_CVU_ENABLED")) {
            if (enable_CVU()) {
               trace ("CVU resource enabled");
               writeCkptProperty ($ckptName, "IS_CVU_ENABLED", TRUE);
            }
            else {
               trace ("failed to enable CVU resource");
               $success = FALSE;
            }
         }

         if (!isCkptPropertyExists ($ckptName, "IS_CVU_RUNNING")) {
            if (start_CVU()) {
               trace ("CVU resource successfully upgraded");
               writeCkptProperty ($ckptName, "IS_CVU_RUNNING", TRUE);
            }
            else {
               trace("Failed to upgrade CVU resource");
               $success = FALSE;
            }
         }

         if ($success)
         {
           upgradeNodeListener();
         }
      }

      if ($success) {
         writeCkpt($ckptName, CKPTSUC);
      }
      else {
         writeCkpt($ckptName, CKPTFAIL);
         die(dieformat(285));
      }

      return $success;
   }
}

sub upgradeNodeListener
{
  if ($CFG->platform_family eq "windows")
  {
    trace("No role separation for listeners on Windows");
    return;
  }

  upgradeRemovedListener();
  
  my $ckptName = "ROOTCRS_NODECONFIG";
  my %listeners = getCurrentNetLsnrs($CFG->OLD_CRS_HOME);

  foreach my $lsnr (keys %listeners)
  {
    if (isCkptPropertyExists($ckptName, $lsnr))
    {
      trace("Listener '$lsnr' has been upgraded, hence bypassing ...");
      next;
    }

    stopNetLsnr(isOldVersionLT112() ? ($CFG->OLD_CRS_HOME) : $listeners{$lsnr}, $lsnr);
    my $lsnrLsnrhome = $lsnr.":".$listeners{$lsnr};
    removeNetLsnr($CFG->OLD_CRS_HOME, $lsnr);
    writeCkptProperty($ckptName, "LISTENER_REMOVED_NOT_UPGRADED", $lsnrLsnrhome);

    upgradeNetLsnrWithUsername(getLsnrUsername(), $listeners{$lsnr}, $lsnr);
    startNetLsnr(FALSE, getLsnrUsername(), $lsnr);
    writeCkptProperty($ckptName, "LISTENER_UPGRADE_COMPLETED", $lsnr);
    writeCkptProperty($ckptName, $lsnr, "true");
  }
}

sub upgradeRemovedListener
{
  my $ckptName = "ROOTCRS_NODECONFIG";

  if (isCkptPropertyExists($ckptName, "LISTENER_REMOVED_NOT_UPGRADED"))
  {
    my $lsnrLsnrhome = trim(getCkptPropertyValue($ckptName,
                                          "LISTENER_REMOVED_NOT_UPGRADED"));
    my $lsnr = (split(/:/, $lsnrLsnrhome))[0];
    my $lsnrHome = (split(/:/, $lsnrLsnrhome))[1];

    if ((! isCkptPropertyExists($ckptName, "LISTENER_UPGRADE_COMPLETED")) ||
        ($lsnr ne trim(getCkptPropertyValue($ckptName,
                                             "LISTENER_UPGRADE_COMPLETED"))))
    {
      trace("Proceed with upgrading the node listener '$lsnr'. Listener home: $lsnrHome");
      upgradeNetLsnrWithUsername(getLsnrUsername(), $lsnrHome, $lsnr);
      startNetLsnr(FALSE, getLsnrUsername(), $lsnr);
      writeCkptProperty($ckptName, "LISTENER_UPGRADE_COMPLETED", $lsnr);
      writeCkptProperty($ckptName, $lsnr, "true");
    }
  }
}

sub setActiveversion
{
   my $crsctl = crs_exec_path('crsctl');

   my @cmd;
   if ($CFG->FORCE)
   {
     @cmd = ($crsctl, 'set', 'crs', 'activeversion', '-force');
   }
   else
   {
     @cmd = ($crsctl, 'set', 'crs', 'activeversion');
   }

   my $status = system (@cmd);

   if (0 == $status) {
      trace ("@cmd ... passed");
      sleep(60);  # Wait until CRS changes to new engine
   } else {
      my $mycmd = join(' ', @cmd);
      print_error(180, $mycmd, $status);
      return FAILED;
   }

   return SUCCESS;
}

sub isLastNodeToUpgrade
{
   trace ("isLastNodeToUpgrade...");

   if ($CFG->FORCE)
   {
     trace("Force upgrade invoked");
     return TRUE;
   }

   if ($CFG->JOIN) 
   { 
     trace("Join an upgraded cluster");
     return FALSE; 
   }

   if (defined $CFG->isLastNodeToUpgrade)
   {
     my $cachedValue = $CFG->isLastNodeToUpgrade;
     trace("isLastNodeToUpgrade: $cachedValue");
     return $CFG->isLastNodeToUpgrade;
   }

   my $lastnode_to_upgrade = TRUE;
   my $crsctl = crs_exec_path("crsctl");

   # get current releaseversion
   open (QUERYCRS, "$crsctl query crs releaseversion |");
   my $output = <QUERYCRS>;
   close (QUERYCRS);

   my $release_version = getVerInfo($output);

   trace ("release_version=$release_version");

   # get current softwareversion
   my @nodes = split (/,/, $CFG->params('NODE_NAME_LIST'));
   my $software_version;
   my $host = tolower_host();

   foreach my $nodename (@nodes) {
      if (lc($nodename) =~ /\b$host\b/i) { next; }
      $software_version = getcursoftversion($nodename);
      trace ("software_version on $nodename=$software_version");

      # compare version
      if ($software_version ne $release_version) {
         $lastnode_to_upgrade = FALSE;
         last;
      }
   }

   trace("last node to upgrade is $lastnode_to_upgrade");
   $CFG->isLastNodeToUpgrade($lastnode_to_upgrade);
   return $lastnode_to_upgrade;
}

sub vipInfo
{
  my @crs_nodevip_list_old;
  my @crs_version = @{$CFG->oldconfig('ORA_CRS_VERSION')};

  if (($crs_version[0] eq '10' && $crs_version[1] eq '1') &&
      (! isNodeappsExists()))
  {
    my $crs_nodevips = $CFG->params('CRS_NODEVIPS');
    if ($crs_nodevips eq "")
    {
      print_error(397, $CFG->ORA_CRS_HOME);
      exit 1;
    } else {
      $crs_nodevips    =~ s/'//g; # ' in comment to avoid confusion of editors.
      $crs_nodevips    =~ s/"//g; # remove " on Windows
      $crs_nodevips    =~ s/\/\*/\//g; # handle different interface case

      @crs_nodevip_list_old = split (/\s*,\s*/, $crs_nodevips);
    }
  }
  else {
      if ($CFG->platform_family eq "windows") {
        @crs_nodevip_list_old = get_OldVipInfoFromOCRDump();
      }
      else {
        # Retrieve VIP config info from new stack,
        # though we still call binaries from old stack

        my $env = $ENV{'SRVM_TRACE'};
        undef $SRVM_TRACE;
        @crs_nodevip_list_old = get_OldVipInfo();
        $ENV{'SRVM_TRACE'} = $env;
      }
  }

  trace("crs_nodevip_list_old = [@crs_nodevip_list_old]");
  return @crs_nodevip_list_old;
}

sub configLastNode
#---------------------------------------------------------------------
# Function: Configure last node (for upgrade only)
# Args    : [0] crs_nodevip_list - contains viplist
# Returns : TRUE  if success
#           FALSE if failed
#---------------------------------------------------------------------
{
   trace ("Configuring last node");

   my $success        = TRUE;
   my $DHCP_flag      = FALSE;
   my $upgrade_option = "-u";
   my $status;
   my @nodes_to_start;
   my @nodes_to_add;
   
   my @crs_nodevip_list = vipInfo();
   trace("Old nodevip list is  @crs_nodevip_list");

   # set DHCP_flag to TRUE if it's DHCP
   if ($crs_nodevip_list[0] =~ /\bAUTO/) {
      $DHCP_flag = TRUE;
   }

   # for upgrade VIP information would be read from existing OCR.
   $success = upgrade_config();
   if (! $success) { print_error(454); }

   # Start nodeapps
   # The force upgrade node list contains only the nodes that
   # have been upgraded. Otherwise, that's a normal upgrade and
   # all nodes should be involved.
   @nodes_to_add = split (',', $CFG->params('NODE_NAME_LIST'));
   $success = add_Nodeapps($upgrade_option, \@crs_nodevip_list,
                           $DHCP_flag, \@nodes_to_add, \@nodes_to_start);

   # Trigger the active version change.
   if ($success && (! setActiveversion())) {
      $success = FAILED;
   }

   if ($success) { backupOCR(); }

   #Check for old ASM having a different owner than the grid owner
   # and change it to grid owner.

   if (isASMExists() && isOldVersionLT112()) {
       update_ASMowner();
   }

   if ($success &&
      ($CFG->params('ASM_UPGRADE') =~ m/false/i) &&
      (! isASMExists()) && !isFarASM()) {
      trace("Prior version ASM does not exist , Invoking add asm");
      add_ASM();  # add ora.asm
      if ($CFG->ASM_STORAGE_USED) {
         createDiskgroupRes();  # add disk group resource, if necessary
      }

   }

   if ($success &&
       add_GNS() &&
       add_scan() &&
       add_scan_listener() &&
       add_J2EEContainer()) {
       $success = SUCCESS;
   } else {
       $success = FAILED;
   }

   if (($CFG->params('MGMT_DB') =~ m/true/i) && !add_mgmt_db_listener()) {
      $success = FAILED;
   }

   if ($success) {
      add_CVU ();
   }

   if (($CFG->FORCE) && (scalar(@force_upgrade_nodes) > 0))
   {
     trace("Start nodeapps on force upgrade nodes");
     @nodes_to_start = @force_upgrade_nodes;
   }

   if ($success &&
       start_Nodeapps($DHCP_flag, \@nodes_to_start) &&
       start_GNS() &&
       start_scan() &&
       start_scan_listener() &&
       start_J2EEContainer())
   {
       $success = SUCCESS;

   } else {
       $success = FAILED;
   }

   if ($success) {
     start_CVU();
   }

   # wait for cluster to come up
   check_service("cluster", 60);

   if ($success)
   {
     upgradeNodeListener();
   }
  
   return $success;
}

sub is112ASMExists
#-------------------------------------------------------------------------------
# Function:  Check if 11.2 ASM exists
# Args    :  none
# Returns :  TRUE  if     exists
#            FALSE if not exists
#-------------------------------------------------------------------------------
{
   if (check_service("ora.asm", 1)) {
      return TRUE;
   }
   else {
      return FALSE;
   }
}

sub upgrademodel
{
   my $nodeinfo = $_[0];
   my $host = tolower_host ();
   my @old_version = @{$CFG->oldconfig('ORA_CRS_VERSION')};
   my $srcver = join('.',@old_version);
   my $destver = getcursoftversion($host);
   my $success = TRUE;
   my $run_as_owner = FALSE;

   chomp $srcver;

   my $status = srvctl($run_as_owner,
                          "upgrade model  -s $srcver -d $destver -p $nodeinfo");
   if (${status}) {
      trace ("srvctl upgrade model -$nodeinfo  ... passed");
   }
   else {
      print_error(180, "srvctl upgrade model -s $srcver -d $destver " .
                  " -p $nodeinfo", ${status});
      $success = FALSE;
   }

   return $success;
}

sub update_ASMowner
#-------------------------------------------------------------------------------
# Function:  update ASM owner to grid owner
# Args    :  none
# Returns :  none
#-------------------------------------------------------------------------------
{
   trace("Updating ASM owner if required");
   my $CRSCTL  = catfile ($CFG->params('ORACLE_HOME'), 'bin', 'crsctl');
   my $rc;
   my @out;
   my $ohown = $CFG->params('ORACLE_OWNER');
   my $cmd;

   $cmd = "$CRSCTL stat resource -w 'NAME en .asm'";
   @out = system_cmd_capture("$cmd");
   $rc = shift @out;

   my @cmdout = grep(/NAME/, @out);

   trace("ASM resource list is @cmdout");

   foreach my $str (@cmdout) {

     my ($tag, $resname) = split(/=/, $str);
     chomp ($resname);
     if ($resname =~ "ora.asm") {
       trace("skip update owner for ora.asm resource");
       next;
     }
     $cmd = "$CRSCTL getperm resource $resname";
     @out = system_cmd_capture("$cmd");
     $rc = shift @out;

     my @cmdout = grep(/^owner:/, @out);

     trace("$resname ACL is @cmdout");
     my @val = split(/:/, $cmdout[0]);

     my $asmown = $val[1];
     chomp ($asmown);

     trace("asm owner is $asmown");

     if ($asmown =~ $ohown ) {
         trace("ASM resource owner is the same as grid owner.Nothing to be done");
         next;
     }

     $cmd = "$CRSCTL setperm resource $resname -o $ohown";
     @out = system_cmd_capture("$cmd");
     $rc = shift @out;


     if ($rc == 0) {
        trace ("Successfully updated owner for resource $resname");
     } else {
        error ("Failed to update owner for resource $resname");
     }
   }
}

sub getcursoftversion
{
    my $host = $_[0];
    my $cmd;
    my @out;
    my $rc;
    my $verstring;

    if (isOCRonASM()) {
     trace("setting ORAASM_UPGRADE to 1");
     $ENV{'ORAASM_UPGRADE'} = "1";
    }

    setConfiguredCRSHome();
    trace("Getting current software version on $host");
    my $crsctl = catfile(getConfiguredCRSHome(), 'bin', 'crsctl');
    if ($CFG->SIHA) {
      $cmd = "$crsctl query has softwareversion";
    } else {
      $cmd = "$crsctl query crs softwareversion $host";
    }

    @out = system_cmd_capture("$cmd");
    $rc = shift @out;

    if ($rc == 0) {
      $verstring = getVerInfo($out[0]);
      trace( "Got CRS softwareversion for $host: ".join('.', $verstring) );
    }
    else {
      trace ("$cmd ... failed rc=$rc with message:\n @out \n");
      trace ("Getting software version for $host from OCR");
      $verstring = get_softVersionfromOCR($host);
      trace("software version retreived from OCR is $verstring");
    }

    chomp $verstring;
    trace ("The software version on $host is $verstring");
    return $verstring;
}

sub get_softVersionfromOCR
{
   my $nodename = $_[0];
   my $OCRDUMPBIN  = catfile ($CFG->ORA_CRS_HOME, 'bin', 'ocrdump');

   if ($CFG->JOIN) { return get_softVersionfromOCR2($nodename); }

   if (isOldVersionLT112())
   {
     trace("Searching the OCR dump for software version");
     my ($key, $softver);

     open (OCRDUMP, "$OCRDUMPBIN -stdout -noheader -keyname " .
       "SYSTEM.css.node_names.$nodename.nodenum |");
     my @output = <OCRDUMP>;
     close(OCRDUMP);
    
     trace("The ocrdump output for node num on $nodename is @output");
     my @val = grep(/UB4/, @output);
     if (0 == scalar(@val))
     {
       trace("Failed to retrieve the node num for $nodename");
       return $softver;
     }

     my ($valueType, $nodenum) = split(/:/, $val[0]);
     $nodenum = trim($nodenum);
     trace("The node num for $nodename is $nodenum");

     open (OCRDUMP, "$OCRDUMPBIN -stdout -noheader -keyname " .
       "SYSTEM.version.node_numbers.node$nodenum |");
     @output = <OCRDUMP>;
     close(OCRDUMP);
     
     trace("ocrdump output for software version on $nodename is @output");
     my @txt = grep(/ORATEXT/, @output);
     if (0 == scalar(@txt))
     {
       trace("Failed to retrieve the software version on $nodename");
       return $softver;
     }

     ($key, $softver) = split(/:/, $txt[0]);
     $softver = trim($softver);
     trace("The software version on $nodename is $softver");
    
     return $softver;
   }

   trace("Getting crs software verison from ocrdump");
   if ($CFG->platform_family eq "windows") {
      open (OCRDUMP, "$OCRDUMPBIN -stdout -noheader -keyname " .
              "SYSTEM.version.hostnames.$nodename |");
   }
   else {
      open (OCRDUMP, "$OCRDUMPBIN -stdout -noheader -keyname " .
                  "'SYSTEM.version.hostnames.$nodename'|");
   }
   my @output = <OCRDUMP>;
   close (OCRDUMP);

   trace("ocrdump output for software version on $nodename is @output");
   my @txt = grep (/ORATEXT/, @output);
   my ($key, $softver) = split (/:/, $txt[0]);
   $softver = trim($softver);
   trace ("key is $key");
   trace ("softver is $softver");    

   return $softver;
}

sub get_softVersionfromOCR2
{
   my $nodename = $_[0];
   my $OCRDUMPBIN  = catfile ($CFG->ORA_CRS_HOME, 'bin', 'ocrdump');

   trace("Getting crs software verison from ocrdump");
   if ($CFG->platform_family eq "windows") {
      open (OCRDUMP, "$OCRDUMPBIN -stdout -noheader -keyname " .
              "SYSTEM.version.hostnames.$nodename |");
   }
   else {
      open (OCRDUMP, "$OCRDUMPBIN -stdout -noheader -keyname " .
                  "'SYSTEM.version.hostnames.$nodename'|");
   }
   my @output = <OCRDUMP>;
   close (OCRDUMP);

   trace("ocrdump output for software version on $nodename is @output");
   my @txt = grep (/ORATEXT/, @output);
   my ($key, $softver) = split (/:/, $txt[0]);
   $softver = trim($softver);
   trace ("key is $key");
   trace ("softver is $softver");

   if (! $softver)
   {
     trace("Searching the OCR dump for software version");

     open (OCRDUMP, "$OCRDUMPBIN -stdout -noheader -keyname " .
       "SYSTEM.css.node_names.$nodename.nodenum |");
     my @output = <OCRDUMP>;
     close(OCRDUMP);

     trace("The ocrdump output for node num on $nodename is @output");
     my @val = grep(/UB4/, @output);
     if (0 == scalar(@val))
     {
       trace("Failed to retrieve the node num for $nodename");
       return $softver;
     }

     my ($valueType, $nodenum) = split(/:/, $val[0]);
     $nodenum = trim($nodenum);
     trace("The node num for $nodename is $nodenum");

     open (OCRDUMP, "$OCRDUMPBIN -stdout -noheader -keyname " .
       "SYSTEM.version.node_numbers.node$nodenum |");
     @output = <OCRDUMP>;
     close(OCRDUMP);

     trace("ocrdump output for software version on $nodename is @output");
     my @txt = grep(/ORATEXT/, @output);
     if (0 == scalar(@txt))
     {
       trace("Failed to retrieve the software version on $nodename");
       return $softver;
     }

     ($key, $softver) = split(/:/, $txt[0]);
     $softver = trim($softver);
     trace("The software version on $nodename is $softver");
   }

   return $softver;
}

sub upgrade_clscfg
{
   # execute clscfg -upgrade
   my $status;
   my $asmgrp = $CFG->params('ORA_ASM_GROUP');

   if ($CFG->FORCE)
   {
      $status = run_crs_cmd('clscfg', '-upgrade', '-g', $asmgrp, '-lastnode');
   }
   else
   {
      $status = run_crs_cmd('clscfg', '-upgrade', '-g', $asmgrp);
   }

   if ($status == 0) {
      trace ("clscfg -upgrade completed successfully");
   }
   else {
      print_error(180, "clscfg -upgrade", $status);
      exit 1;
   }

   # execute 'oifcfg setif -global'
   if (isLastNodeToUpgrade()) {
      $status = run_crs_cmd('oifcfg', 'setif', '-global');

      # modify ACTION_SCRIPT
      ModActionScript();
   }
}

sub ModActionScript 
#-------------------------------------------------------------------------------
# Function:  Modify all db resources ACTION_SCRIPT and set it to %crs_home%
# Args    :  none
#-------------------------------------------------------------------------------
{
   trace("Modify ACTION_SCRIPT");
   # get db resources
   my $crsctl = crs_exec_path('crsctl');
   my @cmd    = ($crsctl, 'stat', 'res', '-w',
                 '"((TYPE = application) AND (NAME en .db) AND (NAME st ora.))"');
   my @output = system_cmd_capture(@cmd);
   my $rc     = shift @output;
   my @resname = grep(/NAME=/, @output);

   trace("output=@output");
   trace("resname=@resname");

   # modify ACTION_SCRIPT
   my ($racgwrap, $dummy, $res);

   if ($CFG->platform_family eq "windows") {
      $racgwrap = catfile('%CRS_HOME%', 'bin', 'racgwrap.bat');
   }
   else {
      $racgwrap = catfile('%CRS_HOME%', 'bin', 'racgwrap');
   }

   foreach (@resname) {
      ($dummy, $res) = split ('NAME=');
      @cmd = ($crsctl, 'modify', 'res', $res, '-attr',
              "\"ACTION_SCRIPT=$racgwrap\"");
      $rc  = system_cmd(@cmd);

      if ($rc == 0) {
         trace("@cmd ... success");
      }
      else {
         trace("@cmd ... failed");
      }
   }
}

sub modifyClusterName
{
   trace("modify CLUSTER_NAME in crsconfig_params");

   my $params_file = $CFG->paramfile;
   my @params_table = read_file ($params_file);
   my $updateParamsFile = FALSE;
   my $ix = 0;

   foreach my $rec (@params_table) {
      chomp($rec);
      if ($rec =~ m/^CLUSTER_NAME=/) {
         my ($key, $value) = split (/=/, $rec);
         if (! $value) {
            my $cluster_name = getClusterName();
            if ($cluster_name) {
               $params_table[$ix] = 'CLUSTER_NAME=' . "$cluster_name";
               $updateParamsFile = TRUE;
               last;
            }
         }
      }

      $ix++;
   }

   if ($updateParamsFile) {
      # save original params file
      my $save_file = catfile (dirname($params_file), 'crsconfig_params.saved');
      copy_file ($params_file, $save_file);

      # delete old file and create new file
      if (s_remove_file($params_file)) {
         open (SFILE, ">$params_file")
            || die(dieformat(207, $params_file, $!));

         foreach my $rec (@params_table) {
            chomp($rec);
            print SFILE "$rec\n";
         }

         close SFILE;
      }
   }
}

sub getNotupgradedNodes
{
  trace("Find out which nodes didn't get upgraded");
  
  my @notupgradedNodes;
  my $softver;
  my $relver = getcrsrelver($CFG->ORA_CRS_HOME);
  my @clunodes = split(',', $CFG->params('NODE_NAME_LIST'));
  foreach my $clunode (@clunodes)
  {
    if ($CFG->HOST =~ /^$clunode$/i) { next; }
    $softver = getcursoftversion($clunode);
    if (!isVersionMatch($relver, $softver))
    {
      push(@notupgradedNodes, $clunode);
    }
  }

  trace("Not upgraded nodes: @notupgradedNodes");
  return @notupgradedNodes;
}

# For a force upgrade case, remove the ASM node resources of un-upgraded nodes
# so that ASMCA can succeed after rootupgrade.sh 
sub removeASMNodeRes
{
  my $ckptStatus;
  my $ckptName = "ROOTCRS_FORCEUPG_RMASMRES";
  if (isCkptexist($ckptName))
  {
    $ckptStatus = getCkptStatus($ckptName);
    trace("'$ckptName' state is $ckptStatus");

    if (($ckptStatus eq CKPTSUC) && ($CFG->FORCE))
    {
      trace("Removal of ASM node resources of un-upgraded nodes is already done");
      $CFG->wipCkptName("ROOTCRS_STACK");
      return SUCCESS;
    }
  } 

  writeCkpt($ckptName, CKPTSTART);
  $CFG->wipCkptName($ckptName);

  my %nodeNumHash;
  my @nodeNumArray = split(/,/, getNodeNumList($CFG->ORA_CRS_HOME));
  foreach my $elem (@nodeNumArray)
  {
    my @val = split(/:/, $elem);
    $nodeNumHash{${val[0]}} = $val[1];
  }

  my $crsctl = crs_exec_path('crsctl');
  my @notupgradedNodes = getNotupgradedNodes();
  foreach my $node (@notupgradedNodes)
  {
    my $nodeNum = $nodeNumHash{$node};
    my $resName = "ora\.".$node."\.ASM$nodeNum"."\.asm"; 
    my @out = system_cmd_capture($crsctl, "stat", "resource", $resName);

    my $asmhome = crsctlRemoveASM($resName, $node);
    srvctlRemoveASM($resName, $node, $asmhome);
  }

  writeCkpt($ckptName, CKPTSUC); 
  $CFG->wipCkptName("ROOTCRS_STACK");
  return SUCCESS;
}

sub crsctlRemoveASM
{
  my $resName   = $_[0];
  my $node_name = $_[1];
  my $ckptName  = "ROOTCRS_FORCEUPG_RMASMRES";
  my $asmhome;

  trace("Romve ASM resource <$resName> on node <$node_name>");
  my $crsctl = crs_exec_path('crsctl');
  my @output = system_cmd_capture($crsctl, "stat", "resource", $resName);
  my $rc     = shift(@output);
  if ((0 == $rc) && (scalar(grep(/CRS-2613/, @output)) > 0))
  {
    trace("ASM resource <$resName> does not exist");
  }
  else
  {
    trace("Trying to retrieve ASM home for <$resName>");
    @output = system_cmd_capture($crsctl, "stat", "res", $resName, "-p");
    $rc     = shift(@output);
    if (0 != $rc)
    {
      writeCkpt($ckptName, CKPTFAIL); 
      $CFG->wipCkptName("ROOTCRS_STACK");
      die(dieformat(180, "crsctl stat res $resName -p", $rc));
    }

    foreach my $line (@output)
    {
      if ($line =~ /ACTION_SCRIPT/)
      {
        my $actionScript = (split(/=/, $line))[1];
        $asmhome = (split(/bin/, $actionScript))[0];
        trace("ASM home = $asmhome");
        chop($asmhome);
        trace("ASM home after chopping: $asmhome");
        last;
      }
    }

    if (! $asmhome)
    { 
      writeCkpt($ckptName, CKPTFAIL);
      $CFG->wipCkptName("ROOTCRS_STACK");
      die("Unexpected error in retrieving ASM home for ASM reource '$resName'\n");
    }
    else
    {
      writeCkptProperty($ckptName, $resName, $asmhome);
    }

    trace("Remove ASM resource: <$resName>");
    @output = system_cmd_capture($crsctl, "delete", "resource", $resName, "-f");
    $rc = shift @output;
    if (0 != $rc)
    {
      writeCkpt($ckptName, CKPTFAIL);
      $CFG->wipCkptName("ROOTCRS_STACK");
      die(dieformat(180, "crsctl delete resource $resName", $rc));
    }
  }

  if (! $asmhome)
  {
    trace("Reading from CKPT file ...");
    $asmhome = trim(getCkptPropertyValue($ckptName, $resName));
  }

  if (! $asmhome)
  {
    writeCkpt($ckptName, CKPTFAIL);
    $CFG->wipCkptName("ROOTCRS_STACK");
    die("Unexpected error in retrieving ASM home for ASM reource '$resName'\n") 
  }

  return $asmhome;
}

sub srvctlRemoveASM
{
  my $resName   = $_[0];
  my $node_name = $_[1];
  my $asmhome   =  $_[2];
  my $ckptName  = "ROOTCRS_FORCEUPG_RMASMRES";

  trace("Remove OCR keys for ASM resource: <$resName>");
  my $old_env = $ENV{'ORACLE_HOME'};
  $ENV{'ORACLE_HOME'} = $asmhome;

  trace("Invoke srvctl from ASM home: $asmhome");
  my $srvctl = catfile($asmhome, 'bin', 'srvctl');
  my @output = system_cmd_capture($srvctl, 'remove', 'asm', '-n', $node_name, '-f');

  if ($old_env)
  {
    $ENV{'ORACLE_HOME'} = $old_env;
  }
  else
  {
    undef $ORACLE_HOME;
  }

  my $rc = shift(@output);
  if ((0 != $rc) && (2 != $rc))
  {
    writeCkpt($ckptName, CKPTFAIL);
    $CFG->wipCkptName("ROOTCRS_STACK");
    die(dieformat(180, "srvctl remove asm -n $node_name -f", $rc));
  }

  return SUCCESS;
}

sub asmcaJoinCluster
{
  my $asmca = catfile($CFG->ORA_CRS_HOME, "bin", "asmca");
  my @runasmca = ($asmca, '-silent', '-joinCluster');
 
  trace ("Executing as " . $CFG->params('ORACLE_OWNER') . ": @runasmca");
  my $status = run_as_user($CFG->params('ORACLE_OWNER'), @runasmca);
  if (0 != $status)
  {
    trace("Failed to upgrade ASM on a joining node; return error: $status");
    return FAILED;
  }

  my $localNode = $CFG->HOST;
  srvctl(TRUE, "start asm -n $localNode");  

  return SUCCESS;
}

sub getClusterName
{
   my $CLUSTER_NAME = $CFG->params('CLUSTER_NAME');
   if ($CLUSTER_NAME)
   {
     trace("CLUSTER_NAME=$CLUSTER_NAME");
     return $CLUSTER_NAME;
   }

   my $olsnodes = catfile($CFG->ORA_CRS_HOME, 'bin', 'olsnodes');
   my @cmd = ($olsnodes, '-c');
   my @out = system_cmd_capture(@cmd);
   my $rc = shift @out;
   my $cluster_name;

   if ($rc == 0) {
      $cluster_name = $out[0];
   }

   chomp($cluster_name);
   trace("cluster_name=$cluster_name");
   $CFG->params('CLUSTER_NAME', $cluster_name);
   return $cluster_name;
}

sub upgsihamodel
#-------------------------------------------------------------------------------
# Function: call srvctl upgrade model for SIHA
# Args    : none
# Returns : SUCCESS or FAILED
#-------------------------------------------------------------------------------
{

   trace("Invoking srvctl upgrade model for SIHA");

   upgrademodel(FIRST_NODE);
   upgrademodel(LAST_NODE);
}

=head2 cleanASMFiles

  Function for removing ASM files during upgrade from 11201/11202

=head3 Parameters
  
  None

=head3 Returns

  None

=cut

sub cleanASMFiles
{
  if ($CFG->platform_family eq 'windows') { return; }

  # The setasmgid file is present in /opt/oracle/bin in 11.2.0.1 and 11.2.0.2.
  # It needs to be cleaned only for those versions. 
  my @oldCrsVer = @{$CFG->oldconfig('ORA_CRS_VERSION')};
  my $verinfo = join('.', @oldCrsVer);
  if ((0 != versionComparison($verinfo, "11.2.0.1.0")) &&
      (0 != versionComparison($verinfo, "11.2.0.2.0")))
  {
    trace("Old version not eligible. Skipping clean ...");
    return;
  }

  trace("Cleaning up ASM files ...");

  my $optorcldir    = $CFG->params('EXTERNAL_ORACLE');
  my $optorclbindir = $CFG->params('EXTERNAL_ORACLE_BIN');
  my $ORACLE_OWNER  = $CFG->params('ORACLE_OWNER');
  my $ORA_DBA_GROUP = $CFG->params('ORA_DBA_GROUP');

  if (-d $optorclbindir)
  {
    my @res;

    opendir(DIR, $optorclbindir);
    my @files = readdir(DIR);
    close DIR;

    foreach my $file (@files)
    {
      if ($file =~ /^setasmgid/)
      {
        s_remove_file("$optorclbindir/$file");
      }
      else
      {
        push(@res, $file);
      }
    }

    my $noEmpty = 0;
    foreach my $file (@res)
    {
      if (($file eq '.') || ($file eq '..'))
      {
        next;
      }

      $noEmpty = 1;
    }

    if (0 == $noEmpty)
    {
      trace("Removing $optorclbindir");
      rmtree($optorclbindir);
    }
  }

  # We don't want to fail the upgrade if the clean fails, so 
  # we just trace it and move forward.
  if (-d $optorclbindir)
  {
    trace("Could not reomve the directory '$optorclbindir'");
  }

  if (-d $optorcldir)
  {
    # When $optorcldir is owned by root
    if (-o "$optorcldir")
    {
      trace("Change the owner of $optorcldir to $ORACLE_OWNER");
      if (FAILED == s_set_ownergroup($ORACLE_OWNER,
                                      $ORA_DBA_GROUP, $optorcldir))
      {
        trace("Could not change ownership on $optorcldir");
      }
    }

    # Skipping this if $optorcldir/.ssh exists
    my $sshDir = catfile($optorcldir, '.ssh');
    if (! (-e $sshDir))
    {
      trace("Change the permissons of $optorcldir");
      if (FAILED == s_set_perms("0775", $optorcldir))
      {
        trace("Could not change permissions on $optorcldir");
      }
    }
  }
}

=head2 upgradeASM 

  Perform ASM rolling/non-rolling upgrade

=head3 Parameters

  [0] nodeInfo  :  FIRST_NODE or LAST_NODE
  [1] isRolling :  TRUE or FALSE
  [2] oldCRSHOME:  old CRS home
  [3] nodeList  :  <node1>:<number1>,<node2>:<number2>...
  [4] localNode :  local node name

=head3 Returns

  SUCCESS or FAILED

=cut

sub upgradeASM
{
  my $nodeInfo   = $_[0];
  my $isRolling  = $_[1];
  my $oldCRSHome = $_[2];
  my $nodeList   = $_[3];
  my $localNode  = $_[4];
  my $upgMode;
  my $asmca;
  my @runasmca;
  my $status;
  my $ret = SUCCESS;

  trace("nodeInfo   = $nodeInfo");
  trace("isRolling  = $isRolling");
  trace("oldCRSHome = <$oldCRSHome>");
  trace("nodeList   = <$nodeList>");
  trace("localNode  = <$localNode>");

  $asmca = catfile($CFG->ORA_CRS_HOME, "bin", "asmca");

  # Do not change the order of these parameters as asmca requires the
  # parameters to be in a specific order or it will fail
  ($isRolling) ? ($upgMode = "false") : ($upgMode = "true");

  if ($nodeInfo eq FIRST_NODE)
  {
    trace("Starting ASM upgrade on first node");
    @runasmca = ($asmca, '-silent', '-upgradeLocalASM', '-firstNode',
                  '-nonRolling', $upgMode, '-oldCRSHome', $oldCRSHome,
                  '-localNode', $localNode, '-nodeListUpgrade', $nodeList);
  }
  else
  {
    trace("Calling asmca with lastNode option is unnecessary");
    return FAILED;
  }

  trace ("Executing as " . $CFG->params('ORACLE_OWNER') . ": @runasmca");
  $status = run_as_user($CFG->params('ORACLE_OWNER'), @runasmca);

  if (0 != $status)
  {
    $ret = FAILED;
    print_error(164);
  }

  return $ret;
}

=head2 upgCreateCredDomains
  
  Create the credential domains for ASM and Gridhome during upgrade
    
=head3 Parameters

  None 
  
=head3 Returns
  
  None
  
=cut

sub upgCreateCredDomains
{
  trace("Create the credential domains for ASM and Gridhome");
  
  createCredDomain('ASM', 'OLR') || die(dieformat(377));

  if (isLastNodeToUpgrade())
  {
    createCredDomain('ASM', 'OCR') || die(dieformat(377));
    createCredDomain('GRIDHOME', 'OCR')  || die(dieformat(380));
  }
}

=head2 preForceupgChecks
  
  Pre-checks for force upgrade
    
=head3 Parameters

  None 
  
=head3 Returns
  
  SUCCESS if successful 
  FAILED if failed
  
=cut

sub preForceupgChecks
{
  trace("Performing force upgrade pre checks ...");

  my $relver = getcrsrelver($CFG->ORA_CRS_HOME);  
  my @activever = get_crs_version($CFG->oldconfig('ORA_CRS_HOME'));
  # This checks that the cluster has not already been upgraded
  if (isVersionMatch($relver, join('.', @activever)))
  {
    my $ckptstatus = getCkptStatus("ROOTCRS_STACK");
    if ($ckptstatus eq CKPTSUC)
    {
      print_error(445, $relver);
      return FAILED;
    }
  }

  my $softver = getcursoftversion($CFG->HOST);
  # This checks that the current node has been upgraded
  if (!(isVersionMatch($relver, $softver)))
  {
    print_error(446);
    return FAILED;
  }

  my @upgradedNodes;
  my @notupgradedNodes;

  my @clunodes = split(',', $CFG->params('NODE_NAME_LIST'));
  my %nodeStatus = getNodeStatus($CFG->ORA_CRS_HOME);
  foreach my $clunode (@clunodes)
  {
    if (1 == $nodeStatus{$clunode}) # active node
    {
      push(@upgradedNodes, $clunode);
    }
    else
    {
      $softver = getcursoftversion($clunode);
      # The inactive node list should only contain un-upgraded nodes
      if (!isVersionMatch($relver, $softver))
      {
        push(@notupgradedNodes, $clunode);
      }
      else
      {
        trace("The node $clunode may have been upgraded ".
              "but is inactive right now");
        push(@upgradedNodes, $clunode);
      }
    }
  }

  trace("Cluster nodes : @clunodes");
  trace("Upgraded nodes  : @upgradedNodes");
  trace("Not upgraded nodes: @notupgradedNodes");

  my $forceupg = 1;
  # We must make sure that the force upgrade is applicable to
  # all active nodes 
  foreach my $clunode (@upgradedNodes)
  {
    if ($CFG->HOST =~ /^$clunode$/i) { next; }
    $softver = getcursoftversion($clunode);
    if (!isVersionMatch($relver, $softver))
    {
      print_error(447, $clunode);
      $forceupg = 0;
    }
  }

  if (0 == $forceupg)
  {
    return FAILED;
  }

  @force_upgrade_nodes = @upgradedNodes;

  return SUCCESS;
}

sub preUpgChecks
{
  if (! $CFG->UPGRADE) { return; }
 
  my $profile = catfile($CFG->params('GPNPGCONFIGDIR'), 'gpnp',
                         'profiles', 'peer', 'profile.xml');
  if (isOldVersionLT112())
  {
    if ($CFG->JOIN) { return; }
    if ((! isFirstNodeToUpgrade()) && (! -e $profile))
    {
      die(dieformat(448));
    }
  }
}

=head2 get_CSSD_loglevel_fromOld
  
  The function for retrieving CSSD log level from the old stack
    
=head3 Parameters

  None 
  
=head3 Returns
  
   Current CSSD log level if successful 
  -1                      if failed
  
=cut

sub get_CSSD_loglevel_fromOld
{
  my $ll = -1;
  my @cmd;
  my @out;
  my $rc;
  my $CRSCTL = catfile($CFG->OLD_CRS_HOME, 'bin', 'crsctl');

  if (isOldVersionLT112())
  {
    trace("Attempt to retrieve CSSD log level from pre-11.2");
    @cmd =  ($CRSCTL, 'get', 'css', 'trace');
    @out = system_cmd_capture(@cmd);
    $rc = shift(@out);

    if ((0 == $rc) && ($out[0] =~ /(\d+)/))
    {
      $ll = $1;
      trace("Log level obtained is: $ll");
    }
  }
  else
  {
    trace("Attempt to retrieve CSSD log level from 11.2 or post-11.2");
    @cmd = ($CRSCTL, 'get', 'log', 'css', "\"CSSD\"");
    @out = system_cmd_capture(@cmd);
    $rc = shift(@out);
    
    if ((0 == $rc) && ($out[0] =~ /Log Level:\s+(\d+)/))
    {
      $ll = $1;
      trace("Log level obtained is: $ll");
    }
  }

  return $ll;
}

=head2 set_CSSD_loglevel 
  
  The function for setting log level for CSSD
    
=head3 Parameters

  [0] Log level set for CSSD 
  [1] CRS home
  
=head3 Returns
  
  SUCCESS or FAILED
  
=cut

sub set_CSSD_loglevel
{
  my $ll = $_[0];
  my $crshome = $_[1];  
  my $CRSCTL = catfile((($crshome) ? ($crshome) : ($CFG->ORA_CRS_HOME)),
                        'bin', 'crsctl');

  trace("Attempt to set CSSD log level to $ll");

  my @cmd = ($CRSCTL, 'set', 'log', 'css', "\"CSSD=$ll\""); 
  my @out = system_cmd_capture(@cmd);
  my $rc = shift(@out);

  if (0 != $rc)
  {
    trace("Failed to set CSSD log level to $ll");
    return FAILED;
  }

  return SUCCESS;
}




1;
