#!/usr/local/bin/perl
#
# $Header: has/install/crsconfig/oraafd.pm /main/1 2012/09/21 03:52:21 shmubeen Exp $
#
# oraafd.pm
#
# Copyright (c) 2008, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      oraafd.pm - Library module for ASM Filter Driver root install functions.
#
#    DESCRIPTION
#      oraafd.pm - Contains initial installation and deinstallation
#                   routines for AFD.
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#    shmubeen    07/18/12 - Initial Creation
#

package oraafd;
use strict;
use English;
use File::Temp qw/ tempfile /;
use File::Spec::Functions;
use File::Find ();

use crsutils;

use Exporter;
use vars qw(@ISA @EXPORT @EXPORT_OK);
@ISA = qw(Exporter);

my @exp_func  = qw(installAFDriver
                   removeAFDriver
                   unloadAFDriver
                   isAFDSupported);

push @EXPORT, @exp_func;

sub installAFDriver
{
   my $afdroot;
   my $ret = SUCCESS;
   my $afdInstall = FALSE;

   #shmubeen TODO: remove later
   $afdInstall = uc($ENV{'ORA_ENABLE_AFD_INSTALL'});
   if ( $afdInstall ne "TRUE" )
   {
     trace("AFD disabled because of ENV in test mode");
     return $ret;
   }

   trace("installAFDriver");
   if (is_dev_env())
   {
      $afdInstall = uc($ENV{'ORA_ENABLE_AFD_INSTALL'});

      # if this ENV is not set then we give up early
      if ( $afdInstall ne "TRUE" )
      {
         trace("AFD disabled because of ENV in test mode");
         return $ret;
      }
   }

   if ($CFG->platform_family eq 'unix') {
      $afdroot = catfile ($CFG->ORA_CRS_HOME, 'bin', 'afdroot');
   }
   elsif ($CFG->platform_family eq 'windows') {
      trace("AFD is not supported on this platform");
      $ret = FAILED; 
      return $ret;
   }

   if (-e $afdroot) {
      my $cmd = "$afdroot install";
      
      trace("Executing '$cmd'");
      my @output = system_cmd_capture($cmd);
      my $rc = shift @output;

      if ($rc == 0) {
        trace("AFD is installed successfully."); 
      }
      elsif ($rc != 2) {
         # 629 covers unload failures.
         # 650 covers install and load failures.
         #   (driver currently running, driver in use can't copy file,
         #    driver unable to load because Windows thinks its running
         #    but its not.)
         if ((scalar(grep(/629/, @output))) ||
             (scalar(grep(/650/, @output))) > 0)
         {
           $ret = REBOOT;
         }
      }
      else{
        $ret = FAILED;
        trace("AFD installation failed.");
      }
   }
   else {
      trace("$afdroot not found");
      if (isAFDSupported())
      {
        # if afdroot not found and AFD supported, we have a problem
        # some required files are not here.
        trace("AFD is supported on this platform, but install files are missing.");
        $ret = FAILED;
      }
      else
      {
        # If afdroot not found and AFD not supported,
        # then assume everything is okay.
        trace("AFD is not supported on this platform.");
        $ret = SUCCESS;
      }
   }

  trace("AFD driver install status is $ret");
  return $ret;
}

sub removeAFDRoot
#-------------------------------------------------------------------------------
# Function: remove AFD drivers
#-------------------------------------------------------------------------------
{
   my $chkpoints = $_;
   my $afdroot;
   my $ret;  

   if ($CFG->platform_family eq 'unix') {
      $afdroot = catfile ($CFG->ORACLE_HOME, 'bin', 'afdroot');
   }
   elsif ($CFG->platform_family eq 'windows') {
      trace("AFD is not supported on this platform");
      $ret = FAILED; 
      return $ret;
   }

   if (! (-e $afdroot)) {
      trace ("AFD is not configured");
      $ret = FAILED; 
      return $ret;
   }

   if ( $chkpoints == 1 )
   { 
     if (!isCkptexist("ROOTCRS_AFDUNINST")) 
     {
       trace("Writing checkpoint for AFD driver uninstall");
       writeCkpt("ROOTCRS_AFDUNINST", CKPTSTART);
       $CFG->wipCkptName("ROOTCRS_AFDUNINST");
     }
   }

   if ( ($chkpoints == 0) || !isCkptSuccess("ROOTCRS_AFDUNINST"))
   {
      $CFG->wipCkptName("ROOTCRS_AFDUNINST");

      my @cmd    = ($afdroot, 'uninstall');
      trace ("Executing @cmd");
      my @output = system_cmd_capture(@cmd);
      my $rc = shift @output;

      if ( $rc == 0 ) {
         trace ("@cmd ... success");
         trace("AFD drivers uninstall completed");
         if( $chkpoints == 1 )
         {
           writeCkpt("ROOTCRS_AFDUNINST", CKPTSUC);
         }
         $ret = SUCCESS;
      }
      elsif ( $rc != 2 )
      {
         if ((scalar(grep(/9118/, @output))) ||
             (scalar(grep(/9119/, @output))) ||
             (scalar(grep(/9348/, @output))) > 0)
         {
            # We couldn't unload the old drivers or load the new ones.
            trace("ASM Filter drivers unable to be uninstalled.");
            print color 'bold';
            print_error(434);
            print color 'reset';
          }
          if ($chkpoints == 1)
           {
              writeCkpt("ROOTCRS_AFDUNINST", CKPTFAIL);
           }
           exit 1;
      }
      else 
      {
         trace ("@cmd ... failed");
         if( $chkpoints == 1 )
         {
           writeCkpt("ROOTCRS_AFDUNINST", CKPTFAIL);
         }
         exit 1;
      }
   }
}

sub LoadAFDriver
{
   my $chkpoints = $_;
   my $afdload;
   my $afdroot;
   my $ret;

   if ($CFG->platform_family eq 'unix') {
      $afdroot = catfile ($CFG->ORACLE_HOME, 'bin', 'afdload');
   }

   if (! (-e $afdload)) {
      trace ("AFD is not configured");
      return;
   }

   if ( $chkpoints == 1 )
   {
     if (!isCkptexist("ROOTCRS_AFDLOAD"))
     {
        trace("Writing checkpoint for AFD driver Load");
        writeCkpt("ROOTCRS_AFDLOAD", CKPTSTART);
        $CFG->wipCkptName("ROOTCRS_AFDLOAD");
     }
   } 

   if ( ($chkpoints == 0 ) || !isCkptSuccess("ROOTCRS_AFDLOAD"))
   {
      $CFG->wipCkptName("ROOTCRS_AFDLOAD");

      my @cmd    = ($afdload, 'start');
      trace ("Executing @cmd");
      my @output = system_cmd_capture(@cmd);
      my $rc = shift @output;

      if ( $rc == 0 ) {
         trace ("@cmd ... success");
         trace("AFD drivers Loading completed");
         if( $chkpoints == 1 )
         {
           writeCkpt("ROOTCRS_AFDLOAD", CKPTSUC);
         }  
      }
      else
      {
         trace ("@cmd ... failed");
         if( $chkpoints == 1 )
         {
           writeCkpt("ROOTCRS_AFDLOAD", CKPTFAIL);
         }
         exit 1;
      }
   }
}

sub isAFDSupported
{
   my $afddriverstate;
   my $AFD_supported = FALSE;

   #
   # We return early now to disable AFD in install scripts.
   # AFD will be enabled after sufficient testing.
   #
   return $AFD_supported;

   if ($CFG->platform_family eq 'unix') {
      $afddriverstate = catfile($CFG->ORACLE_HOME, 'bin', 'afddriverstate');
   }
   elsif ($CFG->platform_family eq 'windows') {
      trace("AFD is not supported on this platform");
      return FALSE;
   }

   # check if AFD is supported
   if (! (-e $afddriverstate)) {
      trace ("$afddriverstate not found");
      return FALSE;
   }

   my @cmd = ($afddriverstate, "supported");
   my @out = system_cmd_capture(@cmd);
   my $rc  = shift @out;

   if ($rc == 0) {

      # In dev env, we disable FD.
      if (!is_dev_env() || ($ENV{'ORA_ENABLE_AFD_INSTALL'} eq "true")) {
         $AFD_supported = TRUE;
         trace ("AFD is supported");
      }
      else {
         $AFD_supported = FALSE;
         trace ("AFD is not supported");
      }
   }
   else {
      $AFD_supported = FALSE;
      trace ("AFD is not supported");
   }

  return $AFD_supported;
}

sub unloadAFDriver
{
   my $afdload;
   my $ret = FAILED;

   if ($CFG->platform_family eq 'unix') {
      $afdload = catfile($CFG->ORACLE_HOME, 'bin', 'afdload');
   }

   # check if AFD is supported
   if (! (-e $afdload)) {
      trace ("$afdload not found");
      return FAILED;
   }

   my @cmd = ($afdload, "stop");
   my @out = system_cmd_capture(@cmd);
   my $rc  = shift @out;
   
   if ($rc == '') {
      $ret = SUCCESS;
   }
   else {
      $ret = FAILED;
      trace ("AFD is not supported");
   }

  return $ret;
}

sub isAFDLoaded
{
   my $AFD_loaded = FALSE;
   my $afddriverstate;
   my $ret;

   #
   # We return early now to disable AFD in install scripts.
   # AFD will be enabled after sufficient testing.
   #
    return $AFD_loaded;

   if ($CFG->platform_family eq 'unix') {
      $afddriverstate = catfile($CFG->ORACLE_HOME, 'bin', 'afddriverstate');
   }

   # check if AFD is supported
   if (! (-e $afddriverstate)) {
      trace ("$afddriverstate not found");
      return FAILED;
   }

   my @cmd = ($afddriverstate, "loaded");
   my @out = system_cmd_capture(@cmd);
   my $notloaded = grep (/false/i, @out);
   my $loaded = grep (/true/i, @out);

   if ($loaded eq 1) {
      $ret = "loaded";
      trace ("AFD is loaded");
    }
   elsif($notloaded eq 1) {
      $ret = "not loaded";
      trace ("AFD is not loaded");
   }
   else
   {
     trace ("Error in executing $afddriverstate"); 
     $ret = FAILED; 
   }

  return $AFD_loaded;
}

sub actionAFD
{
 my $action = shift;
 my $ret = FAILED;
 my $retval = FALSE;

 if (isAFDSupported())
 {
   $retval = isAFDLoaded();
   if($retval eq 0)
   {
      return;
   } 
   if ($retval eq "loaded") 
   {
      if($action eq "stop")
      {
        $retval = unloadAFDriver();   
        if ($retval eq 1)
        {
          trace("AFD is unloaded"); 
          $ret = SUCCESS;
        } 
        else
        {
          trace("Error while unloading AFD");
        }   
      }
   }   
   elsif ($action eq "start")
   {
     loadAFD();  
   } 
 }
}
