# Copyright (c) 2011, 2013, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      s_crsutils.pm - <one-line expansion of the name>
#
#    DESCRIPTION
#      OSD module for Windows
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    xyuan       04/16/13 - Fix bug 16471215 - call modifyEMService.pl
#    xyuan       04/04/13 - Fix bug 16579079
#    xyuan       03/15/13 - Fix bug 16495293
#    sidshank    03/04/13 - fix bug 16382128.
#    sidshank    01/25/13 - fix bug 16185238
#    sidshank    01/24/13 - fix bug 16175611.
#    gmaldona    10/31/12 - Create temporary file in OS TEMP directory
#    jmunozn     10/11/12 - Add s_get_qosctl_path function
#    sidshank    09/12/12 - fix bug 14511564.
#    sidshank    08/30/12 - fix bug 14530963.
#    shmubeen    08/08/12 - afd install functions
#    ysharoni    07/18/12 - bug 14266142 add node name to orcl owner in icacls
#    sidshank    05/23/12 - fix bug 14106919
#    sidshank    04/09/12 - Add Path module. Remove restore/redirect stdout
#                           subroutines.
#    ysharoni    03/27/12 - add s_gpnp_wallets_set_access_win
#    gmaldona    02/24/12 - I added a new function called s_run_as_user3
#    sidshank    02/07/12 - Remove OCFS related code
#    sidshank    01/12/12 - Adding dummy routine s_install_initd
#    sidshank    11/09/11 - Fix for tne bug 13352502
#    rvadraha    10/25/11 - Bug13247694, Fix ocfs upgrade
#    sidshank    08/22/11 - removing the workaround for 12739826
#    dpham       05/01/11 - New for 12c
# 
package s_crsutils;

use strict;
use Win32;
use Win32::NetAdmin qw(DOMAIN_ALIAS_RID_ADMINS GetAliasFromRID
                       LocalGroupIsMember GroupIsMember);
use Win32::TieRegistry (Delimiter => '/');
use Win32::Service;
use Win32API::File  qw(DeleteFile);
use File::Spec::Functions;
use File::Path;
use File::Temp qw/ tempfile /;
use Cwd;

# root script module
use crsutils;

# export vars and functions
use Exporter;
use vars qw(@ISA @EXPORT @EXPORT_OK);

@ISA = qw(Exporter);

my @exp_func = qw(s_check_SuperUser s_set_ownergroup s_reset_crshome
                  s_reset_crshome1 s_set_perms s_osd_setup
                  s_check_CRSConfig s_validate_olrconfig s_get_olr_file
                  s_validate_ocrconfig s_validateOCR s_reset_srvconfig
                  s_register_service s_unregister_service s_check_service
                  s_start_service s_run_as_user s_run_as_user2 s_init_scr
                  s_get_config_key s_isLink s_get_platform_family s_copyOCRLoc
                  s_getOldCrsHome s_redirect_souterr s_restore_souterr
                  s_stop_OldCrsStack s_RemoveInitResources s_CleanTempFiles
                  s_setParentDirOwner s_resetParentDirOwner s_checkOracleCM
                  s_ResetOLR s_ResetOCR s_ResetVotedisks s_createConfigEnvFile
                  s_isRAC_appropriate s_createLocalOnlyOCR s_is92ConfigExists
                  s_configureCvuRpm s_removeCvuRpm s_remove_file s_getAbsLink
                  s_removeGPnPprofile s_crf_check_bdbloc s_crf_remove_itab
                  s_is_HAIP_supported s_is_HAIP_NonFatal s_CheckNetworkConfig
                  s_houseCleaning s_add_upstart_conf s_NSCleanUp s_getGroupName
                  s_set_ownergroup_win 
		  s_configureAllRemoteNodes s_removeSCR s_install_initd
                  s_checkolrbackup s_restoreInitScripts s_restoreASMFiles 
                  s_restoreolrloc s_gpnp_wallets_set_access_win
                  s_run_as_user3 s_copyRegKey s_stopService s_isServiceRunning
		  s_stopDeltOldASM s_upgrade_services s_deltService s_setPermsASMDisks  
		  s_rm_afdinit_rclevel s_rm_afdinit_init s_rm_afdinit_rclevel
                  s_get_qosctl_path s_delete_ASM_Services_downgrade 
                 );

push @EXPORT, @exp_func;

####---------------------------------------------------------
#### Function for checking and returning Super User name
# ARGS : 1
# ARG1 : Program name
sub s_check_SuperUser
{
    trace ("Checking for super user privileges");

    my $superUser = $ENV{'USERNAME'};
    my $groupName = s_getGroupName();
    trace ("superUser=$superUser groupName=$groupName");

    # get group name for Administrators
    if (! $groupName) {
        return "";
    }

    # get user-name
    my $userName   = Win32::LoginName();
    $userName      =~ tr/a-z/A-Z/;
    my $errorMsg   = "User must be \"SYSTEM\", or $userName must be " .
                     "or a member of $groupName group to run root script";
    trace ("user=$userName");

    # get SYSTEM
    my $systemName = 'SYSTEM';
    if (! is_dev_env ()) {
       $systemName = s_getSystem();
       $systemName =~ tr/a-z/A-Z/;
    }

    if ($userName eq $systemName) {
       trace ("User has $superUser privileges");
       return $superUser;
    }

    # check if local user has privileges
    if (!(LocalGroupIsMember("", $groupName, $userName) ||
          GroupIsMember("", $groupName, $userName))) 
    {
       # local user does not have privileges
       # now check if ORACLE_OWNER has privileges
       
          $userName = $CFG->params('ORACLE_OWNER');
          trace ("verifying Admin privilege for user=$userName");

          if (!(LocalGroupIsMember("", $groupName, $userName) ||
             GroupIsMember("", $groupName, $userName))) 
          { 
             error ("$errorMsg");
             return "";
          }
    }

    trace ("User has $superUser privileges");

    return $superUser;
}

# This function gets only "Administrators" (non-qualified) group name.
sub s_getGroupName
{
   # get group name for Administrators
   my $groupName;

   if (! GetAliasFromRID("", DOMAIN_ALIAS_RID_ADMINS, $groupName)) {
      error ("GetAliasFromRID failed");
   }

   return $groupName;
}

####---------------------------------------------------------
#### Function for setting user and group on a specified path
# ARGS : 3
# ARG1 : Oracle owner
# ARG2 : Oracle group 
# ARG3 : file
sub s_set_ownergroup
{
    # Note: this function is a no-op on NT

    return SUCCESS;
}

# OBSOLETED - do not use
sub s_set_ownergroup_win
#----------------------------------------------------------------
# Function: Use cacls to change file permissions
# Args    : owner, group, file
#----------------------------------------------------------------
{
   my ($owner, $group, $file) = @_;

   if (! $owner) {
      error ("Null value passed for Oracle owner");
      return FAILED;
   }

   if (! $group) {
      error ("Null value passed for group name");
      return FAILED;
   }

   if (! $file) {
      error ("Null value passed for file or directory path");
      return FAILED;
   }

   if (! (-e $file)) {
      error ("The path \"" . $file . "\" does not exist");
      return FAILED;
   }

   # set permission
   my $cmd = "cmd /c cacls $file /E /G \"$group\":F \"$owner\":F > NUL";
   if ($CFG->DEBUG) { trace ("Invoking: $cmd"); }
   system ("$cmd");
   
   return SUCCESS;
}

#----------------------------------------------------------------
# Use ACLs to change permissions for directory lists - used for gpnp wallets.
#
# It uses icacls utility to traverse directories and set necessary ACLs.
# icacls suported from W2K8R2 and W7Enterprise.
# Older utilities cacls and xcacls have various issues and are obsoleted.
#        
# ARGS: 3
# ARG1 - orauser name (on windows - DOMAIN\USER)
# ARG2 - ref to list of private directories
# ARG3 - ref to list of directories with added read access
#----------------------------------------------------------------
sub s_gpnp_wallets_set_access_win
{
   my $orauser      = $_[0];
   my $pvt_dirs_ref = $_[1];
   my $pub_dirs_ref = $_[2];
   my @pvt_dirs = @{$pvt_dirs_ref};
   my @pub_dirs = @{$pub_dirs_ref};


#   # Skip ACL set in dev env.
#   if (is_dev_env ()) {
#      return SUCCESS;
#   }

   # Check if ORACLE_OWNER returned by OUI has a domain qulifier.
   # Since 12, OUI will return non-qualified-name for local users, and
   # domain-qualified-name for domain users.
   # icacls behavior for non-qualified names is following: 
   # if node-local-user exists, it is used, else if domain-user exist, it is
   # used instead. To avoid ambiguity, we will prepend node-local-users
   # passed without qualifier in script on every node.

   # see if owner is node-local (does not have a domain qualifier)
   # if so, prepend with local host name
   if (0 > index($orauser,'\\')) {
      my $host = tolower_host();
      $orauser = "$host\\$orauser";  
   }

   my $ispubdirs = FALSE;
   my $resstatus = SUCCESS;
   foreach my $dirs_ref (\@pvt_dirs, \@pub_dirs) { 
      my @dirs = @{$dirs_ref};
      foreach (@dirs) {
         # For every given dir and its content, set ACLs:
         # icacls preferred for Windows > W2K3 R2
         my @cmd = ('icacls.exe', $_,       # for given dir
                       '/inheritance:r',    # remove inherited rights
                       '/grant:r',          # grant full access
                       '*S-1-5-18:F',       # NT AUTHORITY\SYSTEM
                       '/grant:r',          # grant full access, obj/cont inh
                       '*S-1-5-18:(OI)(CI)(F)',     # NT AUTHORITY\SYSTEM
                       '/grant:r',          # grant full access
                       '*S-1-5-32-544:F',   # BUILTIN\Administrators
                       '/grant:r',          # grant full access
                       '*S-1-5-32-544:(OI)(CI)(F)', # BUILTIN\Administrators
                       '/grant:r',          # grant full access
                       "\"$orauser\":F",    # crsuser
                       '/grant:r',          # grant full access
                       "\"$orauser\":(OI)(CI)(F)",  # crsuser
                       '/remove:g',         # revoke granted
                       '*S-1-5-11',         # NTA\Authenticated Users
                       '/t',                # traverse dir content
                       '/c'                 # continue on error
                    );
         # for public directories, add read grant for authorized users
         if ($ispubdirs) {
           push( @cmd, '/grant:r',          # grant read/exec access
                       '*S-1-5-11:RX',      # NTA\Authenticated Users
                       '/grant:r',
                       '*S-1-5-11:(OI)(CI)(RX)', # NTA\Authenticated Users
                    );
         }
         # execute cmd
         my @out = system_cmd_capture(@cmd);
         my $rc  = shift @out;
         trace("out=@out");

	 if ($rc == 0) {
	    trace ("@cmd ... success");
	 }
	 else {
	    trace ("@cmd ... failed with rc=", $rc);
            $resstatus = FAILED;
         }
      }
      $ispubdirs = TRUE;  # next process public dirs;
   }
   if (! $resstatus) {
      trace("#################################################"); 
      trace("Setting restricred access to gpnp wallets encountered problems."); 
      trace("Affected directories: ".join(' ',@pvt_dirs)."  "
                                    .join(' ',@pub_dirs));
      trace("Make sure filesystem supports icacls and ACLs.");
      trace("#################################################"); 

      error("There was a problem setting restricted access to GPnP wallets. "
           ."System security compromised.");
   }
   return $resstatus;
}

####---------------------------------------------------------
#### Function for setting permissions on a specified path
# ARGS : 2
# ARG1 : permissions
# ARG3 : file/dir
sub s_set_perms
{
    # Note: this function is a no-op on NT

    return SUCCESS;
}

####-----------------------------------------------------------------------
#### Function for installing the Oracle ARP driver
# ARGS: 0
sub s_create_arp_service
{
  if (is_dev_env ())
  {
    return SUCCESS;
  }

  my ($string, $major, $minor, $build, $id, $spmajor, $spminor, $suitemask, $producttype) =
    Win32::GetOSVersion();

  # OS                      ID    MAJOR   MINOR
  # Windows Vista            2      6       0
  # Windows Server 2008      2      6       0
  # Windows 7                2      6       1
  # Windows Server 2008 R2   2      6       1

  # PRODUCTTYPE
  # 1 - Workstation (NT 4, 2000 Pro, XP Home, XP Pro, Vista)
  # 2 - Domaincontroller
  # 3 - Server (2000 Server, Server 2003, Server 2008)

  if ((6 == $major) && (0 == $minor || 1 == $minor) && (2 == $id)
       && (3 == $producttype))
  {
    trace("Installing the Oracle ARP driver");

    my $netcfg = catfile($ENV{SYSTEMROOT}, 'system32', 'netcfg.exe');
    if (! -e $netcfg)
    {
      trace("Invalid executable '$netcfg'");
      return FAILED;
    }

    my $infPath = catfile($CFG->ORA_CRS_HOME, 'bin', 'oraarpdrv.inf');
    $infPath =~ s{\\}{\\\\\\\\}g;
    trace("The location of INF is '$infPath'");

    my @cmd = ($netcfg, "-l", $infPath, "-c", "p", "-i", "orcl_ndisprot");
    my @out = system_cmd_capture(@cmd);
    my $rc = shift @out;
    if (0 == $rc)
    {
      trace("@cmd ... succeeded");
    }
    else
    {
      trace("@cmd ... failed with rc=", $rc >> 8);
      return FAILED;
    }

    # Start the service oraarpdrv
    if (SUCCESS != s_startService("oraarpdrv"))
    {
      trace("Failed to start the Oracle ARP service");
      return FAILED;
    }
  }

  trace("Succeeded in installing the Oracle ARP driver");
  return SUCCESS;
}

####-----------------------------------------------------------------------
#### Function for creating and starting Oracle Fence Service
# ARGS: 0
sub s_create_start_FenceServ
{
   my $ORACLE_HOME = $CFG->params('ORACLE_HOME');
   my ($cmd, $status);

   # Create Oracle Fence Service
   trace ("Creating Oracle Fence Service...");
   my $crssetup = catfile ($ORACLE_HOME,
                              "bin", "crssetup.exe");
   $cmd = "$crssetup installFence";
   if ($CFG->DEBUG) { trace ("Invoking: $cmd"); }

   $status = system ("$cmd");
   if ($status == 0) {
     trace ("Create Oracle Fence Service successfully");
   }
   else {
     error ("Create Oracle Fence Service failed");
     return FAILED;
   }

   # Start "OraFenceService"
   trace("Starting Oracle Fence Service ...");
   if (SUCCESS != s_startService ("OraFenceService")) {
      return FAILED;
   } 
       
   return SUCCESS;
}

####-----------------------------------------------------------------------
#### Function for performing NT-specific setup
# ARGS: 0
sub s_osd_setup
{
   # if in ADE env, skip these steps and return success
   if (is_dev_env ()) {
      return SUCCESS;
   }

   #set perms on  ASM Links if any.
   s_setPermsASMDisks();    

   if (! $CFG->UPGRADE)
   {
     # Create & start Oracle Fence Service
     trace("Create and start Oracle Fence Service");
     s_create_start_FenceServ();
   }

   # Install Oracle ARP Driver
   #if (! s_create_arp_service())
   #{
   #  return FAILED;
   #}
   return SUCCESS;
}

####-----------------------------------------------------------------------
#### Function for removing the Oracle ARP driver
# ARGS: 0
sub s_remove_arp_service
{
  if (is_dev_env ())
  {
    return SUCCESS;
  }

  my ($string, $major, $minor, $build, $id, $spmajor, $spminor, $suitemask, $producttype) =
    Win32::GetOSVersion();

  # OS                      ID    MAJOR   MINOR
  # Windows Vista            2      6       0
  # Windows Server 2008      2      6       0
  # Windows 7                2      6       1
  # Windows Server 2008 R2   2      6       1

  # PRODUCTTYPE
  # 1 - Workstation (NT 4, 2000 Pro, XP Home, XP Pro, Vista)
  # 2 - Domaincontroller
  # 3 - Server (2000 Server, Server 2003, Server 2008)

  if ((6 == $major) && (0 == $minor || 1 == $minor) && (2 == $id)
       && (3 == $producttype))
  {
    trace("Removing the Oracle ARP driver");

    if (!s_stopService("oraarpdrv"))
    {
      trace("Unable to stop the Oracle ARP service");
      return FAILED;
    }

    my $netcfg = catfile($ENV{SYSTEMROOT}, 'system32', 'netcfg.exe');
    if (! -e $netcfg)
    {
      error("Invalid executable '$netcfg'");
      return FAILED;
    }

    my @cmd = ($netcfg, "-u", "orcl_ndisprot");
    my @out = system_cmd_capture(@cmd);
    my $rc = shift @out;
    if (0 == $rc)
    {
      trace("@cmd ... succeeded");
    }
    else
    {
      trace("@cmd ... failed with rc=", $rc >> 8);
      return FAILED;
    }
  }

  trace("Succeeded in removing the Oracle ARP driver");
  return SUCCESS;
}

sub s_isServiceExists
#-------------------------------------------------------------------------------
# Function: Check if Windows service exists
# Args    : service name
#-------------------------------------------------------------------------------
{
   my $svcName = $_[0];
   my %srvc_status;
   
   Win32::Service::GetStatus("", $svcName, \%srvc_status);
  
   if ($srvc_status{"CurrentState"} =~ /[1-7]/) {
      return TRUE;
   }
   else {   
      return FALSE;
   }

}

sub s_isServiceRunning
#-------------------------------------------------------------------------------
# Function: Check Windows service is running or start_pending
# Args    : service name
#-------------------------------------------------------------------------------
{
   my $svcName = $_[0];
   my %srvc_status;
   
   # check if service is running
   Win32::Service::GetStatus("", $svcName, \%srvc_status);
  
   # 4 means service is running
   if ($srvc_status{"CurrentState"} == 4) {
      if ($CFG->DEBUG) { trace ("$svcName is running..."); }
      return TRUE;
   }
   else {   
      return FALSE;
   }
}

sub s_startService
#----------------------------------------------------------------
# Function: Start Windows service
# Args    : service name
#----------------------------------------------------------------
{
   my $svcName = $_[0];
   
   if (s_isServiceRunning ($svcName)) { 
      return SUCCESS;
   }  

   trace ("Starting $svcName...");
   if (! Win32::Service::StartService ("", $svcName)) {
      error ("Start of $svcName failed");
      return FAILED;
   }

   # wait for service to start
   my $retries = 5;
   my $srv_running = FALSE;
   
   while ($retries && (! $srv_running)) {
      if (s_isServiceRunning ($svcName)) { 
         $srv_running = TRUE;
      }
      else {
         trace ("Waiting for $svcName to start");
         sleep (60);
         $retries--;
      }
   }   
       
   if (! $srv_running) {
      error ("Error $svcName failed to start");
      return FAILED;
   }

   return SUCCESS;
}

sub s_deltService
#-------------------------------------------------------------------------------
# Function: Delete Windows service
# Args    : 1 - service name
#-------------------------------------------------------------------------------
{
   my $svcName = $_[0];

   if (s_isServiceExists($svcName)) {
      my $ORACLE_HOME = $CFG->params('ORACLE_HOME');
      if (s_stopService($svcName)) {
         my $oradim = catfile ($ORACLE_HOME, 
                               'bin', 'oradim.exe');
         my @cmd    = ($oradim, "-ex", "s", "delete", "\"$svcName\"");
         my $rc     = system(@cmd);

         if ($rc == 0) {
            trace ("@cmd ... success");
            return SUCCESS;
         }
         else {
            trace ("@cmd ... failed with rc=", $rc >> 8);
            return FAILED;
         }
      }
   }

   return SUCCESS;
}


###------------------------------------------------------------------------
### Fucntion for deleting the ASM services during downgrade
###------------------------------------------------------------------------

sub s_delete_ASM_Services_downgrade
{
   my $ORACLE_HOME = $CFG->params('ORACLE_HOME');  
   my $success = TRUE;
   my $svc_name;
   my %srvc_list;

   Win32::Service::GetServices("",\%srvc_list);

   foreach $svc_name (keys %srvc_list)
   {
     if ($svc_name =~ /^OracleASMService\+ASM/i)
     {
        trace("ASM Service identified is $svc_name");
	my @cmd1 = ("sc","qc",$svc_name);
	my @capout1;
  
        my $rc = s_run_as_user2(\@cmd1,$CFG->params('ORACLE_OWNER'), \@capout1);   

        if ($rc != 0)
        {
           $success = FALSE;
        }

        trace("executed @cmd1 with results... @capout1");

        if ($success)
        {
           foreach(@capout1)
           {
             chomp($_);
                 
             if ($_ =~ /BINARY_PATH_NAME/i)
             {
                my @res = split (/\s+/,trim($_));
                trim($res[2]);

		#Correct the path seprators to one standard before comparing them
                $ORACLE_HOME =~ s/\\\\/\\/g;
                $ORACLE_HOME =~ s/\//\\/g;
                $res[2] =~ s/\\\\/\\/g;
                $res[2] =~ s/\//\\/g;

                trace("binary path of service is $res[2] and oracle home is $ORACLE_HOME");
                #Check if the ASM Service binary is from current grid home location.
                if ($res[2] =~ /\Q$ORACLE_HOME/i) {
                  trace("attempting to delete $svc_name");
                   s_deltService($svc_name);
                }
             }
           }
        }
      }
    }
}
####-----------------------------------------------------------------------
#### Function for checking if CRS is already configured
# ARGS: 2
# ARG1: hostname
# ARG2: crs user
sub s_check_CRSConfig
{

    # ignore all args on NT

    # Check if OCR registry entry exists.
    # XXX: do we need any additional checks?
    my $OCRLOC = $CFG->params('OCRLOC');

    if ($Registry->{"LMachine/$OCRLOC/"}) {
        trace ("HKLM/$OCRLOC/ is already configured\n");
        return TRUE;
    }
    else {
        trace ("HKLM/$OCRLOC/ is NOT configured\n");
        return FALSE;
    }
}

####-----------------------------------------------------------------------
#### Function for validating OLR keys and creating them if they do not exist
# ARGS: 2
# ARG1 : Complete path of OLR location
# ARG2 : CRS Home
sub s_validate_olrconfig
{
    my $olrlocation = $_[0];
    my $crshome     = $_[1];
    my $OLRLOC      = $CFG->params('OLRLOC');

    trace ("Validating OLR registry keys for OLR location " . $olrlocation);

    ## @todo Check existing OLR_LOC. If it exists, then check value of
    #olrconfig_loc property. If it's same as the one passed on the call then go
    #ahead. Else, throw an error msg and quit the installation.
    my $idx = rindex ($OLRLOC, '/') + 1;
    my $parentkey = substr ($OLRLOC, 0, $idx);
    my $key = substr ($OLRLOC, $idx);
    trace ("OLRLOC=$OLRLOC parentkey=$parentkey key=$key");
    $Registry->{"LMachine/$parentkey"} = {
        "$key/" => {
        "/olrconfig_loc" => "$olrlocation",
        "/crs_home" => "$crshome",
        },
    };

    return SUCCESS;
}

####---------------------------------------------------------
#### Function for validating ocr.loc file
# ARGS: 2
# ARG1 : ocrlocations
# ARG2 : isHas
sub s_validate_ocrconfig
{
    trace ("Validating OCR locations in OCR registry key");

    my $ocrlocations = $_[0];
    my $isHas        = $_[1];
    my $OCRLOC       = $CFG->params('OCRLOC');
    my ($ocrlocation,
        $ocrmirrorlocation,
        $ocrlocation3,
        $ocrlocation4,
        $ocrlocation5) = split (/\s*,\s*/, $ocrlocations);

    trace ("Setting ocr location " . $ocrlocation);
    $Registry->{"LMachine/$OCRLOC/"} = {
        "/ocrconfig_loc" => "$ocrlocation",
    };

    if ($ocrmirrorlocation) {
        trace ("Setting ocr mirror location " . $ocrmirrorlocation);
        $Registry->{"LMachine/$OCRLOC/"} = {
            "/ocrmirrorconfig_loc" => "$ocrmirrorlocation",
        };
    }

    if ($ocrlocation3) {
        trace ("Setting ocr location3 " . $ocrlocation3);
        $Registry->{"LMachine/$OCRLOC/"} = {
            "/ocrconfig_loc3" => "$ocrlocation3",
        };
    }

    if ($ocrlocation4) {
        trace ("Setting ocr location4 " . $ocrlocation4);
        $Registry->{"LMachine/$OCRLOC/"} = {
            "/ocrconfig_loc4" => "$ocrlocation4",
        };
    }

    if ($ocrlocation5) {
        trace ("Setting ocr location5 " . $ocrlocation5);
        $Registry->{"LMachine/$OCRLOC/"} = {
            "/ocrconfig_loc5" => "$ocrlocation5",
        };
    }

    if ($isHas) {
        $Registry->{"LMachine/$OCRLOC/"} = {
            "/local_only" => "TRUE",
        };
    } else {
        $Registry->{"LMachine/$OCRLOC/"} = {
            "/local_only" => "FALSE",
        };
    }

    return SUCCESS;
}

####---------------------------------------------------------
#### Validating OCR locations based on existing ocr settings
# ARGS: 3
# ARG1 : Path for Oracle CRS home
# ARG2 : Cluster name
# ARG3 : Comma separated OCR locations
sub s_validateOCR
{
    if ($CFG->addnode) {
       return SUCCESS;
    }

    my $crshome = $_[0];
    my $clustername = $_[1];
    my $ocrlocations = $_[2];
    my $status = SUCCESS;

    trace ("Setting OCR locations in registry");
    s_validate_ocrconfig ($ocrlocations, 0) or {$status = FAILED};

    # XXX: do we need to do anything more here?
    return $status;
}

####---------------------------------------------------------
#### Function for invalidating srvconfig_loc in srvconfig.loc file
sub s_reset_srvconfig
{

    # XXX: currently a no-op on NT; do we need to do anything here?  Like, say,
    # remove some registry (sub)keys??
    return SUCCESS;
}

####---------------------------------------------------------
#### Function for resetting crshome user and permissions
sub s_reset_crshome
{

    # currently a no-op on NT;
    return SUCCESS;
}

####---------------------------------------------------------
#### Function for registering daemon/service with init
# ARGS: 1
# ARG1: daemon to be registered
sub s_register_service
{

    my $srv = $_[0];
    my $ORACLE_HOME = $CFG->params('ORACLE_HOME');
    if ( $CFG->DOWNGRADE ) {
       $ORACLE_HOME = $CFG->OLD_CRS_HOME;
    }
    trace ("srv=$srv ho=$ORACLE_HOME");
    my $SRVBIN = catfile ($ORACLE_HOME, "bin", $srv);
    my $status = system ("$SRVBIN install");

    # XXX: ignore error in Windows dev env as mktwork would have already
    # registered this service
    if ((0 != $status) && !is_dev_env ()) {
        return FAILED;
    }

    return SUCCESS;

}

####---------------------------------------------------------
#### Function for unregistering daemon/service
# ARGS: 1
# ARG1: daemon to be unregistered
sub s_unregister_service
{
    my $srv = $_[0];
    my $ORACLE_HOME = $CFG->params('ORACLE_HOME');
    my $SRVBIN = catfile ($ORACLE_HOME, "bin", $srv);
    my $status = system ("$SRVBIN remove");
    if (0 != $status) {
        return FAILED;
    }

    return SUCCESS;
}

####---------------------------------------------------------
#### Function for starting daemon/service
# ARGS: 3
# ARG1: daemon to be started
# ARG2: user under whom daemon/service needs to be started
sub s_start_service
{
    my $srv  = $_[0];
    my $user = $_[1]; # this arg is ignored on NT

    my $SRVBIN = catfile ($CFG->params('ORACLE_HOME'), "bin", $srv);
    my $status = system ("$SRVBIN start");

    if (0 == $status) {
        trace ("$srv is starting");
        print  "$srv is starting\n";
    } else {
        trace("failed path = $SRVBIN");
        error ("$srv failed to start");
        return FAILED;
    }

    return SUCCESS;
}

####---------------------------------------------------------
#### Function for stopping daemon/service
# ARGS: 2
# ARG1: daemon to be stopped
# ARG2: user under whom daemon/service needs to be stopped
sub s_stop_service
{
    my $srv = $_[0];
    my $user = $_[1]; # this arg is ignored on NT
    my $ORACLE_HOME = $CFG->params('ORACLE_HOME');
    my $SRVBIN = catfile ($ORACLE_HOME, "bin", $srv);
    my $status = system ("$SRVBIN stop");
    if (0 != $status) {
        return FAILED;
    }

    return SUCCESS;
}

####---------------------------------------------------------
#### Function for checking daemon (OSD actions)
# ARGS: 2
# ARG1: daemon to be checked
# ARG2: is daemon running?
sub s_check_service
{
   # no-op on NT; generic actions in check_service() are sufficient
   return SUCCESS;
}

####---------------------------------------------------------
#### Function for initializing SCR settings
# ARGS: 0
sub s_init_scr
{
   # this function is a no-op on NT
   return SUCCESS;
}

####---------------------------------------------------------
#### Function for running a command as given user
# ARGS: 2
# ARG1: cmd to be executed
# ARG2: user name
sub s_run_as_user
{
    my $cmd  = $_[0];
    # ARG2 is ignored on NT; command is always run as current user

    trace ("s_run_as_user: Running $cmd");
    return system ($cmd);
}

####---------------------------------------------------------
#### Function for running a command as given user, returning back 
#### stdout/stderra output
# ARGS: 3
# ARG1: ref to cmdlist argv list to be executed
# ARG2: user name, can be undef
# ARG3: ref to resulting array of stderr/out, can be undef
sub s_run_as_user2
{
    my $cmdlistref = $_[0];
    my @cmdlist = @{$cmdlistref};
    my $usr = $_[1]; # ARG2 is ignored on NT; 
                     # command is always run as current user
    my $capoutref = $_[2];
    my $rc = -1;
    my $cmd = join( ' ', @cmdlist );

    # capture stdout/stderr, if requested
    if (defined($capoutref))
    {
      @{$capoutref} = ();

      trace ("s_run_as_user2: Running $cmd");

      # system() with stdout/stderr capture. 
      # Note that this is a portable notation in perl
      # see http://perldoc.perl.org/perlfaq8.html
      open (CMDOUT, "$cmd 2>&1 |" );
      @{$capoutref} = <CMDOUT>;
      close (CMDOUT); # to get $?
      $rc = $?;
    }
    else  # regular system() call
    {
      $rc = s_run_as_user( $cmd, $usr );
    }
    if ($rc == 0) {
        trace ("$cmdlist[0] successfully executed\n");
    }
    elsif ($rc == -1) {
        trace ("$cmdlist[0] failed to execute\n");
    }
    elsif ($rc & 127) {
        trace ("$cmdlist[0]  died with signal %d, %s coredump\n",
            ($rc & 127),  ($rc & 128) ? 'with' : 'without');
    }
    else {
        trace ("$cmdlist[0] exited with rc=%d\n", $rc >> 8);
    }
    return $rc;
}

####---------------------------------------------------------
#### Function for running a command as given user and inject
#### one value into stdin
# ARGS: 3
# ARG1: user name
# ARG2: cmd to be executed
# ARG3: value to be injected into stdin
sub s_run_as_user3
{
   # read parameters
   # ARG1 is ignored on NT; command is always run as current user
   my $user  = $_[0];
   my $cmd   = $_[1];
   my $param = $_[2];
   
   # create the temporary file and leave it closed.
   my (undef, $cmdout) = tempfile("oracleXXXXX", OPEN => 0, TMPDIR => 1);
   
   # create final command
   my $cmd2 = join (" ", "|", @{$cmd}, ">>$cmdout");
   trace ("s_run_as_user3  Invoking \"$cmd2\"");
   
   # execute the command
   open(COMMAND, $cmd2);
   
   # inject the parameter
   print COMMAND $param;
   
   # close the stream
   close COMMAND;
   my $rc = $?;
   
   # read output and delete temp file
   open(COMMAND_OUT, "$cmdout");
   my @out = (<COMMAND_OUT>);
   close COMMAND_OUT;
   
   # remove the file when $cmdout is out of scope
   s_remove_file("$cmdout");
   
   # join result of command and output
   return ($rc,@out);
}

####---------------------------------------------------------
#### Function for getting value corresponding to a key in ocr.loc or olr.loc
# ARGS: 2
# ARG1: ocr/olr
# ARG2: key
sub s_get_config_key
{
   my $src   = $_[0];
   my $key   = $_[1];
   my $OCRLOC = $CFG->params('OCRLOC');
   my $OLRLOC = $CFG->params('OLRLOC');
   my $SRVLOC = $CFG->params('SRVLOC');
   my $value = "";
   $src      =~ tr/a-z/A-Z/;
   my $reg;

   if ($src eq 'OCR') {
      $reg = $OCRLOC;
   }
   elsif ($src eq 'OLR') {
      $reg = $OLRLOC;
   }
   elsif ($src eq 'SRV') {
      $reg = $SRVLOC;
   }

   my $regkey = $Registry->{"LMachine/$reg/"};
   if (keys (%{$regkey})) {
      $value = $regkey->{"/$key"};
   }
   else {
      error ($regkey . " registry key does not exist");
   }

   return $value;
}

####---------------------------------------------------------
#### Function for getting platform family
# ARGS: 0
sub s_get_platform_family
{
    return "windows";
}

####---------------------------------------------------------
#### Function for checking if a path is a link, and if so, return the target
#### path
#### Note: this function is applicable only to Oracle dev env, where a symlink
#### driver is used.  This will not be applicable in production env, and
#### s_isLink() will always return "" (FALSE)
# ARGS: 1
# ARG1: file/dir path
sub s_isLink
{
    my $path = $_[0];
    my $target = "";

    if (!is_dev_env ()) {
        return $target;
    }

    # run qln and get its output into a string
    open (LNKDRV, "qln $path |") or return "";
    my $op = join ("", <LNKDRV>);
    close (LNKDRV);

    # if qln returns a target path for $path, populate $target
    if ($op && ($op =~ m/->/)) {
        my $key;
        my $ptr;
        ($key, $ptr, $target,) = split (/ /, $op);
    }

    return $target;
}

####--------------------------------
#### Function for redirecting output
# ARGS: 1
# ARG1: file to redirect to
sub s_redirect_souterr
{
    # redirect STDOUT/STDERR to a file
    open(SAVEOUT, ">&STDOUT");
    open(SAVEERR, ">&STDERR");

    open(STDOUT, ">$_[0]") or die "Can't redirect stdout";
    open(STDERR, ">&STDOUT") or die "Can't dup stdout";

    select(STDOUT); $| = 1;  # unbuffer
    select(STDERR); $| = 1;  # unbuffer
}


####---------------------------------------------------------
#### Function for restoring output
# ARGS: 0
sub s_restore_souterr
{
    # restore STDOUT/STDERR
    close(STDOUT);
    close(STDERR);

    open(STDOUT, ">&SAVEOUT");
    open(STDERR, ">&SAVEERR");
}

sub s_getOldCrsHome
#-------------------------------------------------------------------------------
# Function: Get old crshome 
# Args    : none
# Return  : old crshome
#-------------------------------------------------------------------------------
{
   my $oldCRSHome = $CFG->params('OLD_CRS_HOME');

   return $oldCRSHome;
}

####---------------------------------------------------------
#### Function for stopping the services from OldCrsHome
# ARGS:  1

sub s_stop_OldCrsStack
{
  my $oldCrsHome = $_[0];
  my $crsctl     = catfile($oldCrsHome, "bin", "crsctl");
  my @cmd	 = ($crsctl, 'stop', 'crs');
  
  my $rc = system(@cmd);

  return $rc;
}

sub s_checkOracleCM 
#----------------------------------------------------------------
# Function: Check for OracleCMService 
# Args    : none
# Return  : TRUE - if found
#----------------------------------------------------------------
{
   my $svcName = "OracleCMService";
   my %status;

   Win32::Service::GetStatus("",$svcName,\%status);
   my $service_status = $status{CurrentState};

   if ($service_status == 4) {
      # 4 means service is running
      return TRUE;
   }
   else {
      return FALSE;
   }
}

sub s_configureAllRemoteNodes
#---------------------------------------------------------------------
# Function: Automatically execute rootcrs.pl on all remote nodes
#           by calling 'crssetup install'
# Args    : 0 
#---------------------------------------------------------------------
{
   trace ("call 'crssetup install' to configure all remote nodes");
   my $ORACLE_HOME = $CFG->params('ORACLE_HOME');
   my $crssetup = catfile ($ORACLE_HOME, "bin", "crssetup.exe");
   my @remote_param = ('-remotenode'); 
   if ($CFG->UPGRADE) {
      push @remote_param, '-upgrade';
   }

   my @cmd = ("$crssetup", "install", "-remoteParams", "\"@remote_param\""); 
   trace ("cmd=@cmd");
   system(@cmd);
   my $rc = $? >> 8;
   trace("rc from crssetup=$rc ");

   if ($rc == 0) {
      return SUCCESS;
   } 
   else {
      return FAILED;
   }
}

sub s_createLocalOnlyOCR
#---------------------------------------------------------------------
# Function: Create local_only OCR
# Args    : none 
#---------------------------------------------------------------------
{
   trace ("Create local_only OCR on Windows...");
   my $OCRLOC = $CFG->params('OCRLOC');
     
   # create local_only OCR
   $Registry->{"LMachine/$OCRLOC/"} = {
               "/local_only" => "TRUE",
       	       };

}

sub s_ResetOLR
#---------------------------------------------------------------------
# Function: Reset OLR
# Args    : 0
#--------------------------------------------------------------------
{
   my $olrdisk = s_get_config_key("olr", "olrconfig_loc");
   trace("Removing OLR disk: $olrdisk");

   s_remove_file($olrdisk);

   # remove olr registry key
   my $OLRLOC = $CFG->params('OLRLOC');
   my $value  = delete $Registry->{"LMachine/$OLRLOC/"};
}

sub s_ResetOCR
{
   trace ("Reset OCR");
   my $OCRLOC    = $CFG->params('OCRLOC');
   my $SRVCONFIG;
   if ($CFG->defined_param('SRVCONFIG')) {
       $SRVCONFIG = $CFG->params('SRVCONFIG');
   }

   my ($ocr_loc, $ocr_mirror_loc, $ocr_loc3, $ocr_loc4, $ocr_loc5);

   if ($CFG->DOWNGRADE) {
      if ($CFG->oldcrsver eq "9.2") {
	 DowngradeTo9i ();
      } 
      else {
	 DowngradeTo10or11i ();
      }

      return SUCCESS;
   }

   s_ResetOLR();

   if (! $CFG->LASTNODE) {
      # remove ocr registry key
      my $value = delete $Registry->{"LMachine/$OCRLOC/"};
      return SUCCESS;
   }

   $ocr_loc = get_ocrdisk();
   $ocr_mirror_loc = get_ocrmirrordisk();
   $ocr_loc3 = get_ocrloc3disk();
   $ocr_loc4 = get_ocrloc4disk();
   $ocr_loc5 = get_ocrloc5disk();

   if (! -f $ocr_loc) {
      # ocr.loc file does not exist. Take ocr location of srvconfig.loc. 
      if (-f $SRVCONFIG) {
         $ocr_loc = get_srvdisk();
      }
   }

   if ($ocr_mirror_loc) {
      trace("Removing OCR mirror device: $ocr_mirror_loc");
      s_remove_file($ocr_mirror_loc);
   }

   if ($ocr_loc3) {
      trace("Removing OCR mirror device 3: $ocr_loc3");
      s_remove_file($ocr_loc3);
   }

   if ($ocr_loc4) {
      trace("Removing OCR mirror device 4: $ocr_loc4");
      s_remove_file($ocr_loc4);
   }

   if ($ocr_loc5) {
      trace("Removing OCR mirror device 5: $ocr_loc5");
      s_remove_file($ocr_loc5);
   }

   # reset OCR device if it's not on ASM
   if (($CFG->LASTNODE)    &&
       (! $CFG->DOWNGRADE) &&
       (! $CFG->ASM_STORAGE_USED)) 
   {
      trace("Removing OCR device: $ocr_loc");
      s_remove_file($ocr_loc);
   }

   # remove ocr.loc 
   my $value = delete $Registry->{"LMachine/$OCRLOC/"};
}

sub s_remove_file
#-------------------------------------------------------------------------------
# Function: Remove file on Windows
# Args    : File
#-------------------------------------------------------------------------------
{
   my $file = $_[0];
   my $status = SUCCESS;


   if (-e $file) {
      trace("Removing file: $file");
      if (0 == DeleteFile($file)) {
        trace("Failed to remove file: $file");
        $status = FAILED;
      } else {
        trace("Successfully removed file: $file");
      }
   }
   return $status;
}


sub s_ResetVotedisks
#-------------------------------------------------------------------------------
# Function: Reset voting disks
# Args    : [0] list of voting disks
#-------------------------------------------------------------------------------
{
   my @votedisk_list = @_;
   my $vdisk;

   trace ("Reset voting disks:@votedisk_list");
   trace ("CRS_STORAGE_OPTION:" . 
           $CFG->params('CRS_STORAGE_OPTION'));

   if ($CFG->params('CRS_STORAGE_OPTION') != 1) {
      foreach $vdisk (@votedisk_list) {
         trace("Removing voting disk: $vdisk");
         s_remove_file($vdisk);
      }
   }
}

sub s_CleanTempFiles
{
   # this function is a no-op on NT
   return SUCCESS
}

sub s_createConfigEnvFile
{
   # this function is a no-op on NT
   return SUCCESS
}

sub s_isRAC_appropriate
{
   # this function is a no-op on NT
   return SUCCESS
}

sub s_removeCvuRpm
{
   # this function is a no-op on NT
   return SUCCESS
}

sub s_is92ConfigExists
#-------------------------------------------------------------------------------
# Function: Check if Oracle configuration exists in 9.2
# Args    : none
# Returns : TRUE  if     exists
#           FALSE if not exists
#-------------------------------------------------------------------------------
{
   trace ("Checking repository used for 9i installations");

   # Check if osd9i key exists
   if ($Registry->{"LMachine/Software/oracle/osd9i/"}) {
      trace ("Oracle 92 configuration and SKGXN library exists");
      return TRUE;
   }

   trace ("Oracle 92 configuration and SKGXN library not exists");
   return FALSE;
}

sub s_RemoveInitResources
{
   # this function is a no-op on NT
   return SUCCESS
}

sub s_copyOCRLoc
#-------------------------------------------------------------------------------
# Function: copy OCR location from other node
# Args    : 0 
#-------------------------------------------------------------------------------
{
   trace ("Creating ocr key using ocr key from other node");

   my @node_list = getCurrentNodenameList();
   my $OCRLOC    = $CFG->params('OCRLOC');
   my $HOST      = $CFG->HOST;
   my $rc	 = FAILED;
   my $node;

   foreach $node (@node_list) {
      if ($node !~ /\b$HOST\b/i) {
	 trace("get OCR key from node=$node");
	 my $remKey = $Registry->Connect("\\\\$node", 
	   			  	 "LMachine/$OCRLOC");
	 my @vnames = $remKey->ValueNames;
         my ($vname, $vdata);

	 foreach $vname (@vnames) {
            $vdata = $remKey->GetValue("$vname");

            print ("creating $vname=$vdata from node=$node\n");
            trace ("creating $vname=$vdata from node=$node");
            $Registry->{"LMachine/$OCRLOC/"} = {
         	        "$vname" => "$vdata",
            };
         }

	 $rc = SUCCESS;
 	 last;

      }
      else {
         trace("Avoiding self copy of ocr on node: $node");
      }
   }

   return $rc;
}

sub s_houseCleaning
{
   s_remove_arp_service();

   s_clsecho_uninstall();

   s_removeFenceServ();

   s_deltService("Oracle Object Service");

   s_deltService("OracleOHService");

   my $LMachine = "LMachine/System/CurrentControlSet/Services/";
   my ($key, $key_param);

}

sub s_deltRegKey
#-------------------------------------------------------------------------------
# Function: Delete registry key and its subkeys
# Args    : 1 (key) 
#-------------------------------------------------------------------------------
{
   my $key      = $_[0];
   my $checkkey = $Registry->Open("$key") || return SUCCESS;

   my @subkeys = $Registry->{"$key"}->SubKeyNames;
   my ($delt, $subkey);

   # delete all subkeys 
   # Fixme: this function should be recursively delete
   foreach $subkey (@subkeys) {
      print("delete subkey=$subkey\n");
      trace("delete subkey=$subkey");
      $delt = delete $Registry->{"$key/$subkey/"};
   }

   # delete key 
   $delt = delete $Registry->{"$key/"};
}

sub s_getAbsLink
{
   # this function is a no-op on NT
   return SUCCESS;
}

sub s_removeSCR
{
   my $key   = "SCR";
   my $value = delete $Registry->{"LMachine/Software/oracle/$key/"};
}

sub s_clsecho_uninstall
#-------------------------------------------------------------------------------
# Function: Call "clsecho -uninstall"
# Args    : 1 
#-------------------------------------------------------------------------------
{
   if ($CFG->platform_family eq "windows")
   {
     my $ORA_CRS_HOME = $CFG->ORA_CRS_HOME;
     my $clsecho    = catfile ($ORA_CRS_HOME, "bin", "clsecho.exe");

     my @cmd = ($clsecho, '-uninstall');
     system_cmd(@cmd);
   }

    return SUCCESS;
}

sub s_removeFenceServ
#-------------------------------------------------------------------------------
# Function: remove oracle fence service
# Args    : 0 
#-------------------------------------------------------------------------------
{
   trace("Stopping Oracle Fence Service ...");
   s_stopService("OraFenceService");

   my $ORACLE_HOME = $CFG->params('ORACLE_HOME');
   my $crssetup = catfile($ORACLE_HOME, 'bin', 'crssetup.exe');
   my $cmd	= "$crssetup deinstallfence";
   trace ("Remove Oracle Fence Service... $cmd");
   print ("Remove Oracle Fence Service... $cmd\n");

   my $status = system($cmd);
   if ($status == 0) {
      trace ("Remove Oracle Fence Service successfully");
   }
   else {
      error ("Remove Oracle Fence Service failed");
      return FAILED;
   }

   return SUCCESS;
}

sub s_getAuthorizedOwner
#-------------------------------------------------------------------------------
# Function: Get authorized owner ("NT AUTHORITY\SYSTEM")
# Args    : none
#-------------------------------------------------------------------------------
{
   # ohasd.exe should be owned by 'NT AUTHORITY\SYSTEM'
   # therefore get its file permission

   my ($owner, $dummy);

   if (is_dev_env ()) {
      $owner = "NT AUTHORITY\\SYSTEM";
      return $owner;
   }

   my $ORACLE_HOME = $CFG->params('ORACLE_HOME');
   my $ohasd_file = catfile($ORACLE_HOME, 'bin', 'ohasd.exe');

   my @out = system_cmd_capture('cacls.exe', "$ohasd_file");
   my $rc  = shift @out;
   trace ("output from cacls=@out");

   if ($rc == 0) {
      my @grep_out = grep(/\SYSTEM:/, @out); # grep for SYSTEM
      if (scalar(@grep_out) > 0) {
         ($owner, $dummy) = split(/:/, trim($grep_out[0]));
      }
   }

   trace ("owner from cacls=$owner");

   if (! $owner) {
      die ("Unable to get authorized owner");
   }

   return $owner;
}

sub s_copyRegKey
{
   my $from_key = $_[0];
   my $to_key   = $_[1];


   trace ("copy registry key from=$from_key to=$to_key");
   $Registry->{"LMachine/$to_key/"} = $Registry->{"LMachine/$from_key/"};
}

sub s_stopService
#-------------------------------------------------------------------------------
# Function: Stop Windows service
# Args    : 1 - service name
#-------------------------------------------------------------------------------
{
   my $svcName     = $_[0];
   my $svc_stopped = FALSE;
   my $retries     = 5;
   my %svc_status;
   
   if (s_isServiceRunning($svcName)) { 
      # stop service
      Win32::Service::StopService("", $svcName);
 
      while ($retries && (! $svc_stopped)) {
   	 # get service status
         Win32::Service::GetStatus("", $svcName, \%svc_status);
  
         # 1 means service stopped
         if ($svc_status{"CurrentState"} == 1) {
            $svc_stopped = TRUE;
         }
         else {
            trace ("Waiting for $svcName to stop");
            sleep (60);
            $retries--;
         }
      }
   }
   else {
      $svc_stopped = TRUE;
   }
   
   if ($svc_stopped) {
      trace ("stop of $svcName ... success");
   }
   else {
      trace ("stop of $svcName ... failed");
   }

   return $svc_stopped;
}

sub s_setPermsASMDisks 
{
 
   trace("executing asmtool to retrieve ASM links");
   my $asmtool;
   my $oradim;
   my $pathprefix = "\\\\\.\\";
   my $devpath ;

   my $success     = TRUE;

   if(is_dev_env())
   {
      $asmtool = catfile($CFG->ORA_CRS_HOME,"rdbms","bin", "asmtool.exe");
      $oradim  = catfile($CFG->ORA_CRS_HOME,"rdbms","bin", "oradim.exe");
   }
   else
   {
      $asmtool = catfile($CFG->ORA_CRS_HOME,"bin", "asmtool.exe");
      $oradim  = catfile($CFG->ORA_CRS_HOME,"bin", "oradim.exe");
   }

   my @cmd1 = ($asmtool, "-list");
   my @capout1;
   my @cmd2;
   my @capout2;

   my $rc = s_run_as_user2(\@cmd1,$CFG->params('ORACLE_OWNER'), \@capout1);

   if ($rc != 0)
   {
      $success = FALSE;
   }

   trace("executed @cmd1 with results... @capout1");

   if ($success)
   {
      foreach(@capout1)
      {
         chomp($_);
         my @res=split(/\s+/);
         trace("Device name is $res[0]\n");
         if ($res[0] =~ /^ORCL/i)
         {
            $devpath = "$pathprefix" . "$res[0]";
            @cmd2 = ($oradim,"-ACL","-setperm","RawDevice","-USER","raw","-OBJTYPE","RawDevice","-OBJPATH",$devpath,"-RECURSE","false");
            
            $rc = s_run_as_user2(\@cmd2,$CFG->params('ORACLE_OWNER'), \@capout2);

            trace("@cmd2 ... result: @capout2");
         }
      }
   }
}

sub s_stopDeltOldASM
{
   my $asm_service = "OracleASMService\+ASM";
   my $success     = TRUE;

   if (s_isServiceExists($asm_service)) {

      my @cmd = ('net', 'stop', $asm_service, '/Y');
      my @out = system_cmd_capture(@cmd);
      trace("out=@out");
      my $rc  = shift @out;

      if ($rc == 0) {
	 trace ("@cmd ... success");
      }
      else {
        trace ("@cmd ... failed with rc=", $rc >> 8);
        $success = FALSE;
      }

      if ($success) {
	 # delete ASM dependencies
	 @cmd = ('sc', 'config', $asm_service, 'depend=', "\"\"");
         @out = system_cmd_capture(@cmd);
         trace("out=@out");
         $rc  = shift @out;

	 if ($rc == 0) {
	    trace ("@cmd ... success");
	 }
	 else {
	    trace ("@cmd ... failed with rc=", $rc >> 8);
            $success = FALSE;
	 }
      }
   }

   if (! $success) {
      die ("Unable to stop $asm_service and its dependencies");
   }


   return $success;
}

sub s_get_olr_file
{
   my $key = $_[0];

   return s_get_config_key("OLR", $key);
}

sub s_getDomainName
{

   return Win32::DomainName();
}

sub s_deltOldServ
{
   my $rc = s_deltService("OracleCRService");

   if ($rc) {
      s_deltService("OracleCSService");
   }

   if ($rc) {
      s_deltService("OracleEVMService");
   }

   return $rc;
}

sub s_crf_check_bdbloc
{
  my $bdbloc = $_[0];

  trace("IPD/OS BDB location checks: $bdbloc");

  # check for existence first
  if (! -d $bdbloc)
  {
    trace("INFO: BDB path $bdbloc does not exist, creating it...\n");
    mkpath($bdbloc, 0, 0755);
    if (! -d $bdbloc)
    {
      trace("INFO: BDB path $bdbloc could not be created.\n");
      return 15;
    }
  }
  else
  {
    if (! -w $bdbloc)
    {
      trace("INFO: BDB path $bdbloc is not writable, changing ");
      trace("permissions on it...\n");
      chmod oct('0755'),"$bdbloc";
    }
  }

  # check for space now. fsutil reports bytes. Check for 2GB per node.
  my $rqrd;
  my $nodelist = $_[1];

  chomp($nodelist);
  my @hosts = split(/[,]+/, $nodelist);

  $rqrd = (@hosts)*2*1024*1024*1024;
  if (open(DFH, "fsutil volume diskfree $bdbloc"))
  {
    while (<DFH>)
    {
      chomp();
      if (!($_ =~ m/Total # of avail free bytes.*/i))
      {
        my @parts = split(/[ ]+/, $_);
        my $avl = $parts[8];
        if ($avl < $rqrd)
        {
          error("Insufficient free space available in BDB path $bdbloc; run oclumon to change BDB location");
          return 13;
        }
      }
    }
    close DFH;
  }
}

sub s_crf_remove_itab
{
  my $instdir = "C:\\Program Files\\oracrf";
  my $cmd;

  trace("Removing OracelOsToolSysmService on Windows.");
  $cmd = catfile("$instdir", "bin", "osysmond.exe");
  system("$cmd stop");
  system("$cmd remove");
}

sub s_resetParentDirOwner
{
   # this function is a no-op on NT
   return SUCCESS;
}

sub s_is_HAIP_supported
{
  return FALSE;
}

sub s_CheckNetworkConfig
{
   # this function is a no-op on NT
  return;
}

sub s_upgrade_services
{
   s_remove_arp_service();
   s_removeFenceServ();
   s_deltService("Oracle Object Service");

   # Create & start Oracle Fence Service 
   trace("Create and start Oracle Fence Service");
   s_create_start_FenceServ();

   trace("Call the EM script to modify EM services");
   s_callEMScript();
}

sub s_callEMScript
{
  if (is_dev_env())
  {
    return SUCCESS;
  }
  
  trace("Calling EM script modifyEMService.pl ...");
  my $perEx= catfile($CFG->ORA_CRS_HOME, "perl", "bin",
                      "perl.exe");
  my $emScript = catfile($CFG->ORA_CRS_HOME, "bin",
                          "modifyEMService.pl");

  my $cmd = "$perEx $emScript";
  my @output = system_cmd_capture($cmd);  
  my $rc = shift @output;
  if (0 == $rc)
  {
    trace("Successfully modified the EM services");
  }
  else
  {
    die(dieformat(456));
  }

  return SUCCESS;
}

sub s_getSystem
{
   my $param_file = $CFG->paramfile;

   open (PARAM, $param_file);
   my @list = grep(/ORACLE_HOME=/, <PARAM>);
   close (PARAM);

   my ($dummy, $crshome) = split (/=/, $list[0]);
   chomp $crshome;
   my $crssetup = catfile ($crshome, 'bin', 'crssetup.exe');
   my @cmd = ($crssetup, 'getsystem');
   trace("@cmd");
   my @out = system_cmd_capture(@cmd);
   my $rc  = shift @out;

   if (0 != $rc) {
      die ("Get SYSTEM failed with rc=", $rc >> 8);
      return FAILED;
   }

   return $out[0];
}

# HAIP is not used on Windows sto stubbed out
sub s_is_HAIP_NonFatal
{
  return FALSE;
}

sub s_NSCleanUp
{
  return FALSE;
}

sub s_removeGPnPprofile
{
  return;
}

sub s_install_initd
{
    # this function is a no-op on NT
    return SUCCESS;
}

# checks if the OLR backup exists
sub s_checkolrbackup
{
  my $olr_loc_bkp = $CFG->params('OLRLOC') . '.bkp';

  if (!($Registry->{"LMachine/$olr_loc_bkp/"}))
  {
    error("Could not find the OLR backup in the registry");
    return FAILED;
  }

  return SUCCESS;
}

sub s_restoreInitScripts
{
  # This sub is a no-op on Windows
  return SUCCESS;
}

sub s_restoreASMFiles
{
  # This sub is a no-op on Windows
  return SUCCESS;
}

# Restores the OLR from its backup
sub s_restoreolrloc
{
  my $olr_loc     = $CFG->params('OLRLOC');
  my $olr_loc_bkp = $olr_loc . '.bkp';

  trace("Restore OLR");
  s_copyRegKey($olr_loc_bkp, $olr_loc);

  return SUCCESS;
} 

sub s_copy_afdinit_init
{
  #no-op
  return SUCCESS;
}

sub s_rm_afdinit_init
{
  #no-op	
  return SUCCESS;
}	

sub s_rm_afdinit_rclevel
{
  #no-op	
  return SUCCESS;
}

#-------------------------------------------------------------------------------
# Function:  Returns the path to the qosctl script
# Args    :  none
# Returns :  Path to the qosctl script
#-------------------------------------------------------------------------------
sub s_get_qosctl_path {
  my $execPath = catfile( $CFG->ORA_CRS_HOME, 'bin', 'qosctl.bat' );
  if ( !( -x "${execPath}" ) ) {
    trace("The file ${execPath} either does not exist or is not executable");
  }
  return $execPath;
}

1;
