
set echo off;
set feedback 0;
set serveroutput on;
set long 10000;
whenever sqlerror exit failure;

declare                                                         
  username varchar2(128 byte);
  not_as_sys_20001 constant varchar(80 byte) :=
    'You can run this script only if logged in as SYSDBA.';

  failing_feasibility_20013 constant varchar(80 byte) :=
    'CSREPAIR failed. The database character set remains unchanged.';
begin
  
  SELECT user INTO username FROM dual;
  if (username <> 'SYS') then
    raise_application_error(-20001, not_as_sys_20001, true);
  end if;
exception
  when others then
    raise_application_error(-20013, failing_feasibility_20013, true);
end;
/ 

rem ===========================================================================
rem Perform the feasibility tests in restricted mode:
rem - can not prevent changes by DBA.
rem - but at least can prevent interferences from concurrent user sessions.
rem   (then connections from non DBA would fail with ORA-12526, ORA-12527).
rem ===========================================================================
declare                                                         
  sqlstmt  varchar2(400 byte);
  v_count  integer;
  current_dbcs_id   number;
  current_dbcs_name varchar2(28 byte);
  assumed_dbcs_id   number;
  assumed_dbcs_name varchar2(28 byte);
  dbcs_charset_type    integer;
  assumed_charset_type integer;

  get_charset_type_failed_20000 constant varchar(80 byte) :=
    'UTL_I18N.GET_MAX_CHARACTER_SIZE reported an error.';

  bad_dmu_version_20002 constant varchar(160 byte) :=
    'The version of the installed DMU repository is not supported by this version of CSREPAIR.';

  bad_dmu_session_20003 constant varchar(160 byte) :=
    'A DMU session is running or the last DMU session ended abruptly in this database. Make sure the DMU is closed in the standard way. Then, rerun CSREPAIR.';

  bad_dmu_scan_20004 constant varchar(160 byte) :=
    'One or more columns have not been scanned by the DMU. Run a full database scan in the DMU before executing CSREPAIR.';

  bad_assumed_dbcs_20005 constant varchar(160 byte) :=
    'Run a full DMU scan with the Assumed Database Character Set database property set to the desired target character set.';

  same_dbcs_20006 constant varchar(160 byte) :=
    'The target character set provided in the Assumed Database Character Set database property is the same as the current database character set.';

  assumed_dbcs_super_ascii_20007 constant varchar(160 byte) :=
    'The target character set provided in the Assumed Database Character Set database property must be a binary superset of US7ASCII.';

  not_multi_to_single_20008 constant varchar(160 byte) :=
    'Changing the database character set from multi-byte to single-byte is not supported.';

  not_single_to_multi_20009 constant varchar(160 byte) :=
    'Changing the database character set from single-byte to multi-byte is not supported.';

  bad_assumed_column_cs_20010 constant varchar(160 byte) :=
    'The assumed character set of one or more columns does not match the assumed database character set.';

  invalid_data_20011 constant varchar(160 byte) :=
    'The DMU has reported data with invalid binary representation. Correct the data before running CSREPAIR.';

  altered_dmu_20012 constant varchar(160 byte) :=
    'One or more tables have been altered since the last DMU scan. Rerun a full database scan in the DMU before executing CSREPAIR.';

  failing_restricted_mode_20014 constant varchar(160 byte) :=
    'CSREPAIR failed. The database character set remains unchanged. Restart the database to restore system parameters modified by the script.';


  bad_dmu_repository_20016_20023 constant varchar(320 byte) :=
    'The DMU repository does not exist or is corrupted. Install the repository and run a full DMU database scan specifying the desired target character set in the Assumed Database Character Set database property.';

  unsupported_dbcs_20024 constant varchar(160 byte) :=
    'The database character set is not supported by CSREPAIR.';

  unsupported_assumed_cs_20025 constant varchar(160 byte) :=
    'The target character set is not supported by CSREPAIR.';











function csr$get_charset_type(cs_name IN varchar2)
  return integer
is
  cs_id        integer;
  max_cs_width integer;
  no_get_maxsize EXCEPTION;
  PRAGMA EXCEPTION_INIT(no_get_maxsize, -904);
begin
  



  begin
    EXECUTE IMMEDIATE
      'SELECT utl_i18n.get_max_character_size(:1) FROM dual'
    INTO max_cs_width
    USING cs_name;

    if (max_cs_width > 1) then
      max_cs_width := 2;
    end if;

    return max_cs_width;

  exception
    when no_get_maxsize then
    

    



    SELECT nls_charset_id(cs_name) INTO cs_id FROM DUAL;

    case                                                   
      when cs_id = 1002  then max_cs_width := 1; 
      when cs_id = 1863  then max_cs_width := 0; 
      when cs_id = 9996  then max_cs_width := 0; 
      when cs_id = 9999  then max_cs_width := 0; 
    else
      if ((cs_id >= 1) and (cs_id <= 798)) then
        max_cs_width := 1;
      elsif ((cs_id >= 829) and (cs_id <= 7999) or
                       
             (cs_id >= 9000) and (cs_id <= 9999)) then
        max_cs_width := 2;
      else
        max_cs_width := 0;                    
      end if;
    end case;

    return max_cs_width;
  end;

exception
  when others then
    raise_application_error(-20000, get_charset_type_failed_20000, true);

end csr$get_charset_type;                                 



begin
  
  EXECUTE IMMEDIATE 'ALTER SYSTEM ENABLE RESTRICTED SESSION';
  EXECUTE IMMEDIATE 'ALTER SYSTEM SET job_queue_processes=0 SCOPE=MEMORY';
  EXECUTE IMMEDIATE 'ALTER SYSTEM SET aq_tm_processes=0 SCOPE=MEMORY';


  
  
  SELECT count(*) INTO v_count FROM dba_objects
    WHERE object_type='TABLE' AND owner='SYSTEM' AND object_name IN
         ('DUM$DATABASE', 'DUM$TABLES', 'DUM$COLUMNS',
          'DUM$TABLE_PROFILES', 'DUM$COLUMN_PROFILES', 'DUM$SCHEMA_PROFILES',
          'DUM$SESSIONS',
          'DUM$ATTRIBUTES', 'DUM$SEGMENTS', 'DUM$TABLESPACES',
          'DUM$TABLE_CHUNKS', 'DUM$EXCEPTIONS', 'DUM$SQLTEXT',
          'DUM$INDEXES');
  




  if (v_count <> 14) then
    raise_application_error(-20023, bad_dmu_repository_20016_20023, true);
  end if;

  sqlstmt := 'SELECT count(*) FROM system.dum$database';
  EXECUTE IMMEDIATE sqlstmt INTO v_count;
  if (v_count <> 1) then
    raise_application_error(-20016, bad_dmu_repository_20016_20023, true);
  end if;

  
  sqlstmt := 'SELECT version FROM system.dum$database';
  EXECUTE IMMEDIATE sqlstmt INTO v_count;
  if (v_count <> 4) then
    
    raise_application_error(-20002, bad_dmu_version_20002, true);
  end if;

  
  sqlstmt := 'SELECT /*+ first_rows(2) */ count(*) FROM system.dum$tables WHERE rownum < 2';
  EXECUTE IMMEDIATE sqlstmt INTO v_count;
  if (v_count < 1) then
    raise_application_error(-20017, bad_dmu_repository_20016_20023, true);
  end if;

  sqlstmt := 'SELECT /*+ first_rows(2) */ count(*) FROM system.dum$columns WHERE rownum < 2';
  EXECUTE IMMEDIATE sqlstmt INTO v_count;
  if (v_count < 1) then
    raise_application_error(-20018, bad_dmu_repository_20016_20023, true);
  end if;

  sqlstmt := 'SELECT /*+ first_rows(2) */ count(*) FROM system.dum$column_profiles WHERE rownum < 2';
  EXECUTE IMMEDIATE sqlstmt INTO v_count;
  if (v_count < 1) then
    raise_application_error(-20019, bad_dmu_repository_20016_20023, true);
  end if;

  sqlstmt := 'SELECT /*+ first_rows(2) */ count(*) FROM system.dum$table_profiles WHERE rownum < 2';
  EXECUTE IMMEDIATE sqlstmt INTO v_count;
  if (v_count < 1) then
    raise_application_error(-20020, bad_dmu_repository_20016_20023, true);
  end if;

  sqlstmt := 'SELECT /*+ first_rows(2) */ count(*) FROM system.dum$schema_profiles WHERE rownum < 2';
  EXECUTE IMMEDIATE sqlstmt INTO v_count;
  if (v_count < 1) then
    raise_application_error(-20021, bad_dmu_repository_20016_20023, true);
  end if;

  


  sqlstmt := 'SELECT count(*) FROM system.dum$sessions';
  EXECUTE IMMEDIATE sqlstmt INTO v_count;
  if (v_count <> 0) then
    if (v_count = 1) then
      raise_application_error(-20003, bad_dmu_session_20003, true);
    else
      raise_application_error(-20022, bad_dmu_repository_20016_20023, true);
    end if;
  end if;


  
  sqlstmt :=
    'SELECT /*+ first_rows(2) */ count(*) FROM system.dum$columns
       WHERE (scan_status IS NULL OR scan_status <> 1)
       AND scheduled_no_conversion <> 1
       AND rownum < 2';
  EXECUTE IMMEDIATE sqlstmt INTO v_count;
  if (v_count > 0) then
    raise_application_error(-20004, bad_dmu_scan_20004, true);
  end if;


  
  
  sqlstmt := 'SELECT ddbcs_id, nls_charset_name(ddbcs_id) FROM system.dum$database';
  EXECUTE IMMEDIATE sqlstmt INTO assumed_dbcs_id, assumed_dbcs_name;
  if (assumed_dbcs_id < 1) then
    raise_application_error(-20005, bad_assumed_dbcs_20005, true);
  end if;


  
  SELECT nls_charset_id(value), value INTO current_dbcs_id, current_dbcs_name
    FROM nls_database_parameters
    WHERE parameter = 'NLS_CHARACTERSET';

  if (assumed_dbcs_id = current_dbcs_id) then
    raise_application_error(-20006, same_dbcs_20006, true);
  end if;


  
  if (UTL_RAW.CONVERT(HEXTORAW('23405b5c5d5e5f607b7c7d7e'),
                       'AMERICAN_AMERICA.AL32UTF8',
                       'AMERICAN_AMERICA.'|| assumed_dbcs_name)
      <> HEXTORAW('23405b5c5d5e5f607b7c7d7e')) then
    raise_application_error(-20007, assumed_dbcs_super_ascii_20007, true);
  end if;

  















  

  dbcs_charset_type := csr$get_charset_type(current_dbcs_name);
  assumed_charset_type := csr$get_charset_type(assumed_dbcs_name);

  if (dbcs_charset_type = 0) then
    raise_application_error(-20024, unsupported_dbcs_20024, true);
  end if;

  if (assumed_charset_type = 0) then
    raise_application_error(-20025, unsupported_assumed_cs_20025, true);
  end if;

  if (dbcs_charset_type <> assumed_charset_type) then
    if (assumed_charset_type = 1) then
      
      raise_application_error(-20008, not_multi_to_single_20008, true);
    else
      
      raise_application_error(-20009, not_single_to_multi_20009, true);
    end if;
  end if;


  

  sqlstmt :=
    'SELECT /*+ first_rows(2) */ count(*) FROM system.dum$columns
       WHERE dccs_id <> :1
       AND dccs_id != 0
       AND dccs_id IS NOT NULL
       AND rownum < 2';
  EXECUTE IMMEDIATE sqlstmt INTO v_count USING assumed_dbcs_id;
  if (v_count <> 0) then
    raise_application_error(-20010, bad_assumed_column_cs_20010, true);
  end if;


  
  sqlstmt :=
    'SELECT /*+ first_rows(2) */ count(*) FROM system.dum$columns
       WHERE invalid > 0
       AND scheduled_no_conversion = 0
       AND rownum < 2';
  EXECUTE IMMEDIATE sqlstmt INTO v_count;
  if (v_count <> 0) then
    raise_application_error(-20011, invalid_data_20011, true);
  end if;


  
  if (dbcs_charset_type = 1) then
    sqlstmt := 'begin SYS.DBMS_DUMA_SYS.REFRESH_REPOSITORY(:1, NULL, TRUE); end;';
  else
    sqlstmt := 'begin SYS.DBMS_DUMA_SYS.REFRESH_REPOSITORY(:1, NULL, FALSE); end;';
  end if;
  EXECUTE IMMEDIATE sqlstmt USING OUT v_count;
  if (v_count <> 0) then
    raise_application_error(-20012, altered_dmu_20012, true);
  end if;

exception
  when others then
    dbms_output.put_line(sqlstmt);
    raise_application_error(-20014, failing_restricted_mode_20014, true);
end;
/ 

rem ===========================================================================
rem The last thing is the only thing to do: ADBCS.
rem ADBCS purges shared pool and changes NLS execution environment.
rem ===========================================================================
declare
  new_cs_name varchar2(28 byte);

  shutdown_abort_20015 constant varchar(160 byte) :=
    'CSREPAIR failed. Shut down the database and contact Oracle Support.';
begin
  EXECUTE IMMEDIATE
    'SELECT nls_charset_name(ddbcs_id) FROM system.dum$database'
  INTO new_cs_name;

  EXECUTE IMMEDIATE
    'ALTER DATABASE CHARACTER SET internal_use ' || new_cs_name;

  dbms_output.put_line('The database character set has been successfully changed to ' ||
    new_cs_name || '. You must restart the database now.');

exception
  when others then
    raise_application_error(-20015, shutdown_abort_20015, true);
end;
/

/* Explicit exit so that this script can be not embedded in other scripts. */
exit;
