create or replace package utl_dbws as

 -------------------------------------------
 -------------------------------------------
 ---                                     ---
 --- Handling of qualified names (QName) ---
 ---                                     ---
 -------------------------------------------
 -------------------------------------------


 SUBTYPE QNAME      IS VARCHAR2(4096);
 TYPE    QNAME_LIST IS TABLE OF QNAME INDEX BY BINARY_INTEGER;

 -- Construct a qualified name
 --   namespaceURI - Namespace URI for the QName, null if none.
 --      localPart - Local part of the QName
 function to_QName(name_Space VARCHAR2, name VARCHAR2) RETURN QNAME;

 -- Return the namespace URI of a qualified name, null if none.
 function get_namespace_URI(name QNAME) RETURN VARCHAR2;

 -- Return the local part of a qualified name
 function get_local_part(name QNAME) RETURN VARCHAR2;
 
 -- The following is a list of predefined namespaces that may be
 -- used in the name_Space parameter of to_QName
 -- 'NSURI_SCHEMA_XSD','xsd'        - Namespace URI for XML Schema XSD
 -- 'NSURI_SCHEMA_XSI','xsi'        - Namespace URI for XML Schema XSI
 -- 'NSURI_SOAP_ENCODING','soapenc' - Namespace URI for SOAP 1.1 Encoding
 -- 'NSURI_SOAP_ENVELOPE','soapenv' - Namespace URI for SOAP 1.1 Envelope
 -- 'NSURI_SOAP_NEXT_ACTOR'         - Namespace URI for SOAP 1.1 next actor role


 -------------------------------------------
 -------------------------------------------
 ---                                     ---
 --- Service instantiation based on WSDL ---
 ---                                     ---
 -------------------------------------------
 -------------------------------------------

 SUBTYPE SERVICE IS NUMBER;

 -- Create a Service instance.
 --   serviceName - QName for the service
 --   Returns a handle to the Service instance.
 function create_service(service_Name QNAME) RETURN SERVICE;

 -- Create a Service instance.
 --   wsdlDocumentLocation - URL for the WSDL document location for the service
 --   serviceName - QName for the service
 --   Returns a handle to the Service instance.
 function create_service(wsdl_Document_Location URITYPE, service_Name QNAME) RETURN SERVICE;

 -- List the qualified names of all of the ports in a service.
 --    service_Handle - Service instance whose ports are returned
 function get_ports(service_Handle SERVICE) RETURN QNAME_LIST;

 -- List the qualified names of all of the operations on a particular
 --    service port.
 --    service_Handle - Service instance whose operations are returned
 --    port           - Qualified name of a service port. NULL if the
 --                     first port of the service is to be used.
 function get_operations(service_Handle SERVICE, port QNAME) RETURN QNAME_LIST;

 -- Release a particular Service instance. This will implicitly
 -- release all Call instances that have been created for this
 -- service instance.
 --    service_Handle - Service instance that is to be released
 procedure release_service(service_Handle SERVICE);

 -- Release all Service instances.
 procedure release_all_services;


 ---------------------------------------------
 ---------------------------------------------
 ---                                       ---
 --- Call instantiation based on a service ---
 ---     port and an operation name        ---
 ---                                       ---
 ---------------------------------------------
 ---------------------------------------------

 SUBTYPE CALL IS NUMBER;

 TYPE ANYDATA_LIST IS TABLE OF ANYDATA INDEX BY BINARY_INTEGER;

 -- Set the proxy address 
 --   proxy - the http proxy address, e.g., www-proxy.us.acme.com:80 
 procedure set_http_proxy(httpProxy VARCHAR2); 

 -- Set the default http read and connection timeout values
 -- to be used for each new HTTPConnection.
 --   timeout - timeout in milliseconds. A time of 0 means wait indefinitely 
 procedure set_http_default_timeout(timeout NUMBER); 
 
 -- Create a Call instance.
 --   serviceHandle - the service instance that is to be called.
 function create_call(service_Handle SERVICE) RETURN CALL;

 -- Create a Call instance.
 --   serviceHandle - the service instance that is to be called.
 --   portName - qualified name for the port. Use first port if this is NULL.
 --   operationName - qualified name for the operation
 function create_call(service_Handle SERVICE, port_Name QNAME, operation_Name QNAME)
                      RETURN CALL;

 -- Release a particular Call instance.
 --    call_Handle - Call instance that is to be released
 procedure release_call(call_Handle CALL);

 -- Set the value of a particular property on a Call.
 --   callHandle - the instance of the call
 --   endpoint   - the endpoint for the call
 procedure set_target_endpoint_address(call_Handle CALL, endpoint VARCHAR2); 


 -- Manipulation of call properties. The following are supported keys
 -- and default settings for standard Call properties.
 --
 -- Key                 - Explanation of Value, Default value.
 -- 'USERNAME'          - User name for authentication 
 -- 'PASSWORD'          - Password for authentication 
 -- 'ENCODINGSTYLE_URI' - Encoding style specified as a namespace URI.
 --                       The default value is the SOAP 1.1 encoding
 --                       'http://schemas.xmlsoap.org/soap/encoding/'
 -- 'OPERATION_STYLE'   - Standard property for operation style.
 --                       Set to 'rpc' if the operation style is rpc;
 --                       'document' if the operation style is document. 
 -- 'SESSION_MAINTAIN'  - This boolean property is used by a service client to indicate whether or
 --                       not it wants to participate in a session with a service endpoint.
 --                       If this property is set to 'true', the service client indicates that it
 --                       wants the session to be maintained. If set to 'false', the session is
 --                       not maintained. The default value for this property is 'false'. 
 -- 'SOAPACTION_USE'    - This boolean property indicates whether or not SOAPAction
 --                       is to be used. The default value of this property is 'false'.
 -- 'SOAPACTION_URI'    - Indicates the SOAPAction URI if the SOAPACTION_USE property
 --                       is set to 'true'. 

 -- Return the value of a particular property on a Call.
 --   callHandle - the instance of the call
 --   key        - the key for the property
 --   Returns the value of the property or null if not set.
 function  get_property(call_Handle CALL, key VARCHAR2) RETURN VARCHAR2; 

 -- Set the value of a particular property on a Call.
 --   callHandle - the instance of the call
 --   key        - the key for the property
 --   value      - the value for the property
 procedure set_property(call_Handle CALL, key VARCHAR2, value VARCHAR2); 

 -- Clear the value of a particular property on a Call.
 --   callHandle - the instance of the call
 --   key        - the key for the property
 procedure remove_property(call_Handle CALL, key VARCHAR2); 

 -------------------------------------------------------------------
 -------------------------------------------------------------------
 -- The following list describes the supported XML types
 --
 -- XML Type                       SQL Type
 -------------------------------------------------------------------
 --
 -- xsd:string                     VARCHAR2
 -- soapenc:string
 --
 -- xsd:int, xsd:long, xsd:short,  NUMBER (no NULL permitted)
 -- xsd:float, xsd:double,         and related SQL types
 -- xsd:boolean
 -- soapenc:boolean, soapenc:float,
 -- soapenc:double, soapenc:int,
 -- soapenc:short, soapenc:byte,
 --
 -- xsd:integer, xsd:decimal,      NUMBER (NULL permitted)
 -- soapenc:decimal                and related SQL types
 -- 
 -- xsd:QName                      VARCHAR2
 --
 -- xsd:dateTime                   DATE, TIMESTAMP,
 --                                TIMESTAMP WITH TIMEZONE
 --                                TIMESTAMP WITH LOCAL TIMEZONE
 --
 -- xsd:base64Binary,              RAW
 -- xsd:hexBinary,
 -- soapenc:base64
 --
 -------------------------------------------------------------------
 -------------------------------------------------------------------
 
 -- List the XML type that is returned by the given call.
 --    call_Handle - Service instance whose return type is returned.
 function get_return_type(call_Handle CALL) RETURN QNAME;

 -- List the XML type of the input parameters of the given call.
 --    call_Handle - Service instance whose input types are returned.
 function get_in_parameter_types(call_Handle CALL) RETURN QNAME_LIST;

 -- List the XML type of the output parameters of the given call.
 --    call_Handle - Service instance whose output types are returned.
 function get_out_parameter_types(call_Handle CALL) RETURN QNAME_LIST;

 -- Invokes a document style operation using a synchronous 
 -- request-response- interaction mode.
 --   callHandle - the instance of the call
 --   request - the request for this invocation.
 --   Returns the response or null.
 function invoke(call_Handle CALL, request SYS.XMLTYPE) return SYS.XMLTYPE;

 -- Invokes a specific operation using a synchronous request-response
 -- interaction mode.
 --   callHandle - the instance of the call
 --   inputParams - The input parameters for this invocation.
 --   Returns the return value or null.
 function invoke(call_Handle CALL, input_Params ANYDATA_LIST) return ANYDATA;

 -- Obtain the output arguments after a call invocation
 --   callHandle - the instance of the call
 --   Returns the output arguments in order. 
 function get_output_values(call_Handle CALL) return ANYDATA_LIST;

 -- Set the type of a parameter of a Call.
 --   callHandle - the instance of the call
 --   xml_name - the xml name of the parameter type 
 --   q_name - the QNAME for the parameter type 
 --   mode - the ParameterMode mode constant 
 procedure add_parameter(call_Handle CALL, xml_name VARCHAR2, q_name QNAME, p_mode VARCHAR2); 

 -- Set the return type of a Call.
 --   callHandle - the instance of the call
 --   retType - the qname of the return type 
 procedure set_return_type(call_Handle CALL, ret_type QNAME);
 
 -- set webservices client trace log level
 procedure set_logger_level(level  VARCHAR2);

end utl_dbws;
/

